(function (Drupal, drupalSettings, once, $) {
  Drupal.behaviors.socialSummariesSidebar = {
    attach: function (context) {
      once('socialSummariesSidebar', '.social-summaries-sidebar', context).forEach(function () {
        // Add message styling based on content
        const messageContainer = document.getElementById('social-summaries-message');
        if (messageContainer) {
          const messageText = messageContainer.textContent.toLowerCase();
          messageContainer.className = '';

          if (messageText.includes('success') || messageText.includes('accepted') || messageText.includes('generated successfully')) {
            messageContainer.classList.add('success');
          } else if (messageText.includes('error') || messageText.includes('failed')) {
            messageContainer.classList.add('error');
          } else {
            messageContainer.classList.add('info');
          }
        }

        // Add loading states for generation buttons
        $('.social-summaries-sidebar input[type="submit"]', context).each(function () {
          const button = $(this);
          const originalText = button.val();

          // Store original text as data attribute
          button.data('original-text', originalText);

          // Add ARIA attributes for accessibility
          if (!button.attr('aria-label')) {
            button.attr('aria-label', originalText);
          }

          button.on('click', function () {
            // Add loading state
            button.prop('disabled', true);
            button.addClass('social-summaries-loading');
            button.attr('aria-busy', 'true');

            // Add spinner
            const spinner = $('<span class="social-summaries-spinner" aria-hidden="true"></span>');
            button.prepend(spinner);

            // Change button text
            let newText = originalText;
            if (button.val().includes('Generate')) {
              newText = 'Generating...';
            } else if (button.val().includes('Regenerate')) {
              newText = 'Regenerating...';
            } else if (button.val().includes('Save to Node')) {
              newText = 'Saving...';
            } else if (button.val().includes('Load Saved')) {
              newText = 'Loading...';
            } else if (button.val().includes('Cancel')) {
              newText = 'Cancelling...';
            }
            button.val(newText);
            button.attr('aria-label', newText);

            // Show loading message with ARIA live region
            const messageContainer = $('#social-summaries-message');
            if (messageContainer.length) {
              messageContainer.html('<div class="social-summaries-loading-message" role="status" aria-live="polite">Processing your request...</div>');
            }

            // Announce to screen readers
            if (Drupal.announce) {
              Drupal.announce('Processing your request...', 'polite');
            }
          });
        });

        // Function to reset button states
        function resetButtonStates() {
          $('.social-summaries-sidebar input[type="submit"]', context).each(function () {
            const button = $(this);

            // Only reset if the button is currently in loading state
            if (button.hasClass('social-summaries-loading') || button.prop('disabled')) {
              button.prop('disabled', false);
              button.removeClass('social-summaries-loading');
              button.removeAttr('aria-busy');
              button.find('.social-summaries-spinner').remove();

              // Reset button text
              const originalText = button.data('original-text');
              if (originalText) {
                button.val(originalText);
                button.attr('aria-label', originalText);
              }
            }
          });
        }

        // Listen for Drupal AJAX events
        $(document).on('ajaxComplete', function (event, xhr, settings) {
          // Small delay to ensure DOM updates are complete
          setTimeout(resetButtonStates, 100);
        });

        // Also listen for general AJAX completion as fallback
        $(document).ajaxComplete(function (event, xhr, settings) {
          setTimeout(resetButtonStates, 100);
          
          // Trigger constraint updates for all textareas after AJAX completion
          $('.social-summaries-content-textarea').each(function() {
            const textarea = $(this);
            const platform = textarea.data('platform');
            if (platform) {
              // Small delay to ensure content is updated
              setTimeout(function() {
                textarea.trigger('social-summaries-content-updated');
              }, 50);
            }
          });
        });

        // Initialize platform constraints with a slight delay to ensure DOM is ready
        setTimeout(function() {
          initializePlatformConstraints(context);
        }, 100);

        // Auto-resize textareas
        $('.social-summaries-content-textarea', context).each(function () {
          const textarea = $(this);
          textarea.on('input', function () {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
          });
        });
      });
    }
  };

  /**
   * Initialize platform constraints for all content textareas.
   */
  function initializePlatformConstraints(context) {
    // Get constraint data from drupalSettings
    const constraints = drupalSettings.socialSummaries ? drupalSettings.socialSummaries.constraints : {};
    
    $('.social-summaries-content-textarea', context).each(function () {
      const textarea = $(this);
      const platform = textarea.data('platform');
      const constraintData = constraints[platform];
      
      if (constraintData) {
        setupConstraintMonitoring(textarea, platform, constraintData);
      }
    });
  }

  /**
   * Setup constraint monitoring for a specific textarea.
   */
  function setupConstraintMonitoring(textarea, platform, constraintData) {
    // Try multiple selectors to find the container
    let container = textarea.closest('.social-summaries-tab-' + platform);
    if (container.length === 0) {
      container = textarea.closest('details');
    }
    if (container.length === 0) {
      container = textarea.closest('.social-summaries-sidebar');
    }
    
    let constraintsContainer = container.find('.social-summaries-constraints');
    if (constraintsContainer.length === 0) {
      // Try to find by ID
      constraintsContainer = $('#social-summaries-constraints-' + platform);
    }
    
    if (constraintsContainer.length === 0) {
      return;
    }

    // Create constraint display
    const constraintDisplay = $('<div class="social-summaries-constraint-display"></div>');
    constraintsContainer.append(constraintDisplay);
    
    // Show the constraints container since it now has content
    constraintsContainer.addClass('has-content').show();

    function updateConstraints() {
      const text = textarea.val();
      const evaluation = evaluateConstraints(platform, text, constraintData);
      // Update display
      updateConstraintDisplay(constraintDisplay, evaluation, constraintData);
    }

    // Update on input and custom events
    textarea.on('input', updateConstraints);
    textarea.on('social-summaries-content-updated', updateConstraints);
    // Initial update
    updateConstraints();
  }

  /**
   * Evaluate text against platform constraints.
   */
  function evaluateConstraints(platform, text, constraintData) {
    const trimmedText = text.trim();
    let count = 0;
    const warnings = [];
    let status = 'good';

    // Count based on constraint type
    switch (constraintData.type) {
      case 'character':
        count = trimmedText.length;

        break;

      case 'word':
        count = trimmedText.split(/\s+/).filter(word => word.length > 0).length;

        break;

      case 'sentence':
        count = trimmedText.split(/[.!?]+/).filter(sentence => sentence.trim().length > 0).length;

        break;
    }

    // Check constraints
    // No hard-limit errors. Only recommended messaging is shown below.

    // No minimum hard-limit warnings.

    // Check recommended ranges
    if (constraintData.recommended_min && count < constraintData.recommended_min) {
      warnings.push(`Below the recommended minimum of ${constraintData.recommended_min} ${constraintData.type}s`);
      if (status === 'good') {
        status = 'warning';
      }
    }
    if (constraintData.recommended_max && count > constraintData.recommended_max) {
      warnings.push(`Exceeds the recommended maximum of ${constraintData.recommended_max} ${constraintData.type}s`);
      if (status === 'good') {
        status = 'warning';
      }
    }

    return {
      count: count,
      type: constraintData.type,
      warnings: warnings,
      status: status,
      constraintData: constraintData
    };
  }

  /**
   * Update the constraint display element.
   */
  function updateConstraintDisplay(display, evaluation, constraintData) {
    const { count, type, warnings, status } = evaluation;

    // Clear previous content
    display.empty();

    // Create counter
    const counter = $('<div class="social-summaries-counter social-summaries-counter--' + status + '"></div>');
    counter.text(count + ' ' + type + (count !== 1 ? 's' : ''));

    // Add recommended range info
    if (constraintData.recommended_min && constraintData.recommended_max) {
      const range = $('<div class="social-summaries-range">Recommended: ' + constraintData.recommended_min + '-' + constraintData.recommended_max + ' ' + type + 's</div>');
      counter.append(range);
    }

    display.append(counter);

    // Add warnings if any
    if (warnings.length > 0) {
      const warningContainer = $('<div class="social-summaries-warnings"></div>');
      warnings.forEach(warning => {
        const warningElement = $('<div class="social-summaries-warning social-summaries-warning--' + status + '">' + warning + '</div>');
        warningContainer.append(warningElement);
      });
      display.append(warningContainer);
    }
    // Update ARIA attributes
    let ariaLabel = count + ' ' + type + (count !== 1 ? 's' : '');
    if (warnings.length > 0) {
      ariaLabel += '. ' + warnings.join('. ');
    }
    display.attr('aria-label', ariaLabel);
    display.attr('role', 'status');
    display.attr('aria-live', 'polite');
  }

  // Fallback initialization when document is ready
  Drupal.attachBehaviors(document);

})(Drupal, drupalSettings, once, jQuery);
