<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\social_summaries\Generator\GenerationManager;
use Drupal\social_summaries\Plugin\SocialSummariesProvider\SocialSummariesProviderManager;
use Drupal\social_summaries\Service\RateLimiter;

/**
 * Generate controller (JSON) - mirrors sidebar functionality.
 */
class GenerateController extends ControllerBase {

  /**
   * The generation manager service.
   *
   * @var \Drupal\social_summaries\Generator\GenerationManager
   */
  protected GenerationManager $manager;

  /**
   * The provider plugin manager.
   *
   * @var \Drupal\social_summaries\Plugin\SocialSummariesProvider\SocialSummariesProviderManager
   */
  protected SocialSummariesProviderManager $providerManager;

  /**
   * The rate limiter service.
   *
   * @var \Drupal\social_summaries\Service\RateLimiter
   */
  protected RateLimiter $rateLimiter;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = new static();
    $instance->manager = $container->get('social_summaries.manager');
    $instance->providerManager = $container->get('plugin.manager.social_summaries_provider');
    $instance->rateLimiter = $container->get('social_summaries.rate_limiter');
    $instance->logger = $container->get('logger.factory')->get('social_summaries');
    return $instance;
  }

  /**
   * Access callback for the generate route.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(NodeInterface $node, AccountInterface $account): AccessResultInterface {
    return AccessResult::allowedIf($node->access('update', $account))
      ->andIf(AccessResult::allowedIfHasPermission($account, 'social_summaries.generate'));
  }

  /**
   * Generate content for a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param string $platform
   *   The content platform (summary, linkedin, x, etc.).
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function generate(NodeInterface $node, string $platform, Request $request): JsonResponse {
    // Additional access check (redundant but safe)
    if (!$node->access('update')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    // CSRF header is enforced at route level.
    // Check rate limits.
    if (!$this->rateLimiter->canMakeCall()) {
      return new JsonResponse(['error' => 'Rate limit exceeded. Please try again later.'], 429);
    }

    // Validate platform parameter.
    $allowed_platforms = ['summary', 'linkedin', 'x', 'newsletter', 'facebook', 'threads', 'all'];
    if (!in_array($platform, $allowed_platforms, TRUE)) {
      return new JsonResponse(['error' => 'Invalid platform parameter'], 400);
    }

    // Read JSON body if present.
    $data = [];
    if (str_starts_with((string) $request->headers->get('Content-Type'), 'application/json')) {
      $decoded = json_decode((string) $request->getContent(), TRUE);
      if (is_array($decoded)) {
        $data = $decoded;
      }
    }

    // Sanitize and validate input parameters.
    $title = $data['title'] ?? $request->query->get('title', (string) $node->label());
    $title = strip_tags($title);
    // Limit title length.
    $title = substr($title, 0, 255);

    $body = $data['body'] ?? $request->query->get('body', $node->hasField('body') ? (string) $node->get('body')->value : '');
    $body = strip_tags($body);
    // Limit body length.
    $body = substr($body, 0, 10000);
    $config = $this->config('social_summaries.settings');
    $preview_only = (bool) ($data['preview_only'] ?? $request->query->get('preview_only', FALSE));

    $platforms = ($platform === 'all') ? ['summary', 'linkedin', 'x', 'newsletter', 'facebook', 'threads'] : [$platform];
    $results = [];
    $errors = [];
    $generated_content = [];

    foreach ($platforms as $p) {
      try {
        $messages = $this->manager->buildMessages($title, $body, $p);
        // Load provider plugin based on configuration.
        $provider_id = (string) ($config->get('provider') ?? $this->providerManager->getDefaultProvider());
        $provider = $this->providerManager->createInstance($provider_id);
        $resp = $provider->generate($messages);
        $text = trim($resp['text'] ?? '');
        $tokens_in = (int) ($resp['usage']['tokens_in'] ?? 0);
        $tokens_out = (int) ($resp['usage']['tokens_out'] ?? 0);
        $model = $resp['model'] ?? 'unknown';

        $generated_content[$p] = [
          'text' => $text,
          'tokens_in' => $tokens_in,
          'tokens_out' => $tokens_out,
          'model' => $model,
        ];

        // Log usage regardless of preview mode.
        $node_id = $node->id() ? (int) $node->id() : NULL;
        $this->manager->logUsage($node_id, $p, $model, $tokens_in, $tokens_out);

        // Only save to node if not preview only.
        if (!$preview_only) {
          switch ($p) {
            case 'summary':
              if ($node->hasField('field_ai_summary')) {
                $node->set('field_ai_summary', $text);
              }

              break;

            case 'linkedin':
              if ($node->hasField('field_ai_social_linkedin')) {
                $node->set('field_ai_social_linkedin', $text);
              }

              break;

            case 'x':
              if ($node->hasField('field_ai_social_x')) {
                $node->set('field_ai_social_x', $text);
              }

              break;

            case 'newsletter':
              if ($node->hasField('field_ai_newsletter_blurb')) {
                $node->set('field_ai_newsletter_blurb', $text);
              }

              break;

            case 'facebook':
              if ($node->hasField('field_ai_social_facebook')) {
                $node->set('field_ai_social_facebook', $text);
              }

              break;

            case 'threads':
              if ($node->hasField('field_ai_social_threads')) {
                $node->set('field_ai_social_threads', $text);
              }

              break;
          }
        }

        $results[$p] = TRUE;
      }
      catch (\Throwable $e) {
        // Log detailed error for administrators.
        $this->logger->error(
          'AI generation failed for {platform} on node {nid}: {error}',
          ['platform' => $p, 'nid' => $node->id(), 'error' => $e->getMessage()]
        );

        // Return generic error message to prevent information disclosure.
        $errors[$p] = 'Content generation failed. Please try again.';
      }
    }

    if (!$preview_only) {
      try {
        $node->setNewRevision(TRUE);
        $node->save();
      }
      catch (\Throwable $e) {
        // Log detailed error for administrators.
        $this->logger->error(
          'Failed to save node {nid}: {error}',
          ['nid' => $node->id(), 'error' => $e->getMessage()]
        );

        // Return generic error message to prevent information disclosure.
        $errors['save'] = 'Failed to save content. Please try again.';
      }
    }

    $response_data = [
      'ok' => empty($errors),
      'results' => $results,
      'errors' => $errors,
    ];

    if ($preview_only) {
      $response_data['generated_content'] = $generated_content;
    }

    return new JsonResponse($response_data);
  }

}
