<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\social_summaries\Form\UsageFilterForm;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for displaying AI usage and cost statistics.
 */
class UsageController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The pager manager service.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  protected PagerManagerInterface $pagerManager;

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected FormBuilderInterface $formBuilderService;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Constructs a UsageController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Pager\PagerManagerInterface $pager_manager
   *   The pager manager service.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   */
  public function __construct(Connection $database, DateFormatterInterface $date_formatter, PagerManagerInterface $pager_manager, FormBuilderInterface $form_builder, RequestStack $request_stack) {
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
    $this->pagerManager = $pager_manager;
    $this->formBuilderService = $form_builder;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('date.formatter'),
      $container->get('pager.manager'),
      $container->get('form_builder'),
      $container->get('request_stack')
    );
  }

  /**
   * Access callback for usage statistics pages.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account): AccessResultInterface {
    return AccessResult::allowedIfHasPermission($account, 'social_summaries.view_usage');
  }

  /**
   * Access callback for cost data pages.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function accessCosts(AccountInterface $account): AccessResultInterface {
    return AccessResult::allowedIfHasPermission($account, 'social_summaries.view_costs');
  }

  /**
   * Display usage statistics overview.
   *
   * @return array
   *   A render array.
   */
  public function overview(): array {
    $build = [];
    $build['#attached']['library'][] = 'social_summaries/usage';

    // Get summary statistics.
    $stats = $this->getSummaryStats();

    $build['summary'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['social-summaries-stats-summary']],
    ];

    $build['summary']['title'] = [
      '#markup' => '<h2>' . $this->t('Usage Overview') . '</h2>',
    ];

    $build['summary']['stats'] = [
      '#theme' => 'item_list',
      '#list_type' => 'ul',
      '#items' => [
        $this->t('Total API calls: @count', ['@count' => number_format($stats['total_calls'])]),
        $this->t('Total tokens used: @count', ['@count' => number_format($stats['total_tokens'])]),
        $this->t('Total cost: $@cost', ['@cost' => number_format($stats['total_cost'], 4)]),
        $this->t('Calls today: @count', ['@count' => number_format($stats['calls_today'])]),
        $this->t('Cost today: $@cost', ['@cost' => number_format($stats['cost_today'], 4)]),
      ],
    ];

    // Get platform breakdown.
    $platform_stats = $this->getPlatformStats();
    if (!empty($platform_stats)) {
      $build['platforms'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['social-summaries-platform-stats']],
      ];

      $build['platforms']['title'] = [
        '#markup' => '<h3>' . $this->t('Usage by Platform') . '</h3>',
      ];

      $platform_items = [];
      foreach ($platform_stats as $platform => $data) {
        $platform_items[] = $this->t('@platform: @calls calls, @tokens tokens, $@cost', [
          '@platform' => ucfirst($platform),
          '@calls' => number_format($data['calls']),
          '@tokens' => number_format($data['tokens']),
          '@cost' => number_format($data['cost'], 4),
        ]);
      }

      $build['platforms']['list'] = [
        '#theme' => 'item_list',
        '#list_type' => 'ul',
        '#items' => $platform_items,
      ];
    }

    return $build;
  }

  /**
   * Display detailed usage log.
   *
   * @return array
   *   A render array.
   */
  public function log(): array {
    $build = [];
    $build['#attached']['library'][] = 'social_summaries/usage';

    // Get filter parameters.
    $request = $this->requestStack->getCurrentRequest();
    $platform = $request->query->get('platform', '');
    $date_from = $request->query->get('date_from', '');
    $date_to = $request->query->get('date_to', '');

    // Build filter form.
    $build['filters'] = $this->formBuilderService->getForm(UsageFilterForm::class);

    // Get usage data.
    $usage_data = $this->getUsageData($platform, $date_from, $date_to);

    if (empty($usage_data)) {
      $build['no_data'] = [
        '#markup' => '<div class="social-summaries-no-data"><p>' . $this->t('No usage data found for the selected criteria.') . '</p></div>',
      ];
      return $build;
    }

    // Build table.
    $build['table'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['social-summaries-usage-table']],
      'table' => [
        '#type' => 'table',
        '#header' => [
          $this->t('Date'),
          $this->t('Node ID'),
          $this->t('Platform'),
          $this->t('Model'),
          $this->t('Tokens In'),
          $this->t('Tokens Out'),
          $this->t('Total Tokens'),
          $this->t('Cost'),
        ],
        '#rows' => [],
      ],
    ];

    foreach ($usage_data as $row) {
      $build['table']['table']['#rows'][] = [
        $this->dateFormatter->format($row->created, 'short'),
        $row->nid ? (int) $row->nid : $this->t('N/A'),
        ucfirst($row->platform),
        $row->model,
        number_format((int) $row->tokens_in),
        number_format((int) $row->tokens_out),
        number_format((int) $row->tokens_in + (int) $row->tokens_out),
        '$' . number_format((float) $row->cost_usd, 4),
      ];
    }

    // Add pager.
    $this->pagerManager->createPager($usage_data['total'] ?? 0, 50);

    $build['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * Display cost analysis.
   *
   * @return array
   *   A render array.
   */
  public function costs(): array {
    $build = [];
    $build['#attached']['library'][] = 'social_summaries/usage';

    // Get cost statistics.
    $cost_stats = $this->getCostStats();

    $build['summary'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['social-summaries-cost-summary']],
    ];

    $build['summary']['title'] = [
      '#markup' => '<h2>' . $this->t('Cost Analysis') . '</h2>',
    ];

    $build['summary']['stats'] = [
      '#theme' => 'item_list',
      '#list_type' => 'ul',
      '#items' => [
        $this->t('Total cost: $@cost', ['@cost' => number_format($cost_stats['total_cost'], 4)]),
        $this->t('Average cost per call: $@cost', ['@cost' => number_format($cost_stats['avg_cost_per_call'], 4)]),
        $this->t('Cost today: $@cost', ['@cost' => number_format($cost_stats['cost_today'], 4)]),
        $this->t('Cost this month: $@cost', ['@cost' => number_format($cost_stats['cost_month'], 4)]),
      ],
    ];

    // Get model cost breakdown.
    $model_stats = $this->getModelCostStats();
    if (!empty($model_stats)) {
      $build['models'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['social-summaries-model-costs']],
      ];

      $build['models']['title'] = [
        '#markup' => '<h3>' . $this->t('Cost by Model') . '</h3>',
      ];

      $model_items = [];
      foreach ($model_stats as $model => $data) {
        $model_items[] = $this->t('@model: @calls calls, $@cost (@percent% of total)', [
          '@model' => $model,
          '@calls' => number_format($data['calls']),
          '@cost' => number_format($data['cost'], 4),
          '@percent' => number_format($data['percent'], 1),
        ]);
      }

      $build['models']['list'] = [
        '#theme' => 'item_list',
        '#list_type' => 'ul',
        '#items' => $model_items,
      ];
    }

    return $build;
  }

  /**
   * Get summary statistics.
   *
   * @return array
   *   Summary statistics.
   */
  protected function getSummaryStats(): array {
    $query = $this->database->select('social_summaries_usage', 'u');
    $query->addExpression('COUNT(*)', 'total_calls');
    $query->addExpression('SUM(tokens_in + tokens_out)', 'total_tokens');
    $query->addExpression('SUM(cost_usd)', 'total_cost');

    $total_stats = $query->execute()->fetchAssoc();

    // Get today's stats.
    $today_start = strtotime('today');
    $query = $this->database->select('social_summaries_usage', 'u');
    $query->condition('created', $today_start, '>=');
    $query->addExpression('COUNT(*)', 'calls_today');
    $query->addExpression('SUM(cost_usd)', 'cost_today');

    $today_stats = $query->execute()->fetchAssoc();

    return [
      'total_calls' => (int) $total_stats['total_calls'],
      'total_tokens' => (int) $total_stats['total_tokens'],
      'total_cost' => (float) $total_stats['total_cost'],
      'calls_today' => (int) $today_stats['calls_today'],
      'cost_today' => (float) $today_stats['cost_today'],
    ];
  }

  /**
   * Get platform statistics.
   *
   * @return array
   *   Platform statistics.
   */
  protected function getPlatformStats(): array {
    $query = $this->database->select('social_summaries_usage', 'u');
    $query->addField('u', 'platform');
    $query->addExpression('COUNT(*)', 'calls');
    $query->addExpression('SUM(tokens_in + tokens_out)', 'tokens');
    $query->addExpression('SUM(cost_usd)', 'cost');
    $query->groupBy('platform');
    $query->orderBy('calls', 'DESC');

    $results = $query->execute()->fetchAllAssoc('platform');

    $stats = [];
    foreach ($results as $platform => $data) {
      $stats[$platform] = [
        'calls' => (int) $data->calls,
        'tokens' => (int) $data->tokens,
        'cost' => (float) $data->cost,
      ];
    }

    return $stats;
  }

  /**
   * Get usage data with filters.
   *
   * @param string $platform
   *   Platform filter.
   * @param string $date_from
   *   Date from filter.
   * @param string $date_to
   *   Date to filter.
   *
   * @return array
   *   Usage data.
   */
  protected function getUsageData(string $platform = '', string $date_from = '', string $date_to = ''): array {
    $query = $this->database->select('social_summaries_usage', 'u');
    $query->fields('u');

    if (!empty($platform)) {
      $query->condition('platform', $platform);
    }

    if (!empty($date_from)) {
      $timestamp = strtotime($date_from);
      if ($timestamp !== FALSE) {
        $query->condition('created', $timestamp, '>=');
      }
    }

    if (!empty($date_to)) {
      $timestamp = strtotime($date_to . ' 23:59:59');
      if ($timestamp !== FALSE) {
        $query->condition('created', $timestamp, '<=');
      }
    }

    $query->orderBy('created', 'DESC');
    $query->range(0, 50);

    return $query->execute()->fetchAll();
  }

  /**
   * Get cost statistics.
   *
   * @return array
   *   Cost statistics.
   */
  protected function getCostStats(): array {
    $query = $this->database->select('social_summaries_usage', 'u');
    $query->addExpression('COUNT(*)', 'total_calls');
    $query->addExpression('SUM(cost_usd)', 'total_cost');
    $query->addExpression('AVG(cost_usd)', 'avg_cost_per_call');

    $total_stats = $query->execute()->fetchAssoc();

    // Get today's cost.
    $today_start = strtotime('today');
    $query = $this->database->select('social_summaries_usage', 'u');
    $query->condition('created', $today_start, '>=');
    $query->addExpression('SUM(cost_usd)', 'cost_today');

    $today_stats = $query->execute()->fetchAssoc();

    // Get this month's cost.
    $month_start = strtotime('first day of this month');
    $query = $this->database->select('social_summaries_usage', 'u');
    $query->condition('created', $month_start, '>=');
    $query->addExpression('SUM(cost_usd)', 'cost_month');

    $month_stats = $query->execute()->fetchAssoc();

    return [
      'total_cost' => (float) $total_stats['total_cost'],
      'avg_cost_per_call' => (float) $total_stats['avg_cost_per_call'],
      'cost_today' => (float) $today_stats['cost_today'],
      'cost_month' => (float) $month_stats['cost_month'],
    ];
  }

  /**
   * Get model cost statistics.
   *
   * @return array
   *   Model cost statistics.
   */
  protected function getModelCostStats(): array {
    $query = $this->database->select('social_summaries_usage', 'u');
    $query->addField('u', 'model');
    $query->addExpression('COUNT(*)', 'calls');
    $query->addExpression('SUM(cost_usd)', 'cost');
    $query->groupBy('model');
    $query->orderBy('cost', 'DESC');

    $results = $query->execute()->fetchAllAssoc('model');

    // Get total cost for percentage calculation.
    $total_query = $this->database->select('social_summaries_usage', 'u');
    $total_query->addExpression('SUM(cost_usd)', 'total_cost');
    $total_cost = $total_query->execute()->fetchField();

    $stats = [];
    foreach ($results as $model => $data) {
      $percent = $total_cost > 0 ? ($data->cost / $total_cost) * 100 : 0;
      $stats[$model] = [
        'calls' => (int) $data->calls,
        'cost' => (float) $data->cost,
        'percent' => $percent,
      ];
    }

    return $stats;
  }

}
