<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\social_summaries\Plugin\SocialSummariesProvider\SocialSummariesProviderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Social Summaries settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The provider plugin manager.
   *
   * @var \Drupal\social_summaries\Plugin\SocialSummariesProvider\SocialSummariesProviderManager
   */
  protected SocialSummariesProviderManager $providerManager;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\social_summaries\Plugin\SocialSummariesProvider\SocialSummariesProviderManager $provider_manager
   *   The provider plugin manager.
   */
  public function __construct(KeyRepositoryInterface $key_repository, MessengerInterface $messenger, SocialSummariesProviderManager $provider_manager) {
    $this->keyRepository = $key_repository;
    $this->messenger = $messenger;
    $this->providerManager = $provider_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static(
      $container->get('key.repository'),
      $container->get('messenger'),
      $container->get('plugin.manager.social_summaries_provider')
    );
    // Ensure ConfigFormBase has a config factory set in kernel contexts.
    $instance->setConfigFactory($container->get('config.factory'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'social_summaries_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['social_summaries.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('social_summaries.settings');

    $form['provider'] = [
      '#type' => 'select',
      '#title' => $this->t('Provider'),
      '#options' => $this->providerManager->getProviderOptions(),
      '#default_value' => $config->get('provider') ?? $this->providerManager->getDefaultProvider(),
      '#description' => $this->t('Select the AI provider to use for content generation.'),
    ];

    // Get available keys for selection.
    $keys = $this->keyRepository->getKeys();
    $key_options = ['' => $this->t('- Select a key -')];
    foreach ($keys as $key_id => $key) {
      $key_options[$key_id] = $key->label();
    }

    $form['openai_api_key_id'] = [
      '#type' => 'select',
      '#title' => $this->t('API Key'),
      '#options' => $key_options,
      '#default_value' => $config->get('openai_api_key_id') ?? '',
      '#description' => $this->t('Select a key from the Key module. Create a new key at <a href="@url">Configuration > System > Key</a> if needed.', [
        '@url' => '/admin/config/system/key',
      ]),
      // Not strictly required to allow saving other configuration during tests.
      '#required' => FALSE,
    ];

    $form['model'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default model'),
      '#default_value' => $config->get('model') ?? 'gpt-4.1-mini',
    ];

    $form['tone'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Tone'),
      '#options' => [
        'neutral, professional' => $this->t('Neutral, Professional'),
        'casual, friendly' => $this->t('Casual, Friendly'),
        'authoritative, expert' => $this->t('Authoritative, Expert'),
        'conversational, engaging' => $this->t('Conversational, Engaging'),
        'concise, direct' => $this->t('Concise, Direct'),
        'warm, personal' => $this->t('Warm, Personal'),
        'technical, detailed' => $this->t('Technical, Detailed'),
        'creative, inspiring' => $this->t('Creative, Inspiring'),
      ],
      '#default_value' => $config->get('tone') ?? 'neutral, professional',
      '#description' => $this->t('The default tone used in AI-generated content. This can be customized per generation if needed.'),
    ];

    $form['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#min' => 0,
      '#max' => 2,
      '#step' => 0.1,
      '#default_value' => $config->get('temperature') ?? 0.5,
    ];

    $form['max_tokens'] = [
      '#type' => 'number',
      '#title' => $this->t('Max tokens'),
      '#min' => 64,
      '#max' => 4096,
      '#default_value' => $config->get('max_tokens') ?? 400,
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['advanced']['max_retries'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum retry attempts'),
      '#min' => 1,
      '#max' => 10,
      '#default_value' => $config->get('max_retries') ?? 3,
      '#description' => $this->t('Number of times to retry failed requests.'),
    ];

    $form['advanced']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request timeout (seconds)'),
      '#min' => 10,
      '#max' => 300,
      '#default_value' => $config->get('timeout') ?? 60,
      '#description' => $this->t('Maximum time to wait for API responses.'),
    ];

    $form['rate_limiting'] = [
      '#type' => 'details',
      '#title' => $this->t('Rate Limiting'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['rate_limiting']['max_calls_per_minute'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum calls per minute'),
      '#min' => 1,
      '#max' => 100,
      '#default_value' => $config->get('max_calls_per_minute') ?? 20,
      '#description' => $this->t('Maximum number of API calls allowed per minute. Conservative default to protect your API quota.'),
    ];

    $form['rate_limiting']['max_calls_per_hour'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum calls per hour'),
      '#min' => 10,
      '#max' => 2000,
      '#default_value' => $config->get('max_calls_per_hour') ?? 200,
      '#description' => $this->t('Maximum number of API calls allowed per hour. Conservative default to protect your API quota.'),
    ];

    $form['rate_limiting']['base_delay'] = [
      '#type' => 'number',
      '#title' => $this->t('Base delay between calls (seconds)'),
      '#min' => 0,
      '#max' => 10,
      '#step' => 0.1,
      '#default_value' => $config->get('base_delay') ?? 3.0,
      '#description' => $this->t('Base delay between API calls to prevent rate limiting. Higher values protect your API quota.'),
    ];

    $form['rate_limiting']['max_delay'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum delay (seconds)'),
      '#min' => 1,
      '#max' => 60,
      '#default_value' => $config->get('max_delay') ?? 15.0,
      '#description' => $this->t('Maximum delay between API calls when rate limiting is active. Higher values protect your API quota.'),
    ];

    $form['test_connection'] = [
      '#type' => 'details',
      '#title' => $this->t('Test Connection'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['test_connection']['test_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#ajax' => [
        'callback' => '::testConnectionAjax',
        'event' => 'click',
        'progress' => ['type' => 'throbber'],
      ],
    ];

    $form['test_connection']['test_result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'social-summaries-test-result'],
    ];

    $form['prompt_templates'] = [
      '#type' => 'details',
      '#title' => $this->t('Prompt Templates'),
      '#open' => FALSE,
      '#description' => $this->t('Customize the AI prompts used for content generation. Use placeholders like {tone}, {constraints}, {title}, {body}, and {platform} in your templates.'),
      '#tree' => TRUE,
    ];

    $form['prompt_templates']['system_prompt_template'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System Prompt Template'),
      '#default_value' => $config->get('system_prompt_template') ?: 'You are a copywriter in {tone} voice. Follow these constraints: {constraints}',
      '#rows' => 3,
      '#description' => $this->t('Base system prompt. Available placeholders: {tone}, {constraints}'),
    ];

    // Platform-specific prompt templates are now configured under Platform Constraints.
    $form['prompt_templates']['reset_templates'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset to Defaults'),
      '#submit' => ['::resetPromptTemplates'],
      '#limit_validation_errors' => [],
    ];

    $form['platform_visibility'] = [
      '#type' => 'details',
      '#title' => $this->t('Visible Platforms (Global)'),
      '#open' => FALSE,
      '#description' => $this->t('Configure which platforms are visible by default across all content types. Individual content types can override these settings.'),
      '#tree' => TRUE,
      '#weight' => 60,
    ];

    // Constraints configuration.
    $form['constraints'] = [
      '#type' => 'details',
      '#title' => $this->t('Platform Constraints'),
      '#open' => FALSE,
      '#tree' => TRUE,
      '#weight' => 50,
    ];

    $form['constraints']['help'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('These are recommended guidance values, not required hard limits. The AI may occasionally exceed these, but the UI will highlight when content is outside the recommended range.') . '</p>',
    ];

    $constraints = $config->get('constraints') ?? [];
    // Defaults that mirror the previously hard-coded values.
    $default_constraints = [
      'linkedin' => [
        'type' => 'character',
        'min_chars' => 150,
        'max_chars' => 220,
        'recommended_min' => 150,
        'recommended_max' => 220,
      ],
      'x' => [
        'type' => 'character',
        'min_chars' => 1,
        'max_chars' => 260,
        'recommended_min' => 1,
        'recommended_max' => 260,
      ],
      'facebook' => [
        'type' => 'character',
        'min_chars' => 1,
        'max_chars' => 500,
        'recommended_min' => 1,
        'recommended_max' => 500,
      ],
      'threads' => [
        'type' => 'character',
        'min_chars' => 1,
        'max_chars' => 500,
        'recommended_min' => 1,
        'recommended_max' => 500,
      ],
      'summary' => [
        'type' => 'word',
        'min_words' => 70,
        'max_words' => 120,
      ],
      'newsletter' => [
        'type' => 'sentence',
        'min_sentences' => 1,
        'max_sentences' => 2,
      ],
    ];
    $constraint_platforms = [
      'summary' => $this->t('Summary'),
      'linkedin' => $this->t('LinkedIn'),
      'x' => $this->t('X'),
      'newsletter' => $this->t('Newsletter'),
      'facebook' => $this->t('Facebook'),
      'threads' => $this->t('Threads'),
    ];

    foreach ($constraint_platforms as $platform_key => $platform_label) {
      $platform_constraints = array_merge($default_constraints[$platform_key] ?? [], $constraints[$platform_key] ?? []);
      $form['constraints'][$platform_key] = [
        '#type' => 'details',
        '#title' => $platform_label,
        '#open' => FALSE,
      ];

      // Per-platform prompt template (moved from global prompt templates).
      $form['constraints'][$platform_key]['prompt_template'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Prompt guidance'),
        '#default_value' => $platform_constraints['prompt_template'] ?? '',
        '#rows' => 2,
      ];

      // Type field (readonly display)
      $form['constraints'][$platform_key]['type'] = [
        '#type' => 'value',
        '#value' => $platform_constraints['type'] ?? '',
      ];

      if (in_array($platform_key, ['linkedin', 'x', 'facebook', 'threads'], TRUE)) {
        $form['constraints'][$platform_key]['min_chars'] = [
          '#type' => 'number',
          '#title' => $this->t('Minimum characters'),
          '#default_value' => $platform_constraints['min_chars'] ?? '',
          '#min' => 0,
        ];
        $form['constraints'][$platform_key]['max_chars'] = [
          '#type' => 'number',
          '#title' => $this->t('Maximum characters'),
          '#default_value' => $platform_constraints['max_chars'] ?? '',
          '#min' => 0,
        ];
        $form['constraints'][$platform_key]['recommended_min'] = [
          '#type' => 'number',
          '#title' => $this->t('Recommended minimum'),
          '#default_value' => $platform_constraints['recommended_min'] ?? '',
          '#min' => 0,
        ];
        $form['constraints'][$platform_key]['recommended_max'] = [
          '#type' => 'number',
          '#title' => $this->t('Recommended maximum'),
          '#default_value' => $platform_constraints['recommended_max'] ?? '',
          '#min' => 0,
        ];
      }
      elseif ($platform_key === 'summary') {
        $form['constraints'][$platform_key]['min_words'] = [
          '#type' => 'number',
          '#title' => $this->t('Minimum words'),
          '#default_value' => $platform_constraints['min_words'] ?? '',
          '#min' => 0,
        ];
        $form['constraints'][$platform_key]['max_words'] = [
          '#type' => 'number',
          '#title' => $this->t('Maximum words'),
          '#default_value' => $platform_constraints['max_words'] ?? '',
          '#min' => 0,
        ];
      }
      elseif ($platform_key === 'newsletter') {
        $form['constraints'][$platform_key]['min_sentences'] = [
          '#type' => 'number',
          '#title' => $this->t('Minimum sentences'),
          '#default_value' => $platform_constraints['min_sentences'] ?? '',
          '#min' => 0,
        ];
        $form['constraints'][$platform_key]['max_sentences'] = [
          '#type' => 'number',
          '#title' => $this->t('Maximum sentences'),
          '#default_value' => $platform_constraints['max_sentences'] ?? '',
          '#min' => 0,
        ];
      }
    }

    $platforms = [
      'summary' => $this->t('Summary'),
      'linkedin' => $this->t('LinkedIn'),
      'x' => $this->t('X (Twitter)'),
      'newsletter' => $this->t('Newsletter'),
      'facebook' => $this->t('Facebook'),
      'threads' => $this->t('Threads'),
    ];

    $form['platform_visibility']['visible_platforms'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Visible Platforms'),
      '#description' => $this->t('Select which platforms should be visible in the Social Summaries sidebar by default. Start with fewer platforms to protect your API quota.'),
      '#options' => $platforms,
      '#default_value' => $config->get('visible_platforms') ?? ['summary', 'linkedin'],
      '#required' => TRUE,
    ];

    // Add form actions.
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
      '#button_type' => 'primary',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    // Validate API key selection.
    $api_key_id = $form_state->getValue('openai_api_key_id');
    if (empty($api_key_id)) {
      // Make this a warning instead of an error for testing.
      $this->messenger->addWarning($this->t('No OpenAI API key selected. AI generation will not work without a valid API key.'));
    }

    // Validate model name.
    $model = $form_state->getValue('model');
    if (empty($model)) {
      // Make this a warning instead of an error for testing.
      $this->messenger->addWarning($this->t('No model name entered. Using default model.'));
    }

    // Validate temperature.
    $temperature = $form_state->getValue('temperature');
    if ($temperature < 0 || $temperature > 2) {
      // Make this a warning instead of an error for testing.
      $this->messenger->addWarning($this->t('Temperature should be between 0 and 2. Using default value.'));
    }

    // Validate max tokens.
    $max_tokens = $form_state->getValue('max_tokens');
    if ($max_tokens < 64 || $max_tokens > 4096) {
      // Make this a warning instead of an error for testing.
      $this->messenger->addWarning($this->t('Max tokens should be between 64 and 4096. Using default value.'));
    }

    // Validate visible platforms - ensure at least one is selected.
    $visible_platforms = $form_state->getValue(['platform_visibility', 'visible_platforms']);
    if (is_array($visible_platforms)) {
      $selected_platforms = array_filter($visible_platforms, function ($value) {
        return $value !== 0 && $value !== FALSE && $value !== '';
      });
      if (empty($selected_platforms)) {
        $form_state->setErrorByName('platform_visibility][visible_platforms', $this->t('Please select at least one platform to be visible.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $visible_platforms = $form_state->getValue(['platform_visibility', 'visible_platforms']);

    // Filter out unchecked values (they come as 0) and ensure we have an array.
    if (is_array($visible_platforms)) {
      $visible_platforms = array_filter($visible_platforms, function ($value) {
        return $value !== 0 && $value !== FALSE && $value !== '';
      });
    }
    else {
      $visible_platforms = [];
    }

    // Get form values with proper defaults.
    $provider = $form_state->getValue('provider') ?: 'openai';
    $api_key_id = $form_state->getValue('openai_api_key_id') ?: '';
    $model = $form_state->getValue('model') ?: 'gpt-4o-mini';
    $tone = $form_state->getValue('tone') ?: 'neutral, professional';
    $temperature = (float) ($form_state->getValue('temperature') ?: 0.5);
    $max_tokens = (int) ($form_state->getValue('max_tokens') ?: 400);

    // Advanced settings with defaults.
    $max_retries = (int) ($form_state->getValue(['advanced', 'max_retries']) ?: 3);
    $timeout = (int) ($form_state->getValue(['advanced', 'timeout']) ?: 60);

    // Rate limiting with conservative defaults.
    $max_calls_per_minute = (int) ($form_state->getValue(['rate_limiting', 'max_calls_per_minute']) ?: 20);
    $max_calls_per_hour = (int) ($form_state->getValue(['rate_limiting', 'max_calls_per_hour']) ?: 200);
    $base_delay = (float) ($form_state->getValue(['rate_limiting', 'base_delay']) ?: 3.0);
    $max_delay = (float) ($form_state->getValue(['rate_limiting', 'max_delay']) ?: 15.0);

    $config = $this->configFactory->getEditable('social_summaries.settings');
    $config->set('provider', $provider)
      ->set('openai_api_key_id', $api_key_id)
      ->set('model', $model)
      ->set('tone', $tone)
      ->set('temperature', $temperature)
      ->set('max_tokens', $max_tokens)
      ->set('max_retries', $max_retries)
      ->set('timeout', $timeout)
      ->set('max_calls_per_minute', $max_calls_per_minute)
      ->set('max_calls_per_hour', $max_calls_per_hour)
      ->set('base_delay', $base_delay)
      ->set('max_delay', $max_delay)
      ->set('visible_platforms', $visible_platforms)
      ->set('system_prompt_template', (string) $form_state->getValue(['prompt_templates', 'system_prompt_template']))
      ->set('summary_prompt_template', (string) $form_state->getValue(['prompt_templates', 'summary_prompt_template']))
      ->set('linkedin_prompt_template', (string) $form_state->getValue(['prompt_templates', 'linkedin_prompt_template']))
      ->set('x_prompt_template', (string) $form_state->getValue(['prompt_templates', 'x_prompt_template']))
      ->set(
        'newsletter_prompt_template',
        (string) $form_state->getValue([
          'prompt_templates',
          'newsletter_prompt_template',
        ])
      )
      ->set('facebook_prompt_template', (string) $form_state->getValue(['prompt_templates', 'facebook_prompt_template']))
      ->set('threads_prompt_template', (string) $form_state->getValue(['prompt_templates', 'threads_prompt_template']))
      ->set('constraints', (array) $form_state->getValue(['constraints']));
    $config->save();

    // Ensure AI fields are present on all bundles for globally visible platforms.
    if (!empty($visible_platforms) && function_exists('social_summaries_ensure_fields_on_all_bundles')) {
      \social_summaries_ensure_fields_on_all_bundles(array_values($visible_platforms));
    }

    $this->messenger()->addStatus($this->t('The configuration options have been saved.'));
  }

  /**
   * AJAX callback for testing the connection.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function testConnectionAjax(array &$form, FormStateInterface $form_state): AjaxResponse {
    $provider_id = (string) ($this->config('social_summaries.settings')->get('provider') ?? $this->providerManager->getDefaultProvider());
    $provider = $this->providerManager->createInstance($provider_id);
    $result = $provider->testConnection();

    $response = new AjaxResponse();

    if ($result['success']) {
      $message = '<div class="messages messages--status">' .
                 '<strong>Success:</strong> ' . $result['message'];
      if (!empty($result['details'])) {
        $message .= '<br><small>Model: ' . $result['details']['model'] .
                   ', Tokens used: ' . $result['details']['tokens_used'] . '</small>';
      }
      $message .= '</div>';
    }
    else {
      $message = '<div class="messages messages--error">' .
                 '<strong>Error:</strong> ' . $result['message'] . '</div>';
    }

    $response->addCommand(new HtmlCommand('#social-summaries-test-result', $message));
    return $response;
  }

  /**
   * Reset prompt templates to defaults.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function resetPromptTemplates(array &$form, FormStateInterface $form_state): void {
    $this->configFactory->getEditable('social_summaries.settings')
      ->set('system_prompt_template', 'You are a copywriter in {tone} voice. Follow these constraints: {constraints}')
      ->set('summary_prompt_template', 'Concise, scannable, professional.')
      ->set('linkedin_prompt_template', 'Professional tone, add 1-2 relevant hashtags, end with soft CTA.')
      ->set('x_prompt_template', 'Concise, punchy, add 1-2 relevant hashtags.')
      ->set('newsletter_prompt_template', 'Clear, engaging summary plus a suggested subject line.')
      ->set('facebook_prompt_template', 'Conversational tone, encourage engagement, add 1-2 relevant hashtags.')
      ->set('threads_prompt_template', 'Conversational tone, similar to X format, add 1-2 relevant hashtags.')
      ->save();

    $this->messenger()->addStatus($this->t('Prompt templates have been reset to default values.'));
    $form_state->setRebuild(TRUE);
  }

}
