<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Plugin\SocialSummariesProvider;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Anthropic provider plugin for Social Summaries.
 *
 * @SocialSummariesProvider(
 *   id = "anthropic",
 *   label = @Translation("Anthropic"),
 *   description = @Translation("Generate content using Anthropic's Claude models."),
 *   api_documentation = "https://docs.anthropic.com/claude/reference/getting-started-with-the-api",
 *   requires_api_key = TRUE,
 *   default_model = "claude-3-5-sonnet-20241022",
 *   max_tokens = 4096,
 *   default_temperature = 0.5
 * )
 */
class AnthropicProvider extends SocialSummariesProviderBase {

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The HTTP client service.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Constructs an AnthropicProvider object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(array $configuration, string $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, KeyRepositoryInterface $key_repository, ClientInterface $http_client, TranslationInterface $string_translation) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $config_factory, $string_translation);
    $this->keyRepository = $key_repository;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('http_client'),
      $container->get('string_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedModels(): array {
    return [
      'claude-3-5-sonnet-20241022' => 'Claude 3.5 Sonnet',
      'claude-3-5-haiku-20241022' => 'Claude 3.5 Haiku',
      'claude-3-opus-20240229' => 'Claude 3 Opus',
      'claude-3-sonnet-20240229' => 'Claude 3 Sonnet',
      'claude-3-haiku-20240307' => 'Claude 3 Haiku',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getRateLimits(): array {
    return [
      'requests_per_minute' => 50,
      'requests_per_hour' => 1000,
      'tokens_per_minute' => 200000,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function calculateCost(int $tokens_in, int $tokens_out, string $model): float {
    // Anthropic pricing as of 2024 (per 1M tokens)
    $pricing = [
      'claude-3-5-sonnet-20241022' => ['input' => 3.00, 'output' => 15.00],
      'claude-3-5-haiku-20241022' => ['input' => 1.00, 'output' => 5.00],
      'claude-3-opus-20240229' => ['input' => 15.00, 'output' => 75.00],
      'claude-3-sonnet-20240229' => ['input' => 3.00, 'output' => 15.00],
      'claude-3-haiku-20240307' => ['input' => 0.25, 'output' => 1.25],
    ];

    if (!isset($pricing[$model])) {
      return 0.0;
    }

    $input_cost = ($tokens_in / 1000000) * $pricing[$model]['input'];
    $output_cost = ($tokens_out / 1000000) * $pricing[$model]['output'];

    return $input_cost + $output_cost;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $config = $this->configFactory->get('social_summaries.settings');

    // Get available keys for selection.
    $keys = $this->keyRepository->getKeys();
    $key_options = ['' => $this->t('- Select a key -')];
    foreach ($keys as $key_id => $key) {
      $key_options[$key_id] = $key->label();
    }

    $form['api_key'] = [
      '#type' => 'select',
      '#title' => $this->t('Anthropic API Key'),
      '#options' => $key_options,
      '#default_value' => $config->get('anthropic_api_key_id') ?? '',
      '#description' => $this->t('Select a key from the Key module. Create a new key at <a href="@url">Configuration > System > Key</a> if needed.', [
        '@url' => '/admin/config/system/key',
      ]),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function generate(array $messages, array $options = []): array {
    $config = $this->configFactory->get('social_summaries.settings');

    // Get API key.
    $api_key_id = $config->get('anthropic_api_key_id');
    if (empty($api_key_id)) {
      throw new \RuntimeException('Anthropic API key not configured.');
    }

    $key = $this->keyRepository->getKey($api_key_id);
    if (!$key) {
      throw new \RuntimeException('Anthropic API key not found.');
    }

    $api_key = $key->getKeyValue();
    if (empty($api_key)) {
      throw new \RuntimeException('Anthropic API key is empty.');
    }

    // Convert OpenAI-style messages to Anthropic format.
    $anthropic_messages = $this->convertMessagesToAnthropicFormat($messages);

    $model = $options['model'] ?? $this->pluginDefinition['default_model'];
    $temperature = $options['temperature'] ?? $this->pluginDefinition['default_temperature'];
    $max_tokens = $options['max_tokens'] ?? 400;

    $payload = [
      'model' => $model,
      'max_tokens' => $max_tokens,
      'temperature' => $temperature,
      'messages' => $anthropic_messages,
    ];

    try {
      $response = $this->httpClient->request('POST', 'https://api.anthropic.com/v1/messages', [
        'headers' => [
          'x-api-key' => $api_key,
          'Content-Type' => 'application/json',
          'anthropic-version' => '2023-06-01',
        ],
        'json' => $payload,
        'timeout' => 60,
      ]);

      $data = json_decode((string) $response->getBody(), TRUE);

      if (!isset($data['content']) || !is_array($data['content']) || empty($data['content'])) {
        throw new \RuntimeException('Invalid response from Anthropic API: no content returned.');
      }

      $text = trim($data['content'][0]['text'] ?? '');
      if (empty($text)) {
        throw new \RuntimeException('Anthropic API returned empty content.');
      }

      $usage = $data['usage'] ?? ['input_tokens' => 0, 'output_tokens' => 0];

      return [
        'text' => $text,
        'usage' => [
          'tokens_in' => (int) ($usage['input_tokens'] ?? 0),
          'tokens_out' => (int) ($usage['output_tokens'] ?? 0),
        ],
        'model' => $model,
      ];
    }
    catch (\Exception $e) {
      throw new \RuntimeException('Anthropic API request failed: ' . $e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function testConnection(): array {
    try {
      $result = $this->generate([
        [
          'role' => 'user',
          'content' => 'Hello, this is a test message. Please respond with "Connection successful."',
        ],
      ]);

      return [
        'success' => TRUE,
        'message' => $this->t('Connection to Anthropic successful.'),
        'details' => [
          'model' => $result['model'] ?? 'unknown',
          'tokens_used' => ($result['usage']['tokens_in'] ?? 0) + ($result['usage']['tokens_out'] ?? 0),
        ],
      ];
    }
    catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => $this->t('Connection failed: @error', ['@error' => $e->getMessage()]),
      ];
    }
  }

  /**
   * Converts OpenAI-style messages to Anthropic format.
   *
   * @param array $messages
   *   OpenAI-style messages.
   *
   * @return array
   *   Anthropic-style messages.
   */
  protected function convertMessagesToAnthropicFormat(array $messages): array {
    $anthropic_messages = [];

    foreach ($messages as $message) {
      $anthropic_messages[] = [
        'role' => $message['role'],
        'content' => $message['content'],
      ];
    }

    return $anthropic_messages;
  }

}
