<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Plugin\SocialSummariesProvider;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Social Summaries provider plugins.
 */
abstract class SocialSummariesProviderBase extends PluginBase implements SocialSummariesProviderInterface, ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a SocialSummariesProviderBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(array $configuration, string $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, TranslationInterface $string_translation) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->stringTranslation = $string_translation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('string_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultConfiguration(): array {
    return [
      'model' => $this->pluginDefinition['default_model'],
      'temperature' => $this->pluginDefinition['default_temperature'],
      'max_tokens' => $this->pluginDefinition['max_tokens'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedModels(): array {
    return [
      $this->pluginDefinition['default_model'] => $this->pluginDefinition['default_model'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getRateLimits(): array {
    return [
      'requests_per_minute' => 60,
      'requests_per_hour' => 1000,
      'tokens_per_minute' => 150000,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function calculateCost(int $tokens_in, int $tokens_out, string $model): float {
    // Default implementation - should be overridden by specific providers.
    return 0.0;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, $form_state): array {
    $config = $this->configFactory->get('social_summaries.settings');

    $form['model'] = [
      '#type' => 'select',
      '#title' => $this->t('Model'),
      '#options' => $this->getSupportedModels(),
      '#default_value' => $config->get('model') ?: $this->pluginDefinition['default_model'],
      '#description' => $this->t('Select the AI model to use for content generation.'),
    ];

    $form['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#min' => 0,
      '#max' => 2,
      '#step' => 0.1,
      '#default_value' => $config->get('temperature') ?: $this->pluginDefinition['default_temperature'],
      '#description' => $this->t('Controls randomness in the output. Lower values are more focused and deterministic.'),
    ];

    $form['max_tokens'] = [
      '#type' => 'number',
      '#title' => $this->t('Max tokens'),
      '#min' => 64,
      '#max' => $this->pluginDefinition['max_tokens'],
      '#default_value' => $config->get('max_tokens') ?: 400,
      '#description' => $this->t('Maximum number of tokens to generate.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array $form, $form_state): void {
    // Default implementation - can be overridden by specific providers.
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array $form, $form_state): void {
    // Default implementation - can be overridden by specific providers.
  }

  /**
   * Gets the plugin label.
   *
   * @return string
   *   The plugin label.
   */
  public function getLabel(): string {
    return $this->pluginDefinition['label'];
  }

  /**
   * Gets the plugin description.
   *
   * @return string
   *   The plugin description.
   */
  public function getDescription(): string {
    return $this->pluginDefinition['description'];
  }

  /**
   * Gets the API documentation URL.
   *
   * @return string
   *   The API documentation URL.
   */
  public function getApiDocumentation(): string {
    return $this->pluginDefinition['api_documentation'];
  }

  /**
   * Checks if the provider requires an API key.
   *
   * @return bool
   *   TRUE if the provider requires an API key, FALSE otherwise.
   */
  public function requiresApiKey(): bool {
    return $this->pluginDefinition['requires_api_key'];
  }

}
