<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Provider;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\ServerException;
use GuzzleHttp\Exception\ClientException;
use Psr\Log\LoggerInterface;

/**
 * OpenAI provider implementation.
 */
class OpenAiProvider implements AiProviderInterface {

  /**
   * The HTTP client service.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs an OpenAiProvider object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, KeyRepositoryInterface $key_repository, LoggerInterface $logger) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->keyRepository = $key_repository;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public function name(): string {
    return 'openai';
  }

  /**
   * {@inheritdoc}
   */
  public function models(): array {
    return [
      ['id' => 'gpt-4o-mini'],
      ['id' => 'gpt-4o'],
      ['id' => 'gpt-4.1-mini'],
      ['id' => 'gpt-3.5-turbo'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function generate(array $messages, array $options = []): array {
    $config = $this->configFactory->get('social_summaries.settings');
    $keyId = $config->get('openai_api_key_id');
    $model = $options['model'] ?? $config->get('model') ?: 'gpt-4.1-mini';
    $temperature = (float) ($options['temperature'] ?? $config->get('temperature') ?: 0.5);
    $max_tokens = (int) ($options['max_tokens'] ?? $config->get('max_tokens') ?: 400);
    $max_retries = (int) ($options['max_retries'] ?? $config->get('max_retries') ?: 3);
    $timeout = (int) ($options['timeout'] ?? $config->get('timeout') ?: 60);

    // Validate configuration.
    if (empty($keyId)) {
      throw new \RuntimeException('OpenAI API key not configured. Please select a key in the Social Summaries settings.');
    }

    $key = $this->keyRepository->getKey($keyId);
    if (!$key) {
      throw new \RuntimeException('Selected OpenAI API key not found.');
    }

    $apiKey = $key->getKeyValue();
    if (empty($apiKey)) {
      throw new \RuntimeException('OpenAI API key is empty.');
    }

    // Validate model.
    $available_models = array_column($this->models(), 'id');
    if (!in_array($model, $available_models)) {
      throw new \RuntimeException("Model '{$model}' is not supported. Available models: " . implode(', ', $available_models));
    }

    $payload = [
      'model' => $model,
      'messages' => $messages,
      'temperature' => $temperature,
      'max_tokens' => $max_tokens,
    ];

    // Attempt request with retry logic.
    for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
      try {
        $resp = $this->httpClient->request('POST', 'https://api.openai.com/v1/chat/completions', [
          'headers' => [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json',
          ],
          'timeout' => $timeout,
          'json' => $payload,
        ]);

        $data = json_decode((string) $resp->getBody(), TRUE);

        // Validate response structure.
        if (!isset($data['choices']) || !is_array($data['choices']) || empty($data['choices'])) {
          throw new \RuntimeException('Invalid response from OpenAI API: no choices returned.');
        }

        $choice = $data['choices'][0];
        if (!isset($choice['message']['content'])) {
          throw new \RuntimeException('Invalid response from OpenAI API: no content in choice.');
        }

        $text = trim($choice['message']['content']);
        if (empty($text)) {
          throw new \RuntimeException('OpenAI API returned empty content.');
        }

        $usage = $data['usage'] ?? ['prompt_tokens' => 0, 'completion_tokens' => 0];

        return [
          'text' => $text,
          'usage' => [
            'tokens_in' => (int) ($usage['prompt_tokens'] ?? 0),
            'tokens_out' => (int) ($usage['completion_tokens'] ?? 0),
          ],
          'model' => $model,
        ];
      }
      catch (ClientException $e) {
        $status_code = $e->getResponse()->getStatusCode();
        $response_body = $e->getResponse()->getBody()->getContents();
        $error_data = json_decode($response_body, TRUE);

        // Handle specific client errors.
        switch ($status_code) {
          case 401:
            throw new \RuntimeException('OpenAI API authentication failed. Please check your API key.');

          case 403:
            throw new \RuntimeException('OpenAI API access forbidden. Please check your API key permissions.');

          case 429:
            // Rate limit - retry with exponential backoff.
            if ($attempt < $max_retries) {
              $retry_after = $e->getResponse()->getHeader('Retry-After')[0] ?? pow(2, $attempt);
              $this->logger->warning('OpenAI rate limit hit, retrying after @seconds seconds (attempt @attempt/@max)', [
                '@seconds' => $retry_after,
                '@attempt' => $attempt,
                '@max' => $max_retries,
              ]);
              sleep($retry_after);
              continue 2;
            }

            throw new \RuntimeException('OpenAI API rate limit exceeded. Please try again later.');

          case 400:
            $error_message = $error_data['error']['message'] ?? 'Bad request';
            throw new \RuntimeException("OpenAI API bad request: {$error_message}");

          default:
            $error_message = $error_data['error']['message'] ?? $e->getMessage();
            throw new \RuntimeException("OpenAI API client error ({$status_code}): {$error_message}");
        }
      }
      catch (ServerException $e) {
        $status_code = $e->getResponse()->getStatusCode();

        // Server errors are retryable.
        if ($attempt < $max_retries) {
          // Exponential backoff.
          $retry_delay = pow(2, $attempt);
          $this->logger->warning('OpenAI server error @code, retrying after @seconds seconds (attempt @attempt/@max)', [
            '@code' => $status_code,
            '@seconds' => $retry_delay,
            '@attempt' => $attempt,
            '@max' => $max_retries,
          ]);
          sleep($retry_delay);
          continue;
        }

        throw new \RuntimeException("OpenAI API server error ({$status_code}). Please try again later.");
      }
      catch (ConnectException $e) {
        // Connection errors are retryable.
        if ($attempt < $max_retries) {
          $retry_delay = pow(2, $attempt);
          $this->logger->warning('OpenAI connection error, retrying after @seconds seconds (attempt @attempt/@max)', [
            '@seconds' => $retry_delay,
            '@attempt' => $attempt,
            '@max' => $max_retries,
          ]);
          sleep($retry_delay);
          continue;
        }

        throw new \RuntimeException('Unable to connect to OpenAI API. Please check your internet connection and try again.');
      }
      catch (RequestException $e) {
        $this->logger->error('OpenAI request failed: @msg', ['@msg' => $e->getMessage()]);

        // For other request exceptions, don't retry.
        throw new \RuntimeException('OpenAI API request failed: ' . $e->getMessage());
      }
    }

    // If we get here, all retries failed.
    throw new \RuntimeException('OpenAI API request failed after ' . $max_retries . ' attempts.');
  }

  /**
   * Test the connection to OpenAI API.
   *
   * @return array
   *   The test result.
   */
  public function testConnection(): array {
    $config = $this->configFactory->get('social_summaries.settings');
    $keyId = $config->get('openai_api_key_id');

    $result = [
      'success' => FALSE,
      'message' => '',
      'details' => [],
    ];

    try {
      // Check if key is configured.
      if (empty($keyId)) {
        $result['message'] = 'OpenAI API key not configured.';
        return $result;
      }

      // Check if key exists.
      $key = $this->keyRepository->getKey($keyId);
      if (!$key) {
        $result['message'] = 'Selected OpenAI API key not found.';
        return $result;
      }

      // Check if key has a value.
      $apiKey = $key->getKeyValue();
      if (empty($apiKey)) {
        $result['message'] = 'OpenAI API key is empty.';
        return $result;
      }

      // Test with a simple request.
      $test_messages = [
        ['role' => 'user', 'content' => 'Hello, this is a test message. Please respond with "Test successful".'],
      ];

      $response = $this->generate($test_messages, ['max_tokens' => 10]);
      if (!empty($response['text'])) {
        $result['success'] = TRUE;
        $result['message'] = 'Connection test successful.';
        $result['details'] = [
          'model' => $response['model'],
          'tokens_used' => $response['usage']['tokens_in'] + $response['usage']['tokens_out'],
        ];
      }
      else {
        $result['message'] = 'Connection test failed: Empty response received.';
      }
    }
    catch (\Exception $e) {
      $result['message'] = 'Connection test failed: ' . $e->getMessage();
      $this->logger->error('OpenAI connection test failed: @error', ['@error' => $e->getMessage()]);
    }

    return $result;
  }

  /**
   * Get rate limit status from API response.
   *
   * @return array
   *   The rate limit status.
   */
  public function getRateLimitStatus(): array {
    // This would require storing the last response headers.
    // For now, return a placeholder structure.
    return [
      'requests_remaining' => 'unknown',
      'requests_reset' => 'unknown',
      'tokens_remaining' => 'unknown',
      'tokens_reset' => 'unknown',
    ];
  }

}
