<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Psr\Log\LoggerInterface;

/**
 * Rate limiting service for AI API calls.
 */
class RateLimiter {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a RateLimiter object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, Connection $database, LoggerInterface $logger) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->logger = $logger;
  }

  /**
   * Check if a call can be made based on rate limits.
   *
   * @return bool
   *   TRUE if a call can be made, FALSE otherwise.
   */
  public function canMakeCall(): bool {
    $config = $this->configFactory->get('social_summaries.settings');
    $max_calls_per_minute = $config->get('max_calls_per_minute') ?? 60;
    $max_calls_per_hour = $config->get('max_calls_per_hour') ?? 1000;

    $current_time = time();
    $minute_ago = $current_time - 60;
    $hour_ago = $current_time - 3600;

    // Check calls in the last minute.
    $minute_calls = $this->database->select('social_summaries_usage', 'u')
      ->condition('created', $minute_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    if ($minute_calls >= $max_calls_per_minute) {
      $this->logger->warning('Rate limit exceeded: @calls calls in the last minute (limit: @limit)', [
        '@calls' => $minute_calls,
        '@limit' => $max_calls_per_minute,
      ]);
      return FALSE;
    }

    // Check calls in the last hour.
    $hour_calls = $this->database->select('social_summaries_usage', 'u')
      ->condition('created', $hour_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    if ($hour_calls >= $max_calls_per_hour) {
      $this->logger->warning('Rate limit exceeded: @calls calls in the last hour (limit: @limit)', [
        '@calls' => $hour_calls,
        '@limit' => $max_calls_per_hour,
      ]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Get recommended delay before next call.
   *
   * @return float
   *   The recommended delay in seconds.
   */
  public function getRecommendedDelay(): float {
    $config = $this->configFactory->get('social_summaries.settings');
    $base_delay = $config->get('base_delay') ?? 1.0;
    $max_delay = $config->get('max_delay') ?? 10.0;

    $current_time = time();
    $minute_ago = $current_time - 60;

    // Count recent calls to determine delay.
    $recent_calls = $this->database->select('social_summaries_usage', 'u')
      ->condition('created', $minute_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Increase delay based on recent call frequency.
    $delay_multiplier = 1 + ($recent_calls * 0.1);
    $recommended_delay = min($base_delay * $delay_multiplier, $max_delay);

    return $recommended_delay;
  }

  /**
   * Wait for rate limit to reset.
   */
  public function waitForRateLimit(): void {
    $delay = $this->getRecommendedDelay();
    if ($delay > 0) {
      $this->logger->info('Rate limiting: waiting @delay seconds', ['@delay' => $delay]);
      sleep((int) ceil($delay));
    }
  }

  /**
   * Get rate limit statistics.
   *
   * @return array
   *   The rate limit statistics.
   */
  public function getRateLimitStats(): array {
    $current_time = time();
    $minute_ago = $current_time - 60;
    $hour_ago = $current_time - 3600;
    $day_ago = $current_time - 86400;

    $stats = [
      'calls_last_minute' => 0,
      'calls_last_hour' => 0,
      'calls_last_day' => 0,
      'total_cost_today' => 0.0,
    ];

    // Get call counts.
    $stats['calls_last_minute'] = $this->database->select('social_summaries_usage', 'u')
      ->condition('created', $minute_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    $stats['calls_last_hour'] = $this->database->select('social_summaries_usage', 'u')
      ->condition('created', $hour_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    $stats['calls_last_day'] = $this->database->select('social_summaries_usage', 'u')
      ->condition('created', $day_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Get total cost for today.
    $stats['total_cost_today'] = $this->database->select('social_summaries_usage', 'u')
      ->condition('created', $day_ago, '>=')
      ->fields('u', ['cost_usd'])
      ->execute()
      ->fetchCol();

    $stats['total_cost_today'] = array_sum($stats['total_cost_today']);

    return $stats;
  }

  /**
   * Check if we should slow down API calls.
   *
   * @return bool
   *   TRUE if we should slow down, FALSE otherwise.
   */
  public function shouldSlowDown(): bool {
    $config = $this->configFactory->get('social_summaries.settings');
    $max_calls_per_minute = $config->get('max_calls_per_minute') ?? 60;
    // 80% of limit.
    $slow_down_threshold = $max_calls_per_minute * 0.8;

    $current_time = time();
    $minute_ago = $current_time - 60;

    $recent_calls = $this->database->select('social_summaries_usage', 'u')
      ->condition('created', $minute_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    return $recent_calls >= $slow_down_threshold;
  }

}
