<?php

declare(strict_types=1);

namespace Drupal\social_summaries\Utility;

/**
 * Utility class for platform-specific content constraints and validation.
 */
class PlatformConstraints {

  /**
   * Platform constraint definitions.
   *
   * @var array
   */
  protected static array $constraints = [];

  /**
   * Get constraint rules for a specific platform.
   *
   * @param string $platform
   *   The platform machine name.
   *
   * @return array|null
   *   The constraint rules for the platform, or NULL if not found.
   */
  public static function getConstraints(string $platform): ?array {
    if (empty(self::$constraints)) {
      // Lazy-load from configuration.
      $config = \Drupal::config('social_summaries.settings');
      $all = (array) $config->get('constraints');
      self::$constraints = is_array($all) ? $all : [];
    }

    return self::$constraints[$platform] ?? NULL;
  }

  /**
   * Evaluate content against platform constraints.
   *
   * @param string $platform
   *   The platform machine name.
   * @param string $text
   *   The text content to evaluate.
   *
   * @return array
   *   An array containing evaluation results with keys:
   *   - count: The current count (characters, words, or sentences)
   *   - type: The constraint type (character, word, sentence)
   *   - within_recommended: TRUE if within recommended range
   *   - within_hard_limit: TRUE if within hard limit
   *   - warnings: Array of warning messages
   *   - status: 'good', 'warning', or 'error'
   */
  public static function evaluate(string $platform, string $text): array {
    $constraints = self::getConstraints($platform);
    if (!$constraints) {
      return [
        'count' => 0,
        'type' => 'unknown',
        'within_recommended' => TRUE,
        'within_hard_limit' => TRUE,
        'warnings' => [],
        'status' => 'good',
      ];
    }

    $text = trim($text);
    $count = 0;
    $warnings = [];
    $status = 'good';

    switch ($constraints['type']) {
      case 'character':
        $count = mb_strlen($text);

        break;

      case 'word':
        $count = str_word_count($text);
        break;

      case 'sentence':
        // Simple sentence counting - split on sentence-ending punctuation.
        $sentences = preg_split('/[.!?]+/', $text, -1, PREG_SPLIT_NO_EMPTY);
        $count = count(array_filter($sentences, function ($s) {
          return trim($s) !== '';
        }));
        break;
    }

    // Remove hard-limit errors; only recommended messaging.
    $within_hard_limit = TRUE;

    // Check minimum limits.
    // Remove minimum hard-limit warnings.
    // Check recommended ranges.
    $within_recommended = TRUE;
    if (isset($constraints['recommended_min']) && $count < $constraints['recommended_min']) {
      $within_recommended = FALSE;
      $warnings[] = sprintf(
        'Content is below the recommended minimum of %d %s.',
        $constraints['recommended_min'],
        $constraints['type'] . 's'
      );
      $status = 'warning';
    }
    if (isset($constraints['recommended_max']) && $count > $constraints['recommended_max']) {
      $within_recommended = FALSE;
      $warnings[] = sprintf(
        'Content exceeds the recommended maximum of %d %s.',
        $constraints['recommended_max'],
        $constraints['type'] . 's'
      );
      $status = 'warning';
    }

    return [
      'count' => $count,
      'type' => $constraints['type'],
      'within_recommended' => $within_recommended,
      'within_hard_limit' => $within_hard_limit,
      'warnings' => $warnings,
      'status' => $status,
    ];
  }

  /**
   * Get a human-readable description of constraints for a platform.
   *
   * @param string $platform
   *   The platform machine name.
   *
   * @return string
   *   A description of the constraints.
   */
  public static function getDescription(string $platform): string {
    $constraints = self::getConstraints($platform);
    if (!$constraints) {
      return 'No specific constraints defined.';
    }

    switch ($constraints['type']) {
      case 'character':
        if (isset($constraints['recommended_min']) && isset($constraints['recommended_max'])) {
          return sprintf(
            '%d-%d characters recommended, %s maximum',
            $constraints['recommended_min'],
            $constraints['recommended_max'],
            (string) ($constraints['max_chars'] ?? 'unlimited')
          );
        }
        if (isset($constraints['max_chars'])) {
          return sprintf('Maximum %d characters', $constraints['max_chars']);
        }
        break;

      case 'word':
        if (isset($constraints['min_words']) && isset($constraints['max_words'])) {
          return sprintf('%d-%d words recommended', $constraints['min_words'], $constraints['max_words']);
        }
        break;

      case 'sentence':
        if (isset($constraints['min_sentences']) && isset($constraints['max_sentences'])) {
          return sprintf('%d-%d sentences recommended', $constraints['min_sentences'], $constraints['max_sentences']);
        }
        break;
    }

    return 'Platform-specific constraints apply.';
  }

  /**
   * Get all available platforms.
   *
   * @return array
   *   Array of platform machine names.
   */
  public static function getAvailablePlatforms(): array {
    if (empty(self::$constraints)) {
      $config = \Drupal::config('social_summaries.settings');
      $all = (array) $config->get('constraints');
      self::$constraints = is_array($all) ? $all : [];
    }
    return array_keys(self::$constraints);
  }

}
