<?php

namespace Drupal\socialfeed\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Twitter (X) API settings.
 *
 * This form allows administrators to configure integration with Twitter/X API
 * to display tweets from a Twitter account on the Drupal site.
 *
 * @package Drupal\socialfeed\Form
 */
class TwitterSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'twitter_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('socialfeed.twitter.settings');

    $form['header'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning">' . $this->t('<h3>Twitter (X) API Setup</h3>
        <p>This module allows you to display tweets from a Twitter/X account on your Drupal site.</p>
        <p><strong>⚠️ Important:</strong> As of 2023, Twitter (now X) has significantly changed its API pricing and access model. The free tier is very limited, and most use cases now require a paid subscription starting at $100/month for Basic access. Review <a href="@pricing" target="_blank">X API pricing</a> before proceeding.</p>
        <h4>Requirements</h4>
        <ul>
          <li><strong>X Developer Account:</strong> You need an approved X Developer account.</li>
          <li><strong>X API Access:</strong> Depending on your needs, you may need a paid API plan (Basic, Pro, or Enterprise).</li>
          <li><strong>App with User Authentication:</strong> Your app must be configured with OAuth 1.0a User Context authentication.</li>
        </ul>
        <h4>Setup Steps</h4>
        <ol>
          <li><strong>Create Developer Account:</strong> Go to the <a href="@portal" target="_blank">X Developer Portal</a> and sign up for a developer account if you don\'t have one.</li>
          <li><strong>Create Project and App:</strong> In the Developer Portal, create a new Project, then create an App within that project.</li>
          <li><strong>Configure App Settings:</strong> In your app settings, enable "OAuth 1.0a" under User authentication settings.</li>
          <li><strong>Get API Keys:</strong> Navigate to your app\'s "Keys and tokens" tab to find:
            <ul>
              <li>API Key (Consumer Key)</li>
              <li>API Key Secret (Consumer Secret)</li>
            </ul>
          </li>
          <li><strong>Generate Access Tokens:</strong> In the same "Keys and tokens" tab, generate:
            <ul>
              <li>Access Token</li>
              <li>Access Token Secret</li>
            </ul>
            These tokens will have the permissions of the account that generated them.
          </li>
          <li><strong>Note API Tier Limitations:</strong> Be aware of rate limits and features available in your API tier.</li>
        </ol>
        <p>For detailed information, see the <a href="@docs" target="_blank">X API Documentation</a>.</p>
        <p><strong>Security Note:</strong> Keep your API keys and tokens secure. Never commit them to version control or expose them publicly.</p>', [
          '@portal' => 'https://developer.twitter.com/en/portal/dashboard',
          '@docs' => 'https://developer.twitter.com/en/docs/twitter-api',
          '@pricing' => 'https://developer.twitter.com/en/products/twitter-api',
        ]) . '</div>',
    ];
    $form['consumer_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key (Consumer Key)'),
      '#default_value' => $config->get('consumer_key'),
      '#description' => $this->t('Found in the <a href="@portal" target="_blank">X Developer Portal</a> → Your Project → Your App → Keys and tokens tab. Also called "Consumer Key" or "API Key".', [
        '@portal' => 'https://developer.twitter.com/en/portal/dashboard',
      ]),
      '#size' => 60,
      '#maxlength' => 100,
      '#required' => TRUE,
    ];
    $form['consumer_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key Secret (Consumer Secret)'),
      '#default_value' => $config->get('consumer_secret'),
      '#description' => $this->t('Found in the <a href="@portal" target="_blank">X Developer Portal</a> → Your Project → Your App → Keys and tokens tab. Also called "Consumer Secret" or "API Key Secret". <strong>Keep this secret and never commit it to version control.</strong>', [
        '@portal' => 'https://developer.twitter.com/en/portal/dashboard',
      ]),
      '#size' => 60,
      '#maxlength' => 100,
      '#required' => TRUE,
    ];
    $form['access_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access Token'),
      '#default_value' => $config->get('access_token'),
      '#description' => $this->t('Generated in the <a href="@portal" target="_blank">X Developer Portal</a> → Your Project → Your App → Keys and tokens tab. Click "Generate" under "Access Token and Secret". This token represents the Twitter account that will be used to fetch tweets.', [
        '@portal' => 'https://developer.twitter.com/en/portal/dashboard',
      ]),
      '#size' => 60,
      '#maxlength' => 100,
      '#required' => TRUE,
    ];
    $form['access_token_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access Token Secret'),
      '#default_value' => $config->get('access_token_secret'),
      '#description' => $this->t('Generated alongside the Access Token in the <a href="@portal" target="_blank">X Developer Portal</a>. <strong>Keep this secret and never commit it to version control.</strong>', [
        '@portal' => 'https://developer.twitter.com/en/portal/dashboard',
      ]),
      '#size' => 60,
      '#maxlength' => 100,
      '#required' => TRUE,
    ];
    $form['tweets_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of Tweets to Display'),
      '#default_value' => $config->get('tweets_count'),
      '#description' => $this->t('The number of tweets to fetch and display from the timeline. Note: API rate limits may apply depending on your API tier.'),
      '#size' => 60,
      '#maxlength' => 100,
      '#min' => 1,
    ];
    // @todo Move these to the block form; Update theme implementation.
    $form['hashtag'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display Hashtags'),
      '#description' => $this->t('Show hashtags from tweet content.'),
      '#default_value' => $config->get('hashtag'),
    ];
    $form['time_stamp'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Tweet Date/Time'),
      '#description' => $this->t('Display when the tweet was posted.'),
      '#default_value' => $config->get('time_stamp'),
    ];
    $form['time_ago'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Twitter-Style Relative Dates'),
      '#description' => $this->t('Display dates in Twitter style (e.g., "2h ago", "3d ago") instead of absolute dates.'),
      '#default_value' => $config->get('time_ago'),
      '#states' => [
        'visible' => [
          ':input[name="time_stamp"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['time_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Date/Time Format'),
      '#default_value' => $config->get('time_format'),
      '#description' => $this->t('PHP date format string (e.g., <code>Y-m-d H:i:s</code> for "2024-03-15 14:30:00" or <code>F j, Y</code> for "March 15, 2024"). Only used when Twitter-style relative dates are disabled. See <a href="@datetime" target="_blank">PHP date format documentation</a> for all available options.', [
        '@datetime' => 'https://www.php.net/manual/en/datetime.format.php',
      ]),
      '#size' => 60,
      '#maxlength' => 100,
      '#states' => [
        'visible' => [
          ':input[name="time_stamp"]' => ['checked' => TRUE],
          ':input[name="time_ago"]' => ['checked' => FALSE],
        ],
      ],
    ];
    $form['trim_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Tweet Text Length'),
      '#default_value' => $config->get('trim_length'),
      '#description' => $this->t('Maximum number of characters to display from tweet text (tweets can be up to 280 characters). Leave empty or set to 0 to show full text.'),
      '#size' => 60,
      '#maxlength' => 280,
      '#min' => 0,
      '#max' => 280,
    ];
    $form['teaser_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('"Read More" Link Text'),
      '#default_value' => $config->get('teaser_text'),
      '#description' => $this->t('Text to display for the "read more" link when tweet text is trimmed. For example, "Read more" or "View full tweet".'),
      '#size' => 60,
      '#maxlength' => 60,
    ];

    $form['style_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Display Style Settings'),
      '#open' => TRUE,
    ];
    $form['style_settings']['use_twitter_style'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Default Twitter UI Style'),
      '#description' => $this->t('Check to apply a Twitter-like layout and styles. Uncheck to output minimal markup you can style yourself.'),
      '#default_value' => $config->get('use_twitter_style') ?? TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('socialfeed.twitter.settings');
    foreach ($form_state->getValues() as $key => $value) {
      $config->set($key, $value);
    }
    $config->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'socialfeed.twitter.settings',
    ];
  }

}
