<?php

namespace Drupal\socialfeed\Services;

use Abraham\TwitterOAuth\TwitterOAuth;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * The collector class for Twitter.
 *
 * @package Drupal\socialfeed\Services
 */
class TwitterPostCollector {

  /**
   * Twitter's consumer key.
   *
   * @var string
   */
  protected string $consumerKey;

  /**
   * Twitter's consumer secret.
   *
   * @var string
   */
  protected string $consumerSecret;

  /**
   * Twitter's access token.
   *
   * @var string
   */
  protected string $accessToken;

  /**
   * Twitter's access token secret.
   *
   * @var string
   */
  protected string $accessTokenSecret;

  /**
   * Twitter's OAuth client.
   *
   * @var \Abraham\TwitterOAuth\TwitterOAuth|null
   */
  protected ?TwitterOAuth $twitter;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * TwitterPostCollector constructor.
   *
   * @param string $consumerKey
   *   Twitter's consumer key.
   * @param string $consumerSecret
   *   Twitter's consumer secret.
   * @param string $accessToken
   *   Twitter's access token.
   * @param string $accessTokenSecret
   *   Twitter's access token secret.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Abraham\TwitterOAuth\TwitterOAuth|null $twitter
   *   Twitter's OAuth Client.
   */
  public function __construct(
    string $consumerKey,
    string $consumerSecret,
    string $accessToken,
    string $accessTokenSecret,
    LoggerChannelFactoryInterface $logger_factory,
    ?TwitterOAuth $twitter = NULL,
  ) {
    $this->consumerKey = $consumerKey;
    $this->consumerSecret = $consumerSecret;
    $this->accessToken = $accessToken;
    $this->accessTokenSecret = $accessTokenSecret;
    $this->logger = $logger_factory->get('socialfeed');
    $this->twitter = $twitter;
    $this->setTwitterClient();
  }

  /**
   * Sets the Twitter client.
   */
  public function setTwitterClient(): void {
    if (NULL === $this->twitter) {
      $this->twitter = new TwitterOAuth(
        $this->consumerKey,
        $this->consumerSecret,
        $this->accessToken,
        $this->accessTokenSecret
      );
    }
  }

  /**
   * Retrieves Tweets from the given accounts home page.
   *
   * @param int $count
   *   The number of posts to return.
   *
   * @return array|object
   *   An array or object of posts.
   */
  public function getPosts(int $count): array|object {
    try {
      $result = $this->twitter->get('statuses/user_timeline', [
        'count' => $count,
        'tweet_mode' => 'extended',
      ]);
      return $result ?? [];
    }
    catch (\Exception $e) {
      $this->logger->error('Twitter API error: @error', ['@error' => $e->getMessage()]);
      return [];
    }
  }

}
