<?php

namespace Drupal\socialfeed\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * The factory collector class for Twitter.
 *
 * @package Drupal\socialfeed\Services
 */
class TwitterPostCollectorFactory {

  /**
   * The default consumer key.
   *
   * @var string
   */
  protected string $defaultConsumerKey;

  /**
   * The default consumer secret.
   *
   * @var string
   */
  protected string $defaultConsumerSecret;

  /**
   * The default access token.
   *
   * @var string
   */
  protected string $defaultAccessToken;

  /**
   * The default access token secret.
   *
   * @var string
   */
  protected string $defaultAccessTokenSecret;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * TwitterPostCollectorFactory constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $config = $config_factory->get('socialfeed.twitter.settings');
    $this->defaultConsumerKey = $config->get('consumer_key') ?? '';
    $this->defaultConsumerSecret = $config->get('consumer_secret') ?? '';
    $this->defaultAccessToken = $config->get('access_token') ?? '';
    $this->defaultAccessTokenSecret = $config->get('access_token_secret') ?? '';
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Creates a pre-configured instance.
   *
   * @param string $consumerKey
   *   The consumer key.
   * @param string $consumerSecret
   *   The consumer secret.
   * @param string $accessToken
   *   The access token.
   * @param string $accessTokenSecret
   *   The access token secret.
   *
   * @return \Drupal\socialfeed\Services\TwitterPostCollector
   *   A fully configured instance from TwitterPostCollector.
   *
   * @throws \Exception
   *   If the instance cannot be created, such as if the ID is invalid.
   */
  public function createInstance(string $consumerKey, string $consumerSecret, string $accessToken, string $accessTokenSecret): TwitterPostCollector {
    return new TwitterPostCollector(
      $consumerKey ?: $this->defaultConsumerKey,
      $consumerSecret ?: $this->defaultConsumerSecret,
      $accessToken ?: $this->defaultAccessToken,
      $accessTokenSecret ?: $this->defaultAccessTokenSecret,
      $this->loggerFactory
    );
  }

}
