<?php

namespace Drupal\Tests\socialfeed\Unit\Services;

use Abraham\TwitterOAuth\TwitterOAuth;
use Drupal\socialfeed\Services\TwitterPostCollector;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * Tests for TwitterPostCollector.
 *
 * @group socialfeed
 * @coversDefaultClass \Drupal\socialfeed\Services\TwitterPostCollector
 */
class TwitterPostCollectorTest extends UnitTestCase {

  /**
   * The Twitter OAuth client mock.
   *
   * @var \Abraham\TwitterOAuth\TwitterOAuth|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $twitterClient;

  /**
   * The logger factory mock.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The logger mock.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The Twitter post collector.
   *
   * @var \Drupal\socialfeed\Services\TwitterPostCollector
   */
  protected $postCollector;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->twitterClient = $this->createMock(TwitterOAuth::class);

    $this->loggerFactory->method('get')
      ->with('socialfeed')
      ->willReturn($this->logger);

    $this->postCollector = new TwitterPostCollector(
      'test_consumer_key',
      'test_consumer_secret',
      'test_access_token',
      'test_access_token_secret',
      $this->loggerFactory,
      $this->twitterClient
    );
  }

  /**
   * Tests getPosts method with successful response.
   *
   * @covers ::getPosts
   */
  public function testGetPostsSuccess(): void {
    $mock_tweets = [
      (object) [
        'id_str' => '123',
        'full_text' => 'Test tweet 1',
        'user' => (object) ['screen_name' => 'testuser'],
      ],
      (object) [
        'id_str' => '124',
        'full_text' => 'Test tweet 2',
        'user' => (object) ['screen_name' => 'testuser'],
      ],
    ];

    $this->twitterClient->expects($this->once())
      ->method('get')
      ->with('statuses/user_timeline', [
        'count' => 10,
        'tweet_mode' => 'extended',
      ])
      ->willReturn($mock_tweets);

    $posts = $this->postCollector->getPosts(10);

    $this->assertCount(2, $posts);
    $this->assertEquals('123', $posts[0]->id_str);
    $this->assertEquals('Test tweet 1', $posts[0]->full_text);
  }

  /**
   * Tests getPosts method with API exception.
   *
   * @covers ::getPosts
   */
  public function testGetPostsWithException(): void {
    $this->twitterClient->expects($this->once())
      ->method('get')
      ->willThrowException(new \Exception('Twitter API error'));

    $this->logger->expects($this->once())
      ->method('error')
      ->with('Twitter API error: @error', ['@error' => 'Twitter API error']);

    $posts = $this->postCollector->getPosts(10);

    $this->assertEmpty($posts);
  }

  /**
   * Tests getPosts method returns empty array on NULL response.
   *
   * @covers ::getPosts
   */
  public function testGetPostsWithNullResponse(): void {
    $this->twitterClient->expects($this->once())
      ->method('get')
      ->willReturn(NULL);

    $posts = $this->postCollector->getPosts(10);

    $this->assertEmpty($posts);
  }

  /**
   * Tests getPosts with different count parameter.
   *
   * @covers ::getPosts
   */
  public function testGetPostsWithDifferentCount(): void {
    $this->twitterClient->expects($this->once())
      ->method('get')
      ->with('statuses/user_timeline', [
        'count' => 25,
        'tweet_mode' => 'extended',
      ])
      ->willReturn([]);

    $posts = $this->postCollector->getPosts(25);

    $this->assertEmpty($posts);
  }

  /**
   * Tests setTwitterClient creates client when NULL.
   *
   * @covers ::setTwitterClient
   * @covers ::__construct
   */
  public function testSetTwitterClientCreatesClient(): void {
    // Create instance without passing TwitterOAuth client.
    $collector = new TwitterPostCollector(
      'key',
      'secret',
      'token',
      'token_secret',
      $this->loggerFactory
    );

    // If no exception, client was created successfully.
    $this->assertInstanceOf(TwitterPostCollector::class, $collector);
  }

  /**
   * Tests that constructor accepts provided Twitter client.
   *
   * @covers ::__construct
   * @covers ::setTwitterClient
   */
  public function testConstructorAcceptsProvidedClient(): void {
    $customClient = $this->createMock(TwitterOAuth::class);

    $collector = new TwitterPostCollector(
      'key',
      'secret',
      'token',
      'token_secret',
      $this->loggerFactory,
      $customClient
    );

    $this->assertInstanceOf(TwitterPostCollector::class, $collector);
  }

}
