<?php

namespace Drupal\soundcite\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Plugin\Field\FieldFormatter\FileFormatterBase;

/**
 * Plugin implementation of the 'soundcite_audio' formatter.
 *
 * @FieldFormatter(
 *   id = "soundcite_audio",
 *   label = @Translation("Soundcite Audio Player"),
 *   field_types = {
 *     "file"
 *   }
 * )
 */
class SoundciteAudioFormatter extends FileFormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'start_time' => '',
      'end_time' => '',
      'plays' => 1,
      'link_text' => '',
      'use_description_as_link_text' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    $form['start_time'] = [
      '#title' => $this->t('Start time (seconds)'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('start_time'),
      '#description' => $this->t('Start playback at this time in seconds. Leave empty to start from beginning.'),
      '#size' => 10,
    ];

    $form['end_time'] = [
      '#title' => $this->t('End time (seconds)'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('end_time'),
      '#description' => $this->t('End playback at this time in seconds. Leave empty to play to end.'),
      '#size' => 10,
    ];

    $form['plays'] = [
      '#title' => $this->t('Number of plays'),
      '#type' => 'number',
      '#default_value' => $this->getSetting('plays'),
      '#description' => $this->t('Number of times the audio should play when clicked.'),
      '#min' => 1,
      '#max' => 10,
    ];

    $form['use_description_as_link_text'] = [
      '#title' => $this->t('Use description as link text'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('use_description_as_link_text'),
      '#description' => $this->t('Replace the file name by its description when available'),
    ];

    $form['link_text'] = [
      '#title' => $this->t('Custom link text'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('link_text'),
      '#description' => $this->t('Custom text to display as the clickable link. Leave empty to use filename. <br/>This will be overridden if "Use description as link text" is enabled and a description exists.'),
      '#size' => 60,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if (!empty($this->getSetting('start_time'))) {
      $summary[] = $this->t('Start: @time seconds', [
        '@time' => $this->getSetting('start_time'),
      ]);
    }

    if (!empty($this->getSetting('end_time'))) {
      $summary[] = $this->t('End: @time seconds', [
        '@time' => $this->getSetting('end_time'),
      ]);
    }

    $summary[] = $this->t('Plays: @plays', [
      '@plays' => $this->getSetting('plays'),
    ]);

    if ($this->getSetting('use_description_as_link_text')) {
      $summary[] = $this->t('Use description as link text');
    }

    if (!empty($this->getSetting('link_text'))) {
      $summary[] = $this->t('Custom link text: @text', [
        '@text' => $this->getSetting('link_text'),
      ]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $files = $this->getEntitiesToView($items, $langcode);

    // Early opt-out if the field is empty.
    if (empty($files)) {
      return $elements;
    }

    // Collect cache tags to be added for each item in the field.
    foreach ($files as $delta => $file) {
      $elements[$delta] = [
        '#theme' => 'soundcite_audio_formatter',
        '#file' => $file,
        '#start_time' => $this->getSetting('start_time'),
        '#end_time' => $this->getSetting('end_time'),
        '#plays' => $this->getSetting('plays'),
        '#link_text' => $this->getSetting('link_text'),
        '#use_description_as_link_text' => $this->getSetting('use_description_as_link_text'),
        '#attributes' => ['class' => ['soundcite']],
        '#cache' => [
          'tags' => $file->getCacheTags(),
        ],
      ];
    }

    // Attach Soundcite library.
    $elements['#attached']['library'][] = 'soundcite/soundcite';

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable($field_definition) {
    // Check if this is a file field
    if ($field_definition->getType() !== 'file') {
      return FALSE;
    }

    // Get field settings to check allowed extensions
    $settings = $field_definition->getSettings();
    if (!empty($settings['file_extensions'])) {
      $extensions = explode(' ', strtolower($settings['file_extensions']));
      $audio_extensions = ['mp3', 'wav', 'ogg', 'm4a', 'aac', 'flac'];

      // Check if any audio extensions are allowed
      if (array_intersect($extensions, $audio_extensions)) {
        return TRUE;
      }
    }

    return FALSE;
  }

}
