import os
import yaml
from yaml.representer import SafeRepresenter

# Custom representer to ensure strings in 'examples' are quoted with single quotes
class SingleQuoted(str):
    pass

def single_quoted_presenter(dumper, data):
    """Always represent strings in 'examples' with single quotes."""
    return dumper.represent_scalar('tag:yaml.org,2002:str', data, style="'")

# Register the custom representer for the SingleQuoted type
yaml.add_representer(SingleQuoted, single_quoted_presenter)

def load_yaml_file(file_path):
    """Load the YAML content from a file."""
    with open(file_path, 'r') as yaml_file:
        return yaml.safe_load(yaml_file)

def find_pattern_fields(content):
    """Find fields of type 'pattern' in the YAML content."""
    pattern_fields = {}
    for top_key, top_value in content.items():
        fields = top_value.get('fields')
        if fields:
            for key, field in fields.items():
                if field.get('type') == 'pattern':
                    pattern_fields[key] = field
    return pattern_fields

def gather_fields(content):
    """Gather pattern fields, other fields, and settings fields from a YAML content."""
    pattern_fields = find_pattern_fields(content)
    other_fields = {}
    setting_fields = {}

    for top_value in content.values():
        fields = top_value.get('fields', {})
        settings = top_value.get('settings', {})

        for key, field in fields.items():
            if field.get('type') != 'pattern':
                other_fields[key] = field

        for setting_key, setting_value in settings.items():
            setting_fields[setting_key] = setting_value

    return pattern_fields, other_fields, setting_fields

def determine_group(use_value):
    """Determine the group based on the 'use' value."""
    if use_value.startswith('@atoms'):
        return 'Atoms'
    elif use_value.startswith('@molecules'):
        return 'Molecules'
    elif use_value.startswith('@organisms'):
        return 'Organisms'
    elif use_value.startswith('@templates'):
        return 'Templates'
    elif use_value.startswith('@pages'):
        return 'Pages'
    else:
        return 'default'

def format_yaml(yaml_str):
    """Format YAML string with specific structure and spacing."""
    lines = yaml_str.splitlines()
    result = ["'$schema': 'https://git.drupalcode.org/project/drupal/-/raw/HEAD/core/assets/schemas/v1/metadata.schema.json'", ""]
    properties_depth = 0
    previous_indent = 0

    for i, line in enumerate(lines):
        stripped_line = line.strip()
        current_indent = len(line) - len(stripped_line)
        
        if stripped_line == "properties:":
            result.append("")  # Add a blank line before nested properties
            result.append(line)
            properties_depth += 1
            previous_indent = current_indent
            continue
        
        if stripped_line == "properties: {}":
            result.append(line)
            result.append("")  # Add a blank line after the empty properties object
            continue
        
        if stripped_line.startswith('-'):
            result.append("  " + line)
            continue

        if stripped_line and stripped_line != "properties:" and properties_depth > 0:
            if previous_indent > current_indent:
                result.append("")  # Add a blank line before nested properties
            previous_indent = current_indent
        result.append(line)

        # Adjust depth tracking
        if current_indent < previous_indent:
            while properties_depth > 0 and current_indent <= previous_indent:
                properties_depth -= 1
                if properties_depth > 0:
                    result.append("")  # Add a blank line before the next sibling properties
                previous_indent = current_indent
        
        if current_indent == 0:
            result.append("")

    return "\n".join(result).rstrip() + "\n"

def resolve_type(field_type):
    """Resolve the field type to its corresponding Python type."""
    string_types = {
        'text', 'textfield', 'strings', 'link', 'details',
        'password', 'checkbox', 'attributes', 'radios', 
        'token', 'url', 'date', 'colorwidget', 'select'
    }
    
    return 'string' if field_type in string_types else field_type

def format_default_value(default):
    """Ensure the default value is in list format and handle booleans and strings on a new line."""
    if isinstance(default, bool) or isinstance(default, str):
        return [default]  # Wrap booleans and strings in a list
    return default

def format_examples(examples):
    """Format examples to ensure they are quoted properly."""
    if not isinstance(examples, list):
        examples = [examples]

    # Wrap each example with SingleQuoted to ensure single quotes in YAML
    return [SingleQuoted(example) for example in examples]

def generate_yaml(file_path):
    """Generate YAML content from the space.yml file."""
    content = load_yaml_file(file_path)
    
    # Extract the label from the first top-level key's value in the content
    top_key = list(content.keys())[0]  # The top key, e.g., "iconcard"
    label = content[top_key].get('label', 'Default Label')  # Get the label or use a default value

    # Determine the group based on the 'use' key's value
    use_value = content[top_key].get('use', '')
    group = determine_group(use_value)

    # Gather fields from the current file content
    pattern_fields, other_fields, setting_fields = gather_fields(content)

    yaml_data = {
        "name": label,  # Dynamically set the name to the label from the YAML file
        "status": "experimental",
        "group": group,  # Set the group based on the 'use' value
        "props": {"type": "object"},
    }

    properties = {}
    for key, field in other_fields.items():
        title = field.get('label', key)
        description = field.get('description', title)
        examples = field.get('preview', [])
        default = field.get('default', field.get('default_value'))  # Add support for default_value

        # Handle 'attributes' key specifically
        key = 'custom_attributes' if key == 'attributes' else key

        # Add default value if present and convert it to list
        properties[key] = {
            "type": resolve_type(field.get('type')),
            "title": title,
            "description": description,
        }

        # Format examples if present
        if examples:
            properties[key]["examples"] = format_examples(examples)
        
        # Add 'default' if it's provided and handle formatting for booleans and strings
        if default is not None:
            properties[key]["default"] = format_default_value(default)

    for key, field in setting_fields.items():
        title = field.get('label', key)
        description = field.get('description', title)
        examples = field.get('preview', [])
        default = field.get('default', field.get('default_value'))

        if field.get('type') == 'select' or field.get('type') == 'radios' or field.get('type') == 'colorwidget':
            properties[key] = {
                "type": resolve_type(field.get('type')),
                "title": title,
                "description": description,
                "enum": list(field.get('options', {}).keys()),
                "default": format_default_value(default),
                "examples": format_examples(list(field.get('options', {}).keys()))
            }
        else:
            key = 'custom_attributes' if key == 'attributes' else key
            properties[key] = {
                "type": resolve_type(field.get('type')),
                "title": title,
                "description": description,
            }

            # Format examples if present
            if examples:
                properties[key]["examples"] = format_examples(examples)

            # Add 'default' if it's provided and handle formatting for booleans and strings
            if default is not None:
                properties[key]["default"] = format_default_value(default)

    if properties:
        yaml_data["props"]["properties"] = properties

    # Initialize slots
    slots = {
        key: {
            "title": field.get('label', key),
            "description": field.get('description', field.get('label', key)),
            "examples": [f"Slot - Add {key} here"]
        }
        for key, field in pattern_fields.items()
    }

    if slots:
        yaml_data["slots"] = slots

    # Dump YAML data with the default PyYAML dumper
    yaml_output = yaml.dump(yaml_data, sort_keys=False, default_flow_style=False, indent=2)
    return format_yaml(yaml_output)

def write_yaml_to_file(file_path, yaml_content):
    """Write the YAML content to a file."""
    with open(file_path, 'w') as file:
        file.write(yaml_content)

def main():
    directory_path = '/home/akshay/Documents/SPACE-SDC/space_ds_21Oct/space_ds/components/02-molecules/space-logo-grid'

    # Traverse the directory and process each space.yml file
    for root, dirs, files in os.walk(directory_path):
        for file in files:
            if file.endswith('.space.yml'):
                # Generate the output file path using the same name as the space.yml file
                output_file_path = os.path.join(root, file.replace('.space.yml', '.component.yml'))
                
                # Generate YAML for the current space.yml file
                yaml_content = generate_yaml(os.path.join(root, file))
                
                # Write the generated YAML to the specified file
                write_yaml_to_file(output_file_path, yaml_content)
                print(f"Generated {output_file_path}")

if __name__ == "__main__":
    main()
