<?php

namespace Drupal\spacer_block\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a 'Spacer Block'.
 *
 * @Block(
 * id = "custom_spacer_block",
 * admin_label = @Translation("Spacer block"),
 * category = @Translation("Spacer block"),
 * )
 */
class SpacerBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function build() {
    $config = $this->getConfiguration();
    // Default to spacer-128.
    $spacer_class = $config['spacer_size'] ?? 'spacer-128';

    // Add the class to the block's attributes.
    $build = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [],
      ],
      '#attached' => [
        'library' => [
          'spacer_block/spacer-styles',
        ],
      ],
      '#html_tag' => 'div',
    ];

    // Add spacer class based on configuration.
    if ($config['use_spacer_custom_size'] && !empty($config['spacer_custom_size'])) {
      $build['#attributes']['style'] = ['height: ' . $config['spacer_custom_size']];
    }
    else {
      $build['#attributes']['class'][] = $spacer_class;
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    // Define the available spacer sizes based on your CSS variables.
    $spacer_options = [
      'spacer-236' => 'Extra Large (236)',
      'spacer-200' => 'Very Large (200)',
      'spacer-156' => 'Large (156)',
      'spacer-128' => 'Medium (128) - Default',
      'spacer-96' => 'Small (96)',
      'spacer-72' => 'Extra Small (72)',
      'spacer-48' => 'Minimal (48)',
      'spacer-36' => 'Micro (36)',
      'spacer-30' => 'Tiny (30)',
      'spacer-24' => 'Smallest (24)',
      'spacer-20' => 'Microscopic (20)',
    ];

    $form['spacer_size'] = [
      '#type' => 'select',
      '#title' => $this->t('Spacer Size'),
      '#default_value' => $config['spacer_size'] ?? 'spacer-128',
      '#options' => $spacer_options,
      '#description' => $this->t('Select the desired size for the spacer.'),
      '#required' => TRUE,
    ];

    $form['use_spacer_custom_size'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Spacer Custom Size'),
      '#default_value' => $config['use_spacer_custom_size'] ?? FALSE,
      '#description' => $this->t('Enable this to define a custom spacer size instead of using predefined options.'),
    ];

    $form['spacer_custom_size'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Spacer Custom Size'),
      '#default_value' => $config['spacer_custom_size'] ?? '',
      '#description' => $this->t('Define a custom size using CSS units (e.g., "50px", "5rem"). This will override the selected spacer size.'),
      '#placeholder' => $this->t('50px'),
      '#states' => [
        'visible' => [
          ':input[name="settings[use_spacer_custom_size]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->configuration['spacer_size'] = $form_state->getValue('spacer_size');
    $this->configuration['use_spacer_custom_size'] = $form_state->getValue('use_spacer_custom_size');
    $this->configuration['spacer_custom_size'] = $form_state->getValue('spacer_custom_size');
  }

}
