<?php

namespace Drupal\spammaster;

use GuzzleHttp\Client;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Service for communicating with the SpamMaster API server.
 */
class SpamMasterApiService {

  /**
   * The HTTP client for making requests.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs the SpamMasterApiService object.
   *
   * @param \GuzzleHttp\Client $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    Client $httpClient,
    ConfigFactoryInterface $configFactory
  ) {
    $this->httpClient = $httpClient;
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('config.factory')
    );
  }

  /**
   * Posts data to the SpamMaster API server.
   *
   * @param string $endpoint
   *   The API URL endpoint.
   * @param array $params
   *   The post data.
   * @param bool $expectResponse
   *   Whether to return the API response (TRUE) or not (FALSE).
   *
   * @return mixed|null
   *   The decoded response if $expectResponse is TRUE, otherwise NULL.
   */
  public function postScanData(string $endpoint, array $params, bool $expectResponse = FALSE) {
    try {
      $response = $this->httpClient->post($endpoint, [
        'form_params' => $params,
      ]);
      if ($expectResponse) {
        $body = (string) $response->getBody();
        // Try to decode JSON, fallback to raw string.
        $decoded = Json::decode($body, TRUE);
        return $decoded ?? $body;
      }
    }
    catch (\Exception $e) {
      // Optionally rethrow or return null.
    }
    return NULL;
  }

  /**
   * Posts license data to the SpamMaster License API endpoint.
   *
   * @param string $endpoint
   *   The API endpoint URL.
   * @param array $params
   *   The license data to post.
   *
   * @return array|string|null
   *   The decoded license API response, raw body, or null on failure.
   */
  public function postLicenseData(string $endpoint, array $params) {
    try {
      $response = $this->httpClient->post($endpoint, [
        'form_params' => $params,
      ]);
      $body = (string) $response->getBody();
      $decoded = Json::decode($body, TRUE);
      return $decoded ?? $body;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

}
