<?php

namespace Drupal\spammaster;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use GuzzleHttp\ClientInterface;

/**
 * Class service.
 */
class SpamMasterUserService {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The form class.
   *
   * @var mixed
   */
  protected $spamuser;

  /**
   * {@inheritdoc}
   */
  public function __construct(AccountProxyInterface $currentUser, ClientInterface $http_client) {
    $this->currentUser = $currentUser;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('http_client')
    );
  }

  /**
   * Retrieves minimal user info, including Gravatar if available.
   *
   * @param mixed $spamuser
   *   Optional user id or object (for future use).
   *
   * @return string
   *   JSON encoded user info: ID, username, avatar (Gravatar if available).
   */
  public function spamMasterGetUser($spamuser = NULL) {
    $this->spamuser = $spamuser;

    $spam_user_id = ($this->currentUser && method_exists($this->currentUser, 'id')) ? $this->currentUser->id() : 0;

    if ($spam_user_id) {
      $spam_username = ($this->currentUser && method_exists($this->currentUser, 'getDisplayName')) ? $this->currentUser->getDisplayName() : '';
      $email = ($this->currentUser && method_exists($this->currentUser, 'getEmail')) ? $this->currentUser->getEmail() : '';
      $avatar = '';
      if (!empty($email)) {
        $avatar = $this->getUserGravatarUrl($email);
      }
      $prespamuser = [
        'ID'       => $spam_user_id,
        'username' => $spam_username,
        'avatar'   => $avatar,
      ];
    }
    else {
      $prespamuser = [
        'ID'       => 'none',
        'username' => 'none',
        'avatar'   => '',
      ];
    }
    $spamuserdata = Json::encode($prespamuser);

    return $spamuserdata;
  }

  /**
   * Get Gravatar if exists for the given email, otherwise return empty string.
   *
   * @param string $email
   *   The email address.
   *
   * @return string
   *   Gravatar URL or empty string.
   */
  protected function getUserGravatarUrl($email) {
    $email = strtolower(trim($email));
    $hash = md5($email);
    $gravatar_url = "https://www.gravatar.com/avatar/$hash?s=120&d=404";

    // Use Guzzle to make a HEAD request (no image download).
    try {
      $response = $this->httpClient->request('HEAD', $gravatar_url, ['http_errors' => FALSE]);
      if ($response->getStatusCode() == 200) {
        return $gravatar_url;
      }
    }
    catch (\Exception $e) {
      // Ignore errors, fallback to empty avatar.
    }
    return '';
  }

}
