<?php

namespace Drupal\spammaster\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class controller.
 */
class SpamMasterLogForm extends ConfigFormBase {

  /**
   * The database connection object.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Symfony\Component\HttpFoundation\RequestStack definition.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Client.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * {@inheritdoc}
   */
  public function __construct(Connection $connection, MessengerInterface $messenger, StateInterface $state, RequestStack $requestStack, TimeInterface $time) {
    $this->connection = $connection;
    $this->messenger = $messenger;
    $this->state = $state;
    $this->requestStack = $requestStack;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('database'),
      $container->get('messenger'),
      $container->get('state'),
      $container->get('request_stack'),
      $container->get('datetime.time'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'spammaster_settings_log_form';
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterStatisticsOnline($form, &$form_state) {
    $spam_get_online = $form_state->getValue('statistics_header')['buttons']['online'];
    if (!empty($spam_get_online)) {
      $online_statistics_url = 'https://www.spammaster.org/websites-stats/';
      $redirect_response = new TrustedRedirectResponse($online_statistics_url);
      $form_state->setResponse($redirect_response, 302);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterDeleteKey($form, &$form_state) {
    $spam_form_key_delete = $form_state->getValue('statistics_header')['table_key'];
    $spammaster_key_date = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    foreach ($spam_form_key_delete as $spam_key_delete) {
      if (!empty($spam_key_delete)) {
        $this->connection->delete('spammaster_keys')
          ->condition('spamkey', 'spammaster', '=')
          ->condition('id', $spam_key_delete, '=')
          ->execute();
        $this->messenger->addMessage($this->t('Saved Spam Master Log deletion.'));
        $this->connection->insert('spammaster_keys')->fields([
          'date' => $spammaster_key_date,
          'spamkey' => 'spammaster-log',
          'spamvalue' => 'Spam Master: log deletion, Id: ' . $spam_key_delete,
        ])->execute();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterDeleteKeysAll() {
    $this->state->set('spammaster.total_block_count', '0');
    $this->connection->delete('spammaster_keys')
      ->condition('spamkey', 'exempt-needle', '!=')
      ->condition('spamkey', 'exempt-needle-straw', '!=')
      ->condition('spamkey', 'exempt-needle-sig-hide', '!=')
      ->condition('spamkey', 'exempt-needle-sig-show', '!=')
      ->condition('spamkey', 'exempt-key', '!=')
      ->condition('spamkey', 'exempt-value', '!=')
      ->condition('spamkey', 'white-transient-haf', '!=')
      ->condition('spamkey', 'white-transient-form', '!=')
      ->execute();
    $this->messenger->addMessage($this->t('Saved Spam Master Statistics & Logs full deletion.'));
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    // Form constructor.
    $form = parent::buildForm($form, $form_state);

    $form['statistics_header'] = [
      '#type' => 'details',
      '#title' => $this->t('<h3>Statistics & Logging</h3>'),
      '#tree' => TRUE,
      '#open' => TRUE,
      '#attached' => [
        'library' => [
          'spammaster/spammaster-styles',
        ],
      ],
    ];

    $spam_master_type = $this->state->get('spammaster.type');
    $spam_master_attached = $this->state->get('spammaster.spam_master_attached');
    $spammaster_protection_total_number = $this->state->get('spammaster.license_protection');
    if ('FULL' === $spam_master_type) {

      $spammaster_total_block_count = $this->state->get('spammaster.total_block_count');
      if (empty($spammaster_total_block_count)) {
        $spammaster_total_block_count = '0';
      }
      // Insert statistics table inside tree.
      $form['statistics_header']['total_block_count'] = [
        '#markup' => $this->t('<h2>Total Firewall Blocks: <span class="spam-master-admin-red spam-master-top-admin-shadow-offline">@total_blocks</span></h2>', ['@total_blocks' => $spammaster_total_block_count]),
        '#attributes' => [
          'class' => [
            'spam-master-admin-red',
            'spam-master-top-admin-shadow-offline',
          ],
        ],
      ];

      $form['statistics_header']['statistics'] = [
        '#type' => 'table',
        '#header' => [
          'logs' => $this->t('All Logs'),
          'system_logs' => $this->t('System Logs'),
          'firewall' => $this->t('Firewall'),
          'honeypot' => $this->t('Honeypot'),
        ],
      ];

      // Generate all logs stats total.
      $spammaster_all_logs = $this->connection->select('spammaster_keys', 'u');
      $spammaster_all_logs->fields('u', ['spamkey']);
      $spammaster_all_logs->condition('u.spamkey', 'exempt-needle', '!=');
      $spammaster_all_logs->condition('u.spamkey', 'exempt-needle-straw', '!=');
      $spammaster_all_logs->condition('u.spamkey', 'exempt-needle-sig-hide', '!=');
      $spammaster_all_logs->condition('u.spamkey', 'exempt-needle-sig-show', '!=');
      $spammaster_all_logs->condition('u.spamkey', 'exempt-key', '!=');
      $spammaster_all_logs->condition('u.spamkey', 'exempt-value', '!=');
      $spammaster_all_logs->condition('u.spamkey', 'white-transient-haf', '!=');
      $spammaster_all_logs->condition('u.spamkey', 'white-transient-form', '!=');
      $spammaster_all_logs_result = $spammaster_all_logs->countQuery()->execute()->fetchField();
      $form['statistics_header']['statistics']['addrow']['logs'] = [
        '#markup' => $this->t('<p>Total Entries: <b>@logs_total</b></p>', [
          '@logs_total' => $spammaster_all_logs_result,
        ]),
      ];

      // Generate system logs stats total.
      $spammaster_system_logs = $this->connection->select('spammaster_keys', 'u');
      $spammaster_system_logs->fields('u', ['spamkey']);
      $spammaster_system_logs->condition('u.spamkey', 'exempt-needle', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'exempt-needle-straw', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'exempt-needle-sig-hide', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'exempt-needle-sig-show', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'exempt-key', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'exempt-value', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'white-transient-haf', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'white-transient-form', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'spammaster-firewall', '!=');
      $spammaster_system_logs->condition('u.spamkey', 'spammaster-honeypot', '!=');
      $spammaster_system_logs_result = $spammaster_system_logs->countQuery()->execute()->fetchField();
      $form['statistics_header']['statistics']['addrow']['system_logs'] = [
        '#markup' => $this->t('<p>Total Entries: <b>@logs_system_total</b></p>', [
          '@logs_system_total' => $spammaster_system_logs_result,
        ]),
      ];

      // Generate firewall stats total.
      $spammaster_firewall = $this->connection->select('spammaster_keys', 'u');
      $spammaster_firewall->fields('u', ['spamkey']);
      $spammaster_firewall->condition('u.spamkey', 'spammaster-firewall', '=');
      $spammaster_firewall_result = $spammaster_firewall->countQuery()->execute()->fetchField();
      $form['statistics_header']['statistics']['addrow']['firewall'] = [
        '#markup' => $this->t('<p>Total Entries: <b>@firewall_total</b></p>', [
          '@firewall_total' => $spammaster_firewall_result,
        ]),
      ];

      // Generate honeypot stats total.
      $spammaster_honeypot = $this->connection->select('spammaster_keys', 'u');
      $spammaster_honeypot->fields('u', ['spamkey']);
      $spammaster_honeypot->condition('u.spamkey', 'spammaster-honeypot', '=');
      $spammaster_honeypot_result = $spammaster_honeypot->countQuery()->execute()->fetchField();
      $form['statistics_header']['statistics']['addrow']['honeypot'] = [
        '#markup' => $this->t('<p>Total Entries: <b>@honeypot_total</b></p>', [
          '@honeypot_total' => $spammaster_honeypot_result,
        ]),
      ];

      // Construct header.
      $header_key = [
        'id' => [
          'data' => $this->t('ID'),
          'field'  => 'id',
          'specifier' => 'id',
          'sort' => 'desc',
        ],
        'date' => [
          'data' => $this->t('Date'),
          'field'  => 'date',
          'specifier' => 'date',
          'sort' => 'desc',
        ],
        'spamkey' => [
          'data' => $this->t('Type'),
          'field'  => 'spamkey',
          'specifier' => 'spamkey',
          'sort' => 'desc',
        ],
        'spamvalue' => [
          'data' => $this->t('Description'),
          'field'  => 'spamvalue',
          'specifier' => 'spamvalue',
          'sort' => 'desc',
        ],
      ];
      // Get table spammaster_keys data.
      $spammaster_spam_key = $this->connection->select('spammaster_keys', 'u')
        ->fields('u', ['id', 'date', 'spamkey', 'spamvalue'])
        ->condition('u.spamkey', 'exempt-needle', '!=')
        ->condition('u.spamkey', 'exempt-key', '!=')
        ->condition('u.spamkey', 'exempt-value', '!=')
        ->condition('u.spamkey', 'exempt-needle-straw', '!=')
        ->condition('u.spamkey', 'exempt-needle-sig-hide', '!=')
        ->condition('u.spamkey', 'exempt-needle-sig-show', '!=')
        ->condition('u.spamkey', 'white-transient-haf', '!=')
        ->condition('u.spamkey', 'white-transient-form', '!=')
        ->extend('Drupal\Core\Database\Query\TableSortExtender')->orderByHeader($header_key)
        ->extend('Drupal\Core\Database\Query\PagerSelectExtender')->limit(20)
        ->execute()->fetchAll();

      $output_key = [];
      foreach ($spammaster_spam_key as $results_key) {
        if (!empty($results_key)) {
          $output_key[$results_key->id] = [
            'id' => $results_key->id,
            'date' => $results_key->date,
            'spamkey' => $results_key->spamkey,
            'spamvalue' => $results_key->spamvalue,
          ];
        }
      }
      // Display table.
      $form['statistics_header']['table_key'] = [
        '#type' => 'tableselect',
        '#header' => $header_key,
        '#options' => $output_key,
        '#empty' => $this->t('No log found'),
      ];
      // Spam log description.
      $form['statistics_header']['description'] = [
        '#markup' => $this->t('<p>Before deleting! Spam Master keeps local logs for a maximum time (check Protection Tools Clean-up options). Older logs are automatically deleted via cron to keep your website clean and fast.</p>'),
      ];
      // Delete button at end of table, calls spammasterdeletekey function.
      $form['statistics_header']['submit'] = [
        '#type' => 'submit',
        '#attributes' => [
          'class' => ['button button--danger'],
        ],
        '#value' => $this->t('Delete Log Entry'),
        '#submit' => ['::spamMasterDeleteKey'],
      ];
      // Delete button at end of table, calls spammasterdeletekeysall function.
      $form['statistics_header']['submit_all'] = [
        '#type' => 'submit',
        '#attributes' => [
          'class' => ['button button--danger'],
        ],
        '#value' => $this->t('Delete all Statistics & Logs -> Caution, no way back'),
        '#submit' => ['::spamMasterDeleteKeysAll'],
      ];
      // Form pager if ore than 25 entries.
      $form['statistics_header']['pager'] = [
        '#type' => 'pager',
      ];

      // Form button description.
      $form['statistics_header']['header_description'] = [
        '#markup' => $this->t('<p>In order to access your online statistics please create an account or login at <a href="@spammaster_base_url" target="@spammaster_target"><b>www.spammaster.org</b></a> website with the license attached email <b>@spam_master_attached</b>.</p>', [
          '@spammaster_base_url' => 'https://www.spammaster.org',
          '@spammaster_target' => '_blank',
          '@spam_master_attached' => $spam_master_attached,
        ]),
      ];
      $form['statistics_header']['buttons'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['statistics-buttons'],
        ],
      ];
      // Add the button inside the container.
      $form['statistics_header']['buttons']['online'] = [
        '#type' => 'submit',
        '#value' => $this->t('Visit Online Statistics Page'),
        '#attributes' => [
          'class' => ['button', 'button--primary'],
        ],
        '#submit' => ['::spamMasterStatisticsOnline'],
      ];
    }
    else {
      // Form description.
      $image_check = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost() . '/modules/spammaster/images/techgasp-checkmark-16.png';
      $spammaster_total_block_count = $this->state->get('spammaster.total_block_count');
      if (empty($spammaster_total_block_count)) {
        $spammaster_total_block_count = '0';
      }
      // Insert statistics table inside tree.
      $form['statistics_header']['total_block_count'] = [
        '#markup' => $this->t('<h2>Total Firewall Blocks: <span class="spam-master-admin-red spam-master-top-admin-shadow-offline">@total_blocks</span></h2>', ['@total_blocks' => $spammaster_total_block_count]),
        '#attributes' => [
          'class' => [
            'spam-master-admin-red',
            'spam-master-top-admin-shadow-offline',
          ],
        ],
      ];
      $form['statistics_header']['header_description'] = [
        '#markup' => $this->t('<p>Thank you for using Spam Master, one of the world’s top real-time spam and exploit databases, tracking over <b>@spammaster_protection_total_number</b> million threats and growing daily. You can access free, detailed logging by registering at <a href="@spammaster_base_url" target="@spammaster_target"><b>www.spammaster.org</b></a> with your licensed email <b>@spam_master_attached</b>. For even more power, consider upgrading to a <a href="@spammaster_url" target="@spammaster_target"><img src="@image_check" alt="Get Pro Key"> <b>PRO KEY</b></a>. In addition to viewing logs directly here and supporting the Spam Master project, you’ll also gain access to our business-class servers for faster, more reliable performance. All of this comes at an affordable yearly cost.</p>', [
          '@spammaster_protection_total_number' => number_format($spammaster_protection_total_number),
          '@spam_master_attached' => $spam_master_attached,
          '@spammaster_base_url' => 'https://www.spammaster.org',
          '@spammaster_url' => 'https://www.techgasp.com/downloads/spam-master-license/',
          '@spammaster_target' => '_blank',
          '@image_check' => $image_check,
        ]),
        '#allowed_tags' => ['p', 'img', 'b'],
      ];
      $form['statistics_header']['header_description']['table'] = [
        '#type' => 'table',
        '#attributes' => [
          'class' => [
            'spammaster-responsive',
          ],
        ],
      ];
      // Insert addrow teaser image.
      $image_teaser1 = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost() . '/modules/spammaster/images/techgasp-logs.jpg';
      $form['statistics_header']['header_description']['table']['images']['teaser1'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_teaser1" alt="Get Spam Master Pro Key">', ['@image_teaser1' => $image_teaser1]),
        '#attributes' => [
          'class' => [
            'spammaster-responsive-49',
          ],
          '#allowed_tags' => ['img'],
        ],
        '#wrapper_attributes' => ['colspan' => 2],
      ];
      $image_teaser2 = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost() . '/modules/spammaster/images/techgasp-statistics.jpg';
      $form['statistics_header']['header_description']['table']['images']['teaser2'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_teaser2" alt="Get Spam Master Pro Key">', ['@image_teaser2' => $image_teaser2]),
        '#attributes' => [
          'class' => [
            'spammaster-responsive-49',
          ],
          '#allowed_tags' => ['img'],
        ],
        '#wrapper_attributes' => ['colspan' => 2],
      ];

      $form['statistics_header']['header_description']['table_1'] = [
        '#type' => 'table',
        '#attributes' => [
          'class' => [
            'spammaster-responsive',
          ],
        ],
      ];
      $form['statistics_header']['header_description']['table_1']['text']['teaser1'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_check" alt="Get Pro Key">', ['@image_check' => $image_check]) . $this->t('&nbsp;All protection features, no extra modules'),
        '#allowed_tags' => ['img'],
      ];
      $form['statistics_header']['header_description']['table_1']['text']['teaser2'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_check" alt="Get Pro Key">', ['@image_check' => $image_check]) . $this->t('&nbsp;Connection to Business RBL Server Cluster'),
        '#allowed_tags' => ['img'],
      ];
      $form['statistics_header']['header_description']['table_1']['text']['teaser3'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_check" alt="Get Pro Key">', ['@image_check' => $image_check]) . $this->t('&nbsp;For heavy duty websites and e-commerce'),
        '#allowed_tags' => ['img'],
      ];
      $form['statistics_header']['header_description']['table_1']['text']['teaser4'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_check" alt="Get Pro Key">', ['@image_check' => $image_check]) . $this->t('&nbsp;Real-Time Firewall Management'),
        '#allowed_tags' => ['img'],
      ];
      $form['statistics_header']['header_description']['table_1']['text1']['teaser1'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_check" alt="Get Pro Key">', ['@image_check' => $image_check]) . $this->t('&nbsp;Central Management for all websites'),
        '#allowed_tags' => ['img'],
      ];
      $form['statistics_header']['header_description']['table_1']['text1']['teaser2'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_check" alt="Get Pro Key">', ['@image_check' => $image_check]) . $this->t('&nbsp;Private, ticketed email support'),
        '#allowed_tags' => ['img'],
      ];
      $form['statistics_header']['header_description']['table_1']['text1']['teaser3'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<img src="@image_check" alt="Get Pro Key">', ['@image_check' => $image_check]) . $this->t('&nbsp;1 year PRO key with full protection'),
        '#allowed_tags' => ['img'],
      ];
      $form['statistics_header']['header_description']['table_1']['text1']['teaser4'] = [
        '#type' => 'markup',
        '#markup' => $this->t('&nbsp;<a class="@spammaster_class" href="@spammaster_url" target="@spammaster_target"><img src="@image_check" alt="Get Pro Key"> Upgrade to PRO</a>', [
          '@spammaster_url' => 'https://www.techgasp.com/downloads/spam-master-license/',
          '@spammaster_target' => '_blank',
          '@spammaster_class' => 'button button--primary',
          '@image_check' => $image_check,
        ]),
        '#allowed_tags' => ['span', 'img', 'a'],
      ];
    }

    // Disable the default submit button.
    $form['actions'] = [];

    return $form;

  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'spammaster.settings_log',
    ];
  }

}
