<?php

namespace Drupal\spammaster;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class service.
 */
class SpamMasterElusiveService {

  /**
   * The database connection object.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * Symfony\Component\HttpFoundation\RequestStack definition.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The Config Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The Time Service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The SpamMasterCollectService Service.
   *
   * @var \Drupal\spammaster\SpamMasterCollectService
   */
  protected $collectService;

  /**
   * The SpamMasterUserService Service.
   *
   * @var \Drupal\spammaster\SpamMasterUserService
   */
  protected $userService;

  /**
   * The SpamMasterApiService Service.
   *
   * @var \Drupal\spammaster\SpamMasterApiService
   */
  protected $apiService;

  /**
   * {@inheritdoc}
   */
  public function __construct(Connection $connection, RequestStack $requestStack, ConfigFactoryInterface $configFactory, StateInterface $state, AccountProxyInterface $currentUser, TimeInterface $time, SpamMasterCollectService $collectService, SpamMasterUserService $userService, SpamMasterApiService $apiService) {
    $this->connection = $connection;
    $this->requestStack = $requestStack;
    $this->configFactory = $configFactory;
    $this->state = $state;
    $this->currentUser = $currentUser;
    $this->time = $time;
    $this->collectService = $collectService;
    $this->userService = $userService;
    $this->apiService = $apiService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('current_user'),
      $container->get('datetime.time'),
      $container->get('spammaster.collect_service'),
      $container->get('spammaster.user_service'),
      $container->get('spammaster.api_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterElusiveCheck($spamelusive): string {
    $spammaster_date = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_license = $spammaster_settings->get('spammaster.license_key');
    $spammaster_subtype = $spammaster_settings->get('spammaster.subtype');
    $spammaster_status = $this->state->get('spammaster.license_status');
    $spammaster_total_block_count = $this->state->get('spammaster.total_block_count');
    $valid_statuses = ['VALID', 'MALFUNCTION_1', 'MALFUNCTION_2'];

    if (in_array($spammaster_status, $valid_statuses, TRUE) && $spammaster_subtype === 'prod') {
      // Call collect service.
      $collectnow = $this->requestStack->getCurrentRequest()->getClientIp();
      $spammaster_collect_service = $this->collectService;
      $is_collected = $spammaster_collect_service->spamMasterGetCollect($collectnow);
      $spamCollection = Json::decode($is_collected);
      $is_collectedIpHost = $spammaster_collect_service->getIpAndHostname($spamCollection['spammasterurl']);
      $spamCollectionIpHost = Json::decode($is_collectedIpHost);

      // User service call.
      $spamuser = ($this->currentUser && method_exists($this->currentUser, 'id')) ? $this->currentUser->id() : 0;
      $spammaster_user_service = $this->userService;
      $is_user = $spammaster_user_service->spamMasterGetUser($spamuser);
      $spamUserData = Json::decode($is_user);

      // Do we have form_id.
      $spammasterIdForm = $spamelusive->get('form_id') ?? FALSE;

      // Exclude malformed or internal links.
      if (!empty($spamCollection['spammasterurl'])) {
        $exclude_substrings = [
          'autodiscover.xml',
          'recipebox',
          'api/recipe-box',
          '1/take/2',
          '1/take/3',
          '1/take/4',
          '1/take/5',
          '1/take/6',
          '1/take/7',
          '1/take/8',
          '1/take/9',
          '1/take/10',
          '1/take/11',
          '1/take/12',
          '1/take/13',
          '1/take/14',
          '1/take/15',
          '1/take/16',
          '1/take/17',
          '1/take/18',
          '1/take/19',
        ];
        foreach ($exclude_substrings as $exclusion) {
          if (stripos($spamCollection['spammasterurl'], $exclusion) !== FALSE) {
            return 'BAIL';
          }
        }
      }

      // phpcs:ignore DrupalPractice.Variables.GetRequestData.SuperglobalAccessed
      $spampost = $_POST;
      if (!empty($spampost) && is_array($spampost)) {
        // Exclude system, search and views.
        if (
          preg_match('/[^a-zA-Z]system_/', $spammasterIdForm) !== 0 &&
          preg_match('/[^a-zA-Z]search_/', $spammasterIdForm) !== 0 &&
          preg_match('/[^a-zA-Z]views_exposed_form_/', $spammasterIdForm) !== 0
        ) {
          return 'BAIL';
        }

        // Exclude simple JavaScript detection payloads.
        if (count($spampost) === 1 && array_key_exists('js', $spampost) && $spampost['js'] === 'true') {
          return 'BAIL';
        }

        // Exempt needle.
        $spammaster_exempt_needle = $this->connection->query("SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :exclude) > :value", [
          ':key' => 'exempt-needle',
          ':exclude' => $spammasterIdForm,
          ':value' => 0,
        ])->fetchField();
        if (!empty($spammaster_exempt_needle)) {
          return 'BAIL';
        }

        $spampoststr = str_replace('=', ' ', urldecode(http_build_query($spampost, '', ' ')));
        // Exclusions by key.
        $spammaster_exempt_key = $this->connection->query("SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :exclude) > :value", [
          ':key' => 'exempt-key',
          ':exclude' => $spampoststr,
          ':value' => 0,
        ])->fetchField();
        if (!empty($spammaster_exempt_key)) {
          return 'BAIL';
        }

        // Exclusions by value.
        $spammaster_exempt_value = $this->connection->query("SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :exclude) > :value", [
          ':key' => 'exempt-value',
          ':exclude' => $spampoststr,
          ':value' => 0,
        ])->fetchField();
        if (!empty($spammaster_exempt_value)) {
          return 'BAIL';
        }

        // Web api check. Create data to be posted.
        $blog_threat_type = 'HAF';
        $result_post_content_json  = Json::encode($spampost);
        $result_post_content_trim  = substr($result_post_content_json, 0, 963);
        $result_post_content_clean = strip_tags($result_post_content_trim);
        if (empty($result_post_content_clean)) {
          $result_post_content_clean = 'is_elusive_d';
        }

        // Collect elusive email.
        $spammasteremail = $spammaster_collect_service->extractFirstEmail($spampoststr);

        // Call api service.
        $spam_master_leaning_url = 'https://www.spammaster.org/core/learn/get_learn_haf.php';

        // Build params.
        $params = [
          'blog_license_key'    => $spammaster_license,
          'blog_threat_ip'      => $spamCollection['spammasterip'],
          'blog_threat_user'    => $is_user,
          'blog_threat_type'    => $blog_threat_type,
          'blog_threat_email'   => $spammasteremail,
          'blog_threat_content' => 'is_elusive_d - ' . $result_post_content_clean,
          'blog_threat_agent'   => $spamCollection['spammasteragent'],
          'blog_threat_refe'    => $spamCollection['spammasterreferer'],
          'blog_threat_dest'    => $spamCollection['spammasterurl'],
          'blog_web_adress'     => $spamCollectionIpHost['address'],
          'blog_server_ip'      => $spamCollectionIpHost['ip'],
        ];

        $response = $this->apiService->postScanData($spam_master_leaning_url, $params, TRUE);

        if (empty($response['threat'])) {
          // Set white transient if on.
          $spammasterWhiteTransient = $this->state->get('spammaster.white_transient');
          if ('1' === $spammasterWhiteTransient) {
            $this->connection->insert('spammaster_keys')->fields([
              'date' => $spammaster_date,
              'spamkey' => 'white-transient-haf',
              'spamvalue' => $spamCollection['spammasterip'],
            ])->execute();
          }
          return 'BAIL';
        }
        else {
          // Insert ip into buffer db.
          $this->connection->insert('spammaster_threats')->fields([
            'date' => $spammaster_date,
            'threat' => $spamCollection['spammasterip'],
          ])->execute();
          // Web positive, return.
          $spammaster_total_block_count_1 = ++$spammaster_total_block_count;
          $this->state->set('spammaster.total_block_count', $spammaster_total_block_count_1);
          $this->connection->insert('spammaster_keys')->fields([
            'date' => $spammaster_date,
            'spamkey' => 'spammaster-firewall',
            'spamvalue' => 'Spam Master: firewall RBL BLOCK, Agent: ' . $spamCollection['spammasteragent'] . ', User ID: ' . $spamUserData['ID'] . ', Username: ' . $spamUserData['username'] . ', Ip: ' . $spamCollection['spammasterip'],
          ])->execute();
          return 'ELUSIVE';
        }
      }
      else {
        return 'BAIL';
      }
    }
    else {
      return 'BAIL';
    }
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterIsNeedle($spamFormId) {

    // Force needle straw.
    $spammaster_exempt_needle_straw = $this->connection->query("SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :straw) > :value", [
      ':key' => 'exempt-needle-straw',
      ':straw' => $spamFormId,
      ':value' => 0,
    ])->fetchField();
    if (!empty($spammaster_exempt_needle_straw)) {
      return "STRAW";
    }
    // Hide sig but force needle.
    $spammaster_exempt_needle_hide_sig = $this->connection->query("SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :sighide) > :value", [
      ':key' => 'exempt-needle-sig-hide',
      ':sighide' => $spamFormId,
      ':value' => 0,
    ])->fetchField();
    if (!empty($spammaster_exempt_needle_hide_sig)) {
      return "SIGHIDDEN";
    }
    // Show sig but exempt needle.
    $spammaster_exempt_needle_show_sig = $this->connection->query("SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :sigshow) > :value", [
      ':key' => 'exempt-needle-sig-show',
      ':sigshow' => $spamFormId,
      ':value' => 0,
    ])->fetchField();
    if (!empty($spammaster_exempt_needle_show_sig)) {
      return "SIGSHOW";
    }
    // Exempt needle.
    $spammaster_exempt_needle = $this->connection->query("SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :exclude) > :value", [
      ':key' => 'exempt-needle',
      ':exclude' => $spamFormId,
      ':value' => 0,
    ])->fetchField();
    if (!empty($spammaster_exempt_needle)) {
      return "NEEDLE";
    }
    return "STRAW";
  }

}
