<?php

namespace Drupal\spammaster;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class service.
 */
class SpamMasterLicService {
  use StringTranslationTrait;

  /**
   * The database connection object.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The Config Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The Time Service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The SpamMasterMailService Service.
   *
   * @var \Drupal\spammaster\SpamMasterMailService
   */
  protected $mailService;

  /**
   * The SpamMasterLicService service.
   *
   * @var \Drupal\spammmaster\SpamMasterLicService
   */
  protected $spamKey;

  /**
   * The SpamMasterLicService service.
   *
   * @var \Drupal\spammmaster\SpamMasterLicService
   */
  protected $spamCron;

  /**
   * The SpamMasterActionService Service.
   *
   * @var \Drupal\spammaster\SpamMasterActionService
   */
  protected $actionService;

  /**
   * The SpamMasterApiService Service.
   *
   * @var \Drupal\spammaster\SpamMasterApiService
   */
  protected $apiService;

  /**
   * {@inheritdoc}
   */
  public function __construct(Connection $connection, MessengerInterface $messenger, ConfigFactoryInterface $configFactory, StateInterface $state, LanguageManagerInterface $language_manager, TimeInterface $time, SpamMasterMailService $mailService, SpamMasterActionService $actionService, SpamMasterApiService $apiService) {
    $this->connection = $connection;
    $this->messenger = $messenger;
    $this->configFactory = $configFactory;
    $this->state = $state;
    $this->languageManager = $language_manager;
    $this->time = $time;
    $this->mailService = $mailService;
    $this->actionService = $actionService;
    $this->apiService = $apiService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('messenger'),
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('language_manager'),
      $container->get('datetime.time'),
      $container->get('spammaster.mail_service'),
      $container->get('spammaster.action_service'),
      $container->get('spammaster.api_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterLicAutoCreation() {
    // Process key data.
    $keyData = $this->processKeyData();
    // Prepare date.
    $spammaster_date = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $spammaster_stamp = date('YmdHis', $this->time->getCurrentTime());
    $hashNounce = 'PW9pdXNkbmVXMndzUw==';
    $spammaster_type = 'FREE';
    $spam_master_auto_update = 'false';
    $spam_master_is_cloudflare = 'false';
    $spamCron = "FALSE";
    // Generate hashes and dates.
    $spam_master_lic_hash = substr(md5(uniqid(mt_rand(), TRUE)), 0, 64) ?: 'md5-' . $spammaster_stamp;
    $spam_master_db_protection_hash = substr(md5(uniqid(mt_rand(), TRUE)), 0, 64) ?: 'md5-' . $spammaster_stamp;
    $spam_master_db_protection_hash_dt = $spammaster_date;

    // Build params for license API call.
    $params = [
      'spam_license_key'    => trim($spam_master_lic_hash),
      'spam_trial_nounce'   => $hashNounce,
      'platform'            => $keyData['system_data']['spammaster_platform'],
      'platform_version'    => $keyData['system_data']['spammaster_platform_version'],
      'platform_type'       => $spammaster_type,
      'spam_master_version' => $keyData['system_data']['spammaster_version'],
      'spam_master_type'    => $keyData['system_data']['spammaster_type'],
      'blog_name'           => $keyData['system_data']['spammaster_site_name'],
      'blog_address'        => $keyData['system_data']['address'],
      'blog_email'          => $keyData['system_data']['spammaster_admin_email'],
      'blog_hostname'       => $keyData['soft_hard']['spammaster_hostname'],
      'blog_ip'             => $keyData['soft_hard']['spammaster_ip'],
      'blog_up'             => $spam_master_auto_update,
      'blog_cloud'          => $spam_master_is_cloudflare,
      'spam_master_db'      => $spam_master_db_protection_hash,
      'spam_master_logs'    => 0,
      'spam_master_cron'    => $spamCron,
    ];

    // Delegate HTTP POST to apiService.
    $spammaster_endpoint = 'https://www.spammaster.org/core/lic/lic_gen.php';
    $response = $this->apiService->postLicenseData($spammaster_endpoint, $params);

    // Extract response or set defaults.
    if (empty($response)) {
      // Server unreachable.
      $spammaster_type_set = 'EMPTY';
      $spammaster_status = 'INACTIVE';
      $spammaster_protection_total_number = '0';
      $spammaster_alert_level_received = '';
      $spammaster_alert_level_p_text = '';
      $spam_master_expires = '0000-00-00 00:00:00';
      $spam_master_attached = '';
    }
    else {
      $spammaster_status = $response['status'] ?? '';
      if ($spammaster_status === 'VALID') {
        // Normal case, populate with server data.
        $spammaster_type_set = $response['type'] ?? 'EMPTY';
        $spammaster_protection_total_number = $response['threats'] ?? '0';
        $spammaster_alert_level_received = $response['alert'] ?? '';
        $spammaster_alert_level_p_text = $response['percent'] ?? '';
        $spam_master_expires = $response['expires'] ?? '0000-00-00 00:00:00';
        $spam_master_attached = $response['attached'] ?? '';
      }
      else {
        // Any other status, fill all variables with status string.
        $spammaster_type_set = 'EMPTY';
        $spammaster_protection_total_number = '0';
        $spammaster_alert_level_received = '';
        $spammaster_alert_level_p_text = '';
        $spam_master_expires = '0000-00-00 00:00:00';
        $spam_master_attached = '';
      }
    }

    // Store received data in module settings.
    $this->configFactory->getEditable('spammaster.settings')
      ->set('spammaster.license_key', $spam_master_lic_hash)
      ->save();

    // Store state values including update flags.
    $lic_values = [
      'spammaster.type' => $spammaster_type_set,
      'spammaster.license_status' => $spammaster_status,
      'spammaster.license_alert_level' => $spammaster_alert_level_received,
      'spammaster.license_protection' => $spammaster_protection_total_number,
      'spammaster.license_probability' => $spammaster_alert_level_p_text,
      'spammaster.license_expires' => $spam_master_expires,
      'spammaster.spam_master_auto_update' => $spam_master_auto_update,
      'spammaster.spam_master_db_protection_hash' => $spam_master_db_protection_hash,
      'spammaster.spam_master_db_protection_hash_dt' => $spam_master_db_protection_hash_dt,
      'spammaster.spam_master_attached' => $spam_master_attached,
      'spammaster.db_install_version_new' => '000',
      'spammaster.db_install_version_221' => '0',
      'spammaster.db_install_version_223' => '0',
      'spammaster.db_install_version_227' => '0',
      'spammaster.db_install_version_229' => '0',
      'spammaster.db_install_version_231' => '0',
      'spammaster.db_install_version_238' => '0',
      'spammaster.db_install_version_242' => '0',
      'spammaster.db_install_version_243' => '0',
      'spammaster.db_install_version_244' => '0',
      'spammaster.db_install_version_246' => '0',
      'spammaster.db_install_version_249' => '0',
      'spammaster.db_install_version_253' => '0',
      'spammaster.db_install_version_254' => '0',
      'spammaster.db_install_version_255' => '0',
      'spammaster.db_install_version_256' => '0',
      'spammaster.db_install_version_257' => '0',
      'spammaster.db_install_version_260' => '0',
      'spammaster.db_install_version_262' => '0',
      'spammaster.db_install_version_265' => '0',
    ];
    $this->state->setMultiple($lic_values);

    // Call mail service.
    $this->mailService->spamMasterLicGenCreation();

  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterLicManualCreation() {

    // Process key data.
    $keyData = $this->processKeyData();
    // Prepare date.
    $spammaster_dt = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $spamCron = "MAN";

    // Build params for license API call.
    $params = [
      'spam_license_key'    => trim($keyData['system_data']['spammaster_license']),
      'platform'            => $keyData['system_data']['spammaster_platform'],
      'platform_version'    => $keyData['system_data']['spammaster_platform_version'],
      'platform_type'       => $keyData['system_data']['spammaster_multisite_joined'],
      'spam_master_version' => $keyData['system_data']['spammaster_version'],
      'spam_master_type'    => $keyData['system_data']['spammaster_type'],
      'blog_name'           => $keyData['system_data']['spammaster_site_name'],
      'blog_address'        => $keyData['system_data']['address'],
      'blog_email'          => $keyData['system_data']['spammaster_admin_email'],
      'blog_hostname'       => $keyData['soft_hard']['spammaster_hostname'],
      'blog_ip'             => $keyData['soft_hard']['spammaster_ip'],
      'blog_up'             => $keyData['system_data']['spam_master_auto_update'],
      'blog_cloud'          => $keyData['system_data']['spam_master_is_cloudflare'],
      'spam_master_db'      => $keyData['hash_protection'],
      'spam_master_logs'    => $keyData['health_view'],
      'spam_master_cron'    => $spamCron,
    ];

    // Delegate HTTP POST to apiService.
    $response = $this->apiService->postLicenseData($keyData['system_data']['spammaster_endpoint'], $params);

    if (!empty($response)) {
      $spammaster_type_set = $response['type'];
      $spammaster_status = $response['status'];
      $spammaster_protection_total_number = $response['threats'];
      $spammaster_alert_level_received = $response['alert'];
      $spammaster_alert_level_p_text = $response['percent'];
      $spam_master_expires = $response['expires'];
      $spam_master_attached = $response['attached'];

      $spama = $response['a'];
      if ('1' === $spama) {
        $this->state->set('spammaster.new_options', '1');
        $this->connection->insert('spammaster_keys')->fields([
          'date' => $spammaster_dt,
          'spamkey' => 'spammaster',
          'spamvalue' => 'Spam Master: Action service called via license manual run.',
        ])->execute();
        // Spam action service.
        $spammaster_action_service = $this->actionService;
        $spammaster_action_service->spamMasterAct($spama);
      }
      // Store state values.
      $manual_values = [
        'spammaster.type'                 => $spammaster_type_set,
        'spammaster.license_status'       => $spammaster_status,
        'spammaster.license_alert_level'  => $spammaster_alert_level_received,
        'spammaster.license_protection'   => $spammaster_protection_total_number,
        'spammaster.license_probability'  => $spammaster_alert_level_p_text,
        'spammaster.license_expires'      => $spam_master_expires,
        'spammaster.spam_master_attached' => $spam_master_attached,
      ];
      $this->state->setMultiple($manual_values);
    }
    // Display status to user.
    $spammaster_statuses = [
      'INACTIVE',
      'MALFUNCTION_1',
      'MALFUNCTION_2',
      'MALFUNCTION_3',
      'MALFUNCTION_4',
      'MALFUNCTION_5',
      'MALFUNCTION_6',
      'MALFUNCTION_7',
      'MALFUNCTION_8',
      'UNSTABLE',
      'HIGH_VOLUME',
      'DISCONNECTED',
      'EXPIRED',
      'CODE_TAMPER_1',
    ];
    if (in_array($spammaster_status, $spammaster_statuses, TRUE)) {
      $this->messenger->addError($this->t('License key @spammaster_license status is: @spammaster_status. Check Spam Master configuration page and read more about statuses.', [
        '@spammaster_license' => $keyData['system_data']['spammaster_license'],
        '@spammaster_status' => $spammaster_status,
      ]));
      // Spam Master log.
      $this->connection->insert('spammaster_keys')->fields([
        'date' => $spammaster_dt,
        'spamkey' => 'spammaster-license',
        'spamvalue' => 'Spam Master: license manual status check: ' . $spammaster_status,
      ])->execute();
    }
    else {
      // Log message.
      $this->connection->insert('spammaster_keys')->fields([
        'date' => $spammaster_dt,
        'spamkey' => 'spammaster-license',
        'spamvalue' => 'Spam Master: license manual status check: ' . $spammaster_status,
      ])->execute();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterLicDaily($spamKey, $spamCron) {

    // Process key data.
    $spammaster_license_status = $this->state->get('spammaster.license_status');
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_subtype = $spammaster_settings->get('spammaster.subtype');
    $valid_statuses = [
      'VALID',
      'MALFUNCTION_1',
      'MALFUNCTION_2',
      'UNSTABLE',
      'HIGH_VOLUME',
      'CODE_TAMPER_1',
    ];
    if (in_array($spammaster_license_status, $valid_statuses, TRUE) && $spammaster_subtype === 'prod') {
      // Colect data.
      $keyData = $this->processKeyData();
      // Prepare date.
      $spammaster_dt = date('Y-m-d H:i:s', $this->time->getCurrentTime());
      if (empty($spamCron)) {
        $spamCron = "EMP";
      }

      $params = [
        'spam_license_key'    => $keyData['system_data']['spammaster_license'],
        'platform'            => $keyData['system_data']['spammaster_platform'],
        'platform_version'    => $keyData['system_data']['spammaster_platform_version'],
        'platform_type'       => $keyData['system_data']['spammaster_multisite_joined'],
        'spam_master_version' => $keyData['system_data']['spammaster_version'],
        'spam_master_type'    => $keyData['system_data']['spammaster_type'],
        'blog_name'           => $keyData['system_data']['spammaster_site_name'],
        'blog_address'        => $keyData['system_data']['address'],
        'blog_email'          => $keyData['system_data']['spammaster_admin_email'],
        'blog_hostname'       => $keyData['soft_hard']['spammaster_hostname'],
        'blog_ip'             => $keyData['soft_hard']['spammaster_ip'],
        'blog_up'             => $keyData['system_data']['spam_master_auto_update'],
        'blog_cloud'          => $keyData['system_data']['spam_master_is_cloudflare'],
        'spam_master_db'      => $keyData['hash_protection'],
        'spam_master_logs'    => $keyData['health_view'],
        'spam_master_cron'    => $spamCron,
      ];

      // Delegate HTTP POST to apiService.
      $response = $this->apiService->postLicenseData($keyData['system_data']['spammaster_endpoint'], $params);

      if (!empty($response)) {
        $spammaster_type_set = $response['type'];
        $spammaster_status = $response['status'];
        $spammaster_protection_total_number = $response['threats'];
        $spammaster_alert_level_received = $response['alert'];
        $spammaster_alert_level_p_text = $response['percent'];
        $spam_master_expires = $response['expires'];
        $spam_master_attached = $response['attached'];

        $spama = $response['a'];
        if ('1' === $spama) {
          $this->state->set('spammaster.new_options', '1');
          $this->connection->insert('spammaster_keys')->fields([
            'date' => $spammaster_dt,
            'spamkey' => 'spammaster',
            'spamvalue' => 'Spam Master: Action service called via license cron.',
          ])->execute();
          // Spam action service.
          $spammaster_action_service = $this->actionService;
          $spammaster_action_service->spamMasterAct($spama);
        }
        // Store state values.
        $daily_values = [
          'spammaster.type'                 => $spammaster_type_set,
          'spammaster.license_status'       => $spammaster_status,
          'spammaster.license_alert_level'  => $spammaster_alert_level_received,
          'spammaster.license_protection'   => $spammaster_protection_total_number,
          'spammaster.license_probability'  => $spammaster_alert_level_p_text,
          'spammaster.license_expires'      => $spam_master_expires,
          'spammaster.spam_master_attached' => $spam_master_attached,
        ];
        $this->state->setMultiple($daily_values);
      }

      // Call mail service for all requests.
      $spammaster_mail_service = $this->mailService;

      // Display status to user.
      if ('VALID' === $spammaster_status) {
        // Log message.
        $this->connection->insert('spammaster_keys')->fields([
          'date' => $spammaster_dt,
          'spamkey' => 'spammaster-cron',
          'spamvalue' => 'Spam Master: cron license success. Status: ' . $spammaster_status,
        ])->execute();
      }
      else {
        // Log message.
        $this->connection->insert('spammaster_keys')->fields([
          'date' => $spammaster_dt,
          'spamkey' => 'spammaster-cron',
          'spamvalue' => 'Spam Master: cron license warning. Status: ' . $spammaster_status,
        ])->execute();

        // Call mail service function.
        $spammaster_mail_service->spamMasterLicMalfunctions();
      }
      $spammaster_license_alert_level = $this->state->get('spammaster.license_alert_level');
      if ('ALERT_3' === $spammaster_license_alert_level) {
        // Log message.
        $this->connection->insert('spammaster_keys')->fields([
          'date' => $spammaster_dt,
          'spamkey' => 'spammaster-cron',
          'spamvalue' => 'Spam Master: cron alert level 3 detected.',
        ])->execute();

        // Call mail service function.
        $spammaster_mail_service->spamMasterLicAlertLevel3();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterLicDisable() {

    // Process key data.
    $spammaster_license_status = $this->state->get('spammaster.license_status');

    // Colect data.
    $keyData = $this->processKeyData();
    $spamCron = "UNI";

    $params = [
      'spam_license_key'    => $keyData['system_data']['spammaster_license'],
      'platform'            => $keyData['system_data']['spammaster_platform'],
      'platform_version'    => $keyData['system_data']['spammaster_platform_version'],
      'platform_type'       => $keyData['system_data']['spammaster_multisite_joined'],
      'spam_master_version' => $keyData['system_data']['spammaster_version'],
      'spam_master_type'    => $keyData['system_data']['spammaster_type'],
      'spam_master_status'  => $spammaster_license_status,
      'blog_name'           => $keyData['system_data']['spammaster_site_name'],
      'blog_address'        => $keyData['system_data']['address'],
      'blog_email'          => $keyData['system_data']['spammaster_admin_email'],
      'blog_hostname'       => $keyData['soft_hard']['spammaster_hostname'],
      'blog_ip'             => $keyData['soft_hard']['spammaster_ip'],
      'blog_up'             => $keyData['system_data']['spam_master_auto_update'],
      'blog_cloud'          => $keyData['system_data']['spam_master_is_cloudflare'],
      'spam_master_db'      => $keyData['hash_protection'],
      'spam_master_logs'    => $keyData['health_view'],
      'spam_master_cron'    => $spamCron,
    ];

    // Delegate HTTP POST to apiService.
    $spammasterEndpoint = 'https://www.spammaster.org/core/lic/get_other.php';
    $response = $this->apiService->postLicenseData($spammasterEndpoint, $params);

    return $response;
  }

  /**
   * Process SpamMaster data.
   *
   * This includes server info, key and hash data.
   *
   * @return array
   *   SpamMaster data information.
   */
  private function processKeyData(): array {
    return [
      'system_data'     => $this->getSystemData(),
      'hash_protection' => $this->getHashKey(),
      'soft_hard'       => $this->getSoftHard(),
      'health_view'     => $this->getHealthView(),
    ];
  }

  /**
   * Get SpamMaster system data.
   *
   * @return array
   *   System information.
   */
  private function getSystemData(): array {
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_settings_protection = $this->configFactory->get('spammaster.settings_protection');
    $spam_master_is_cloudflare = $spammaster_settings_protection->get('spammaster.spam_master_is_cloudflare');
    if (empty($spam_master_is_cloudflare) || '0' === $spam_master_is_cloudflare) {
      $spam_master_is_cloudflare = 'false';
    }
    if ('1' === $spam_master_is_cloudflare) {
      $spam_master_is_cloudflare = 'true';
    }
    $spam_master_auto_update = $this->state->get('spammaster.spam_master_auto_update');
    if (empty($spam_master_auto_update)) {
      $spam_master_auto_update = 'false';
      $this->state->set('spammaster.spam_master_auto_update', $spam_master_auto_update);
    }
    // Prepare lazy values.
    $spam_master_short_date = date('Y-m-d', $this->time->getCurrentTime());
    $lazy_values = [
      'spammaster.spam_master_license_sync_date' => $spam_master_short_date,
      'spammaster.spam_master_license_sync_run' => '0',
    ];
    $this->state->setMultiple($lazy_values);
    // Prepare platform.
    $spammaster_platform_type = 'NO';
    $spammaster_n_websites = '0';
    $spammaster_multisite_joined = $spammaster_platform_type . ' - ' . $spammaster_n_websites;
    // Prepare address.
    $url_options = [
      'absolute' => TRUE,
      'language' => $this->languageManager->getCurrentLanguage(),
    ];
    $address_un = Url::fromRoute('<front>', [], $url_options)->toString();
    $address = rtrim($address_un, '/');
    if (empty($address)) {
      $spammaster_compressed = date('YmdHis', $this->time->getCurrentTime());
      $address = 'https://' . $spammaster_compressed . '.drup';
    }
    $endpoint = 'https://www.spammaster.org/core/lic/get_lic.php';
    $basic_data = [
      'spammaster_endpoint'             => $endpoint,
      'spammaster_license'              => $spammaster_settings->get('spammaster.license_key'),
      'spammaster_platform'             => 'Drupal',
      'spammaster_platform_version'     => \Drupal::VERSION,
      'spammaster_multisite_joined'     => $spammaster_multisite_joined,
      'spammaster_site_name'            => $site_settings->get('name'),
      'address'                         => $address,
      'spammaster_admin_email'          => $site_settings->get('mail'),
      'spammaster_version'              => $this->state->get('spammaster.version'),
      'spammaster_type'                 => $this->state->get('spammaster.type'),
      'spam_master_auto_update'         => $spam_master_auto_update,
      'spam_master_is_cloudflare'       => $spam_master_is_cloudflare,
    ];
    return $basic_data;
  }

  /**
   * Get SpamMaster encryption hash.
   *
   * @return string
   *   Gathered hash.
   */
  private function getHashKey() {
    $spammaster_dt = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $spam_master_db_protection_hash_dt = $this->state->get('spammaster.spam_master_db_protection_hash_dt');
    $spam_master_db_protection_hash = $this->state->get('spammaster.spam_master_db_protection_hash');
    if (empty($spam_master_db_protection_hash_dt)) {
      $this->state->set('spammaster.spam_master_db_protection_hash_dt', $spammaster_dt);
      $spam_master_db_protection_hash_dt = $spammaster_dt;
    }
    // Process db protection hash date time.
    $plus_hash_dt = date('Y-m-d H:i:s', strtotime('+7 days', strtotime($spam_master_db_protection_hash_dt)));
    if ($spammaster_dt >= $plus_hash_dt) {
      // Refresh db protection hash.
      $spam_master_db_protection_hash = substr(md5(uniqid(mt_rand(), TRUE)), 0, 64);
      if (empty($spam_master_db_protection_hash)) {
        $spammaster_compressed = date('YmdHis', $this->time->getCurrentTime());
        $spam_master_db_protection_hash = 'md5-' . $spammaster_compressed;
      }
      $new_hash_values = [
        'spammaster.spam_master_db_protection_hash_dt' => $spammaster_dt,
        'spammaster.spam_master_db_protection_hash' => $spam_master_db_protection_hash,
      ];
      $this->state->setMultiple($new_hash_values);
    }
    return $spam_master_db_protection_hash;
  }

  /**
   * Get Hardware, software information.
   *
   * @return array
   *   Hardware, software information.
   */
  private function getSoftHard(): array {
    if (isset($_SERVER['SERVER_ADDR'])) {
      $spammaster_ip = substr($_SERVER['SERVER_ADDR'], 0, 48);
      // Should bot be localhost.
      if (empty($spammaster_ip) || '0' === $spammaster_ip || '127.0.0.1' === $spammaster_ip) {
        if (isset($_SERVER['SERVER_NAME'])) {
          $spam_master_ip_gethostbyname = gethostbyname($_SERVER['SERVER_NAME']);
          $spammaster_ip = substr($spam_master_ip_gethostbyname, 0, 48);
          if (empty($spam_master_ip_gethostbyname) || '0' === $spam_master_ip_gethostbyname) {
            $spam_master_urlparts = parse_url($spammaster_site_url);
            $spam_master_hostname_temp = $spam_master_urlparts['host'];
            $spam_master_result = dns_get_record($spam_master_hostname_temp, DNS_A);
            $spammaster_ip = substr($spam_master_result[0]['ip'], 0, 48);
          }
        }
        else {
          $spammaster_ip = 'I 000';
        }
      }
      $spammaster_hostname = substr(gethostbyaddr($_SERVER['SERVER_ADDR']), 0, 256);
      if (empty($spammaster_hostname) || '0' === $spammaster_hostname || '127.0.0.1' === $spammaster_hostname) {
        if (isset($_SERVER['SERVER_NAME'])) {
          $spam_master_ho_gethostbyname = gethostbyname($_SERVER['SERVER_NAME']);
          $spammaster_hostname = substr($spam_master_ho_gethostbyname, 0, 256);
          if (empty($spam_master_ho_gethostbyname) || '0' === $spam_master_ho_gethostbyname) {
            $spam_master_urlparts = parse_url($spammaster_site_url);
            $spam_master_hostname_temp = $spam_master_urlparts['host'];
            $spam_master_result = dns_get_record($spam_master_hostname_temp, DNS_A);
            $spammaster_hostname = substr($spam_master_result[0]['ip'], 0, 256);
          }
        }
        else {
          $spammaster_hostname = 'H 000';
        }
      }
    }
    else {
      if (isset($_SERVER['SERVER_NAME'])) {
        $spam_master_ip_gethostbyname = gethostbyname($_SERVER['SERVER_NAME']);
        $spammaster_ip = substr($spam_master_ip_gethostbyname, 0, 48);
        if (empty($spam_master_ip_gethostbyname) || '0' === $spam_master_ip_gethostbyname) {
          $spam_master_urlparts = parse_url($spammaster_site_url);
          $spam_master_hostname_temp = $spam_master_urlparts['host'];
          $spam_master_result = dns_get_record($spam_master_hostname_temp, DNS_A);
          $spammaster_ip = substr($spam_master_result[0]['ip'], 0, 48);
        }
        else {
          $spammaster_ip = 'I 001';
        }
        $spam_master_ho_gethostbyname = gethostbyname($_SERVER['SERVER_NAME']);
        $spammaster_hostname = substr($spam_master_ho_gethostbyname, 0, 256);
        if (empty($spam_master_ho_gethostbyname) || '0' === $spam_master_ho_gethostbyname) {
          $spam_master_urlparts = parse_url($spammaster_site_url);
          $spam_master_hostname_temp = $spam_master_urlparts['host'];
          $spam_master_result = dns_get_record($spam_master_hostname_temp, DNS_A);
          $spammaster_hostname = substr($spam_master_result[0]['ip'], 0, 256);
        }
        else {
          $spammaster_hostname = 'H 001';
        }
      }
      else {
        $spammaster_ip = 'I 002';
        $spammaster_hostname = 'H 002';
      }
    }
    $softhard_data = [
      'spammaster_ip'       => $spammaster_ip,
      'spammaster_hostname' => $spammaster_hostname,
    ];
    return $softhard_data;
  }

  /**
   * Get SpamMaster light health view data.
   *
   * @return string
   *   SpamMaster light health information.
   */
  private function getHealthView() {
    // Get buffer count.
    $spammaster_buffer_size = $this->connection->select('spammaster_threats', 'u');
    $spammaster_buffer_size->fields('u', ['threat']);
    $spammaster_buffer_size_result = $spammaster_buffer_size->countQuery()->execute()->fetchField();
    if (empty($spammaster_buffer_size_result)) {
      $spammaster_buffer_size_result = '1';
    }
    // Get white count.
    $spammaster_white_size = $this->connection->select('spammaster_white', 'u');
    $spammaster_white_size->fields('u', ['white_']);
    $spammaster_white_size_result = $spammaster_white_size->countQuery()->execute()->fetchField();
    if (empty($spammaster_white_size_result)) {
      $spammaster_white_size_result = '0';
    }
    // Get logs count.
    $spammaster_spam_watch_query = $this->connection->select('spammaster_keys', 'u');
    $spammaster_spam_watch_query->fields('u', ['spamkey']);
    $spammaster_spam_watch_result = $spammaster_spam_watch_query->countQuery()->execute()->fetchField();
    if (empty($spammaster_spam_watch_result)) {
      $spammaster_spam_watch_result = '0';
    }
    // Get exempt count.
    $spammaster_exempt_count = $this->connection->select('spammaster_keys', 'u');
    $spammaster_exempt_count->condition('u.spamkey', '%' . $this->connection->escapeLike('exempt') . '%', 'LIKE');
    $spammaster_exempt_count->fields('u', ['spamkey']);
    $spammaster_exempt_query = $spammaster_exempt_count->countQuery()->execute()->fetchField();
    if (empty($spammaster_exempt_query)) {
      $spammaster_exempt_query = '0';
    }
    // Process other health values.
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_subtype = $spammaster_settings->get('spammaster.subtype');
    $spammaster_white_transient = $this->state->get('spammaster.white_transient');
    $spammaster_settings_protection = $this->configFactory->get('spammaster.settings_protection');
    $spammaster_basic_firewall_rules = $spammaster_settings_protection->get('spammaster.basic_firewall_rules');
    // Prepare light health view.
    $spam_count_pre_ar = [
      'buf' => $spammaster_buffer_size_result,
      'whi' => $spammaster_white_size_result,
      'log' => $spammaster_spam_watch_result,
      'exe' => $spammaster_exempt_query,
      'sub' => $spammaster_subtype,
      'whf' => $spammaster_white_transient,
      'fir' => $spammaster_basic_firewall_rules,
    ];
    $healtview_data = Json::encode($spam_count_pre_ar);
    return $healtview_data;
  }

}
