<?php

namespace Drupal\spammaster\EventSubscriber;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Drupal\spammaster\SpamMasterCleanUpService;
use Drupal\spammaster\SpamMasterCollectService;
use Drupal\spammaster\SpamMasterElusiveService;
use Drupal\spammaster\SpamMasterHoneypotService;
use Drupal\spammaster\SpamMasterKeyService;
use Drupal\spammaster\SpamMasterUpdaterService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Class firewall subscriber.
 */
class SpamMasterFirewallSubscriber implements EventSubscriberInterface {

  /**
   * The database connection object.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * Symfony\Component\HttpFoundation\RequestStack definition.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Drupal\Core\Session\AccountProxy definition.
   *
   * @var \Drupal\Core\Session\AccountProxy
   */
  protected $currentUser;

  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The Config Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The Time Service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The SpamMasterKeyService Service.
   *
   * @var \Drupal\spammaster\SpamMasterKeyService
   */
  protected $keyService;

  /**
   * The SpamMasterCollectService Service.
   *
   * @var \Drupal\spammaster\SpamMasterCollectService
   */
  protected $collectService;

  /**
   * The SpamMasterHoneypotService Service.
   *
   * @var \Drupal\spammaster\SpamMasterHoneypotService
   */
  protected $honeyService;

  /**
   * The SpamMasterElusiveService Service.
   *
   * @var \Drupal\spammaster\SpamMasterElusiveService
   */
  protected $elusiveService;

  /**
   * The SpamMasterUpdaterService Service.
   *
   * @var \Drupal\spammaster\SpamMasterUpdaterService
   */
  protected $updaterService;

  /**
   * The SpamMasterCleanUpService Service.
   *
   * @var \Drupal\spammaster\SpamMasterCleanUpService
   */
  protected $cleanUpService;

  /**
   * {@inheritdoc}
   */
  public function __construct(Connection $connection, RequestStack $requestStack, AccountProxyInterface $currentUser, StateInterface $state, ConfigFactoryInterface $configFactory, TimeInterface $time, SpamMasterKeyService $keyService, SpamMasterCollectService $collectService, SpamMasterHoneypotService $honeyService, SpamMasterElusiveService $elusiveService, SpamMasterUpdaterService $updaterService, SpamMasterCleanUpService $cleanUpService) {
    $this->connection = $connection;
    $this->requestStack = $requestStack;
    $this->currentUser = $currentUser;
    $this->state = $state;
    $this->configFactory = $configFactory;
    $this->time = $time;
    $this->keyService = $keyService;
    $this->collectService = $collectService;
    $this->honeyService = $honeyService;
    $this->elusiveService = $elusiveService;
    $this->updaterService = $updaterService;
    $this->cleanUpService = $cleanUpService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('request_stack'),
      $container->get('current_user'),
      $container->get('state'),
      $container->get('config.factory'),
      $container->get('datetime.time'),
      $container->get('spammaster.key_service'),
      $container->get('spammaster.collect_service'),
      $container->get('spammaster.honeypot_service'),
      $container->get('spammaster.elusive_service'),
      $container->get('spammaster.updater_service'),
      $container->get('spammaster.clean_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function checkForRedirection(RequestEvent $event): void {
    $request = $this->requestStack->getCurrentRequest();
    $spammaster_ip = $request->getClientIp();
    $spammaster_date = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $spammaster_date_short = date('Y-m-d', $this->time->getCurrentTime());

    // Module and DB version check.
    $spammaster_version = $this->state->get('spammaster.version');
    if ($spammaster_version === NULL || $spammaster_version !== '2.69') {
      $this->state->set('spammaster.version', '2.69');
    }
    $db_install_version_new = $this->state->get('spammaster.db_install_version_new');
    if ($db_install_version_new === NULL || $db_install_version_new !== '269') {
      $this->state->set('spammaster.db_install_version_new', '269');
    }
    $db_install_version = $this->state->get('spammaster.db_install_version');
    if ($db_install_version === NULL || $db_install_version_new !== $db_install_version) {
      $this->updaterService->spamMasterUpDb(TRUE);
    }
    // Disc notification.
    $spammaster_type = $this->state->get('spammaster.type');
    if ($spammaster_type === 'FREE') {
      $this->updaterService->spamMasterDateCheck($spammaster_date_short);
    }

    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_subtype = $spammaster_settings->get('spammaster.subtype');
    $spammaster_status = $this->state->get('spammaster.license_status');

    // Only proceed for valid statuses and prod subtype.
    $valid_statuses = ['VALID', 'MALFUNCTION_1', 'MALFUNCTION_2'];
    if (!in_array($spammaster_status, $valid_statuses, TRUE) || $spammaster_subtype !== 'prod') {
      return;
    }

    // Lazy license key sync.
    $spammaster_settings_protection = $this->configFactory->get('spammaster.settings_protection');
    $basic_firewall_rules = $spammaster_settings_protection->get('spammaster.basic_firewall_rules');
    $flood_control = $spammaster_settings_protection->get('spammaster.flood_control');
    $spam_master_license_sync_date = $this->state->get('spammaster.spam_master_license_sync_date');
    $spam_master_license_sync_run = $this->state->get('spammaster.spam_master_license_sync_run');
    $sync_date_plus_days = date('Y-m-d', strtotime('+2 days', strtotime($spam_master_license_sync_date ?? $spammaster_date_short)));
    if (
      $spammaster_date_short >= $sync_date_plus_days &&
      $spam_master_license_sync_run !== '1'
    ) {
      $this->state->setMultiple([
        'spammaster.spam_master_license_sync_date' => $spammaster_date_short,
        'spammaster.spam_master_license_sync_run' => '1',
      ]);
      $this->keyService->spamMasterKeyLazy(TRUE, 'LAZY');
    }

    // Gather form_id and honeypot fields from request.
    $form_id = $request->request->get('form_id', FALSE);
    $spammaster_extra_field_1 = $request->request->get('spammaster_extra_field_1', FALSE);
    $spammaster_extra_field_2 = $request->request->get('spammaster_extra_field_2', FALSE);

    // Whitelist check.
    $is_whitelisted = FALSE;
    if ($form_id) {
      $whitelisted_ip = $this->connection->query(
        "SELECT white FROM {spammaster_white} WHERE white = :ip OR white = :formid",
        [':ip' => $spammaster_ip, ':formid' => $form_id]
      )->fetchField();
      $is_whitelisted = !empty($whitelisted_ip);
    }
    else {
      $whitelisted_ip = $this->connection->query(
        "SELECT white FROM {spammaster_white} WHERE white = :ip",
        [':ip' => $spammaster_ip]
      )->fetchField();
      $is_whitelisted = !empty($whitelisted_ip);
    }

    // Administrator role exemption.
    $is_admin = (
      $this->currentUser->hasPermission('administer site configuration') ||
      $this->currentUser->hasPermission('administer nodes')
    );

    // Full exemption for permanent whitelist or admin.
    if ($is_whitelisted || $is_admin) {
      return;
    }

    // Run POST flood control for all non-exempt users.
    if ('1' === $flood_control) {
      if ($request->isMethod('POST')) {
        $flood_control_window = $spammaster_settings_protection->get('spammaster.flood_control_window');
        $flood_control_limit = $spammaster_settings_protection->get('spammaster.flood_control_limit');
        $now = $this->time->getCurrentTime();

        // Use the service’s flood cleanup function.
        $this->cleanUpService->spamMasterCleanUpFloodLogs($flood_control_window + 1, $this->time);

        // Count recent POSTs within window.
        $recent_attempts = $this->connection->query(
          "SELECT COUNT(*) FROM {spammaster_keys} WHERE spamkey = :key AND spamvalue = :val AND date > :since",
          [
            ':key' => 'post-flood',
            ':val' => $spammaster_ip,
            ':since' => date('Y-m-d H:i:s', $now - $flood_control_window),
          ]
        )->fetchField();

        // Check if we've already called the API for this IP during this window.
        $block_marker_time = $this->connection->query(
          "SELECT MAX(date) FROM {spammaster_keys} WHERE spamkey = :key AND spamvalue = :val",
          [
            ':key' => 'post-flood-blocked',
            ':val' => $spammaster_ip,
          ]
        )->fetchField();

        $is_block_marker_recent = FALSE;
        if ($block_marker_time) {
          $is_block_marker_recent = (strtotime($block_marker_time) > ($now - $flood_control_window));
        }

        // Block only if currently flooded.
        if ($recent_attempts >= $flood_control_limit) {
          // Only call the API once per window per IP.
          if (
            !$is_block_marker_recent
            && in_array(
              $basic_firewall_rules,
              [
                '1',
                '2',
              ],
              TRUE
            )
          ) {
            $isElusive = $this->elusiveService->spamMasterFloodCheck($request->request);

            // Insert a block marker with the current time.
            $this->connection->insert('spammaster_keys')->fields([
              'date'      => date('Y-m-d H:i:s', $now),
              'spamkey'   => 'post-flood-blocked',
              'spamvalue' => $spammaster_ip,
            ])->execute();
          }
          $event->setResponse($this->getFirewallBlockResponse(
            $spammaster_ip,
            $request->headers->get('User-Agent'),
            'rate_limit'
          ));
          $event->stopPropagation();
          return;
        }

        // Normal insert for every POST not rate-limited.
        $this->connection->insert('spammaster_keys')->fields([
          'date'      => date('Y-m-d H:i:s', $now),
          'spamkey'   => 'post-flood',
          'spamvalue' => $spammaster_ip,
        ])->execute();
      }
    }

    // Transient whitelist check if enabled (does NOT bypass flood control).
    $transient_whitelisted = FALSE;
    $spammasterWhiteTransient = $this->state->get('spammaster.white_transient');
    if ($spammasterWhiteTransient === '1') {
      // Call clean-up service to delete old transients.
      $spammaster_cleanup_service = $this->cleanUpService;
      $spammaster_cleanup_service->spamMasterCleanUpTransNow($spammasterWhiteTransient);
      $transient_result = $this->connection->query(
        "SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND spamvalue = :value",
        [':key' => 'white-transient-haf', ':value' => $spammaster_ip]
      )->fetchField();
      $transient_whitelisted = !empty($transient_result);
    }

    // Only skip threat/buffer checks for transient whitelist.
    if ($transient_whitelisted) {
      return;
    }

    // IP is in threats buffer and user is anonymous, block.
    $rateLimitSeconds = 300;
    $buffered_ip = $this->connection->query(
      "SELECT threat FROM {spammaster_threats} WHERE threat = :ip",
      [':ip' => $spammaster_ip]
    )->fetchField();
    if (!empty($buffered_ip) && $this->currentUser->isAnonymous()) {
      $lastLogRow = $this->connection->query(
        "SELECT date, spamvalue FROM {spammaster_keys}
         WHERE spamkey = :key
           AND spamvalue LIKE :ip
         ORDER BY date DESC
         LIMIT 1",
        [
          ':key' => 'spammaster-firewall',
          ':ip' => '%Ip: ' . $spammaster_ip,
        ]
      )->fetchAssoc();
      $nowTimestamp = strtotime($spammaster_date);
      $log_message = NULL;
      if (empty($lastLogRow)) {
        // No previous firewall log for this IP.
        $log_message = 'Spam Master: firewall BUFFER BLOCK, Ip: ' . $spammaster_ip;
      }
      elseif (($nowTimestamp - strtotime($lastLogRow['date'])) > $rateLimitSeconds) {
        // Last log was more than 5 min ago, log rate limited DOS.
        $log_message = 'Spam Master: firewall BUFFER BLOCK (rate limited DOS), Ip: ' . $spammaster_ip;
      }
      if ($log_message) {
        $this->connection->insert('spammaster_keys')->fields([
          'date'      => $spammaster_date,
          'spamkey'   => 'spammaster-firewall',
          'spamvalue' => $log_message,
        ])->execute();
        $spammaster_total_block_count = $this->state->get('spammaster.total_block_count');
        $this->state->set(
          'spammaster.total_block_count',
          ($spammaster_total_block_count ?? 0) + 1
        );
      }
      // Use the firewall controller to get the render array.
      $event->setResponse($this->getFirewallBlockResponse(
        $request->getClientIp(),
        $request->headers->get('User-Agent'),
        'buffer'
      ));
      $event->stopPropagation();
      return;
    }

    // Collect service and honeypot logic.
    if ($form_id) {
      $is_collected = $this->collectService->spamMasterGetCollect($spammaster_ip);
      $spamcollection = Json::decode($is_collected);
      $spammasterip = $spamcollection['spammasterip'] ?? $spammaster_ip;
      $spammasteragent = $spamcollection['spammasteragent'] ?? '';
      $spammasterreferer = $spamcollection['spammasterreferer'] ?? '';
      $spammasterurl = $spamcollection['spammasterurl'] ?? '';
      $spammasteruserid = ($this->currentUser && method_exists($this->currentUser, 'id')) ? $this->currentUser->id() : 0;

      // Honeypot check.
      if (!empty($spammaster_extra_field_1) || !empty($spammaster_extra_field_2)) {
        $this->honeyService->spamMasterHoneypotCheck(
          $form_id,
          $spammasterip,
          $spammasteragent,
          $spammaster_extra_field_1 ?: 'empty',
          $spammaster_extra_field_2 ?: 'empty',
          $spammasterreferer,
          $spammasterurl,
          $spammasteruserid
        );
        // Use the firewall controller to get the render array.
        $event->setResponse($this->getFirewallBlockResponse(
          $request->getClientIp(),
          $request->headers->get('User-Agent'),
          'honeypot'
        ));
        $event->stopPropagation();
        return;
      }

      // Elusive check (advanced bot detection).
      if (in_array($basic_firewall_rules, ['1', '2'], TRUE)) {
        $isElusive = $this->elusiveService->spamMasterElusiveCheck($request->request);
        if ($isElusive === 'ELUSIVE') {
          // Use the firewall controller to get the render array.
          $event->setResponse($this->getFirewallBlockResponse(
            $request->getClientIp(),
            $request->headers->get('User-Agent'),
            'buffer'
          ));
          $event->stopPropagation();
          return;
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => ['checkForRedirection'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  private function getFirewallBlockResponse(string $ip, string $agent, string $reason = 'block'): Response {
    $ip_html = htmlspecialchars($ip, ENT_QUOTES, 'UTF-8');
    $agent_html = htmlspecialchars($agent, ENT_QUOTES, 'UTF-8');
    $main = '';
    switch ($reason) {
      case 'rate_limit':
        $main = "<pre>429 Too Many Requests</pre>
<pre>Your activity was temporarily limited due to high posting rate.</pre>";
        $http_code = 429;
        break;

      case 'honeypot':
        $main = "<pre>403 Forbidden</pre>
<pre>Automated or suspicious activity was detected.</pre>";
        $http_code = 403;
        break;

      case 'buffer':
        $main = "<pre>403 Forbidden</pre>
<pre>Your IP is blacklisted due to previous abuse or threat behavior.</pre>";
        $http_code = 403;
        break;

      default:
        $main = "<pre>403 Forbidden</pre>
<pre>Access denied by Spam Master rules.</pre>";
        $http_code = 403;
        break;
    }
    $html = $main .
    "<pre>IP: {$ip_html}</pre>
<pre>Browser: {$agent_html}</pre>
<pre><strong>Hint: Upgrade your browser to the latest version</strong></pre>
<pre>Protected by <a href=\"https://www.spammaster.org/contact/\" rel=\"noopener noreferrer\">Spam Master</a></pre>";
    return new Response($html, $http_code, ['Content-Type' => 'text/html; charset=utf-8']);
  }

}
