<?php

namespace Drupal\spammaster;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class service.
 */
class SpamMasterCollectService {
  /**
   * Fallback IP value when detection fails.
   *
   * @var string
   */
  protected const FALLBACK_IP = 'I 000';

  /**
   * Fallback hostname value when detection fails.
   *
   * @var string
   */
  protected const FALLBACK_HOSTNAME = 'H 000';

  /**
   * Symfony\Component\HttpFoundation\RequestStack definition.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The Config Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The form class.
   *
   * @var \Drupal\Core\Form\FormInterface
   */
  protected $collectnow;

  /**
   * {@inheritdoc}
   */
  public function __construct(RequestStack $requestStack, ConfigFactoryInterface $configFactory) {
    $this->requestStack = $requestStack;
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterGetCollect($collectnow) {

    $this->collectnow = $collectnow;
    $spammaster_settings_protection = $this->configFactory->get('spammaster.settings_protection');
    $basic_firewall_rules = $spammaster_settings_protection->get('spammaster.basic_firewall_rules');
    $request = $this->requestStack->getCurrentRequest();

    // Get remote IP, fallback if missing.
    $ip = $request->getClientIp();
    $spammasterip = substr(!empty($ip) ? $ip : self::FALLBACK_IP, 0, 48);

    // Get remote agent, fallback if missing.
    $userAgent = $request->headers->get('User-Agent');
    if ('1' === $basic_firewall_rules) {
      $spammasteragent = substr($userAgent ?? 'Sniffer', 0, 360);
    }
    else {
      $spammasteragent = substr('Relaxed - ' . ($userAgent ?? 'Sniffer'), 0, 360);
    }

    // Get remote referrer.
    $referer = $request->headers->get('Referer');
    $spammasterreferer = substr($referer ?? 'Direct', 0, 360);

    // Get destination url.
    $scheme = $request->getScheme() ?: 'https';
    $host = $request->getHost() ?: 'Hostname';
    $uri = $request->getRequestUri() ?: '/false';
    $spammasterurl = substr("{$scheme}://{$host}{$uri}", 0, 360);

    $prespamcollection = [
      'spammasterip'      => $spammasterip,
      'spammasteragent'   => $spammasteragent,
      'spammasterreferer' => $spammasterreferer,
      'spammasterurl'     => $spammasterurl,
    ];
    $spamcollection = Json::encode($prespamcollection);

    return $spamcollection;

  }

  /**
   * {@inheritdoc}
   */
  private function resolveIpFromHost($host) {
    if (empty($host)) {
      return FALSE;
    }
    $ip = gethostbyname($host);
    if ($ip && $ip !== '0') {
      return $ip;
    }
    if (function_exists('dns_get_record')) {
      $dns = dns_get_record($host, DNS_A);
      if (!empty($dns[0]['ip'])) {
        return $dns[0]['ip'];
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getIpAndHostname(string $site_url = ''): string {
    $ip = self::FALLBACK_IP;
    $hostname = self::FALLBACK_HOSTNAME;

    $request = $this->requestStack->getCurrentRequest();
    $server_name = $request ? $request->getHost() : ($_SERVER['SERVER_NAME'] ?? NULL);
    $server_addr = $_SERVER['SERVER_ADDR'] ?? NULL;

    if ($server_addr) {
      $ip = substr($server_addr, 0, 48);
      if (empty($ip) || $ip === '0' || $ip === '127.0.0.1') {
        $resolved_ip = $this->resolveIpFromHost($server_name);
        $ip = $resolved_ip ? substr($resolved_ip, 0, 48) : self::FALLBACK_IP;
      }
      $hostname = substr(@gethostbyaddr($server_addr), 0, 256);
      if (empty($hostname) || $hostname === '0' || $hostname === '127.0.0.1') {
        $resolved_host = $this->resolveIpFromHost($server_name);
        $hostname = $resolved_host ? substr($resolved_host, 0, 256) : self::FALLBACK_HOSTNAME;
      }
    }
    else {
      $resolved_ip = $this->resolveIpFromHost($server_name);
      $ip = $resolved_ip ? substr($resolved_ip, 0, 48) : self::FALLBACK_IP;
      $hostname = $resolved_ip ? substr($resolved_ip, 0, 256) : self::FALLBACK_HOSTNAME;
    }
    $address = $this->requestStack->getCurrentRequest()->getHost() ?? 'unknown';

    $iphostVar = [
      'ip'       => $ip,
      'hostname' => $hostname,
      'address'  => $address,
    ];
    $serverCollection = Json::encode($iphostVar);

    return $serverCollection;
  }

  /**
   * Extracts the first valid email address from a string.
   *
   * @param string $subject
   *   The input string to scan for an email address.
   *
   * @return string
   *   The first valid email if found, or a dummy fallback.
   */
  public function extractFirstEmail(string $subject): string {
    $regex = '/(?:[a-z0-9!#$%&\'*+\/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&\'*+\/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])/i';
    preg_match($regex, $subject, $matches);
    if (!empty($matches[0]) && filter_var($matches[0], FILTER_VALIDATE_EMAIL)) {
      return substr($matches[0], 0, 256);
    }
    return 'haf@' . rand(10000000, 99999999) . '.drup';
  }

}
