<?php

namespace Drupal\spammaster;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class service.
 */
class SpamMasterElusiveService {

  /**
   * The database connection object.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * Symfony\Component\HttpFoundation\RequestStack definition.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The Config Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The Time Service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The SpamMasterCollectService Service.
   *
   * @var \Drupal\spammaster\SpamMasterCollectService
   */
  protected $collectService;

  /**
   * The SpamMasterUserService Service.
   *
   * @var \Drupal\spammaster\SpamMasterUserService
   */
  protected $userService;

  /**
   * The SpamMasterApiService Service.
   *
   * @var \Drupal\spammaster\SpamMasterApiService
   */
  protected $apiService;

  /**
   * {@inheritdoc}
   */
  public function __construct(Connection $connection, RequestStack $requestStack, ConfigFactoryInterface $configFactory, StateInterface $state, AccountProxyInterface $currentUser, TimeInterface $time, SpamMasterCollectService $collectService, SpamMasterUserService $userService, SpamMasterApiService $apiService) {
    $this->connection = $connection;
    $this->requestStack = $requestStack;
    $this->configFactory = $configFactory;
    $this->state = $state;
    $this->currentUser = $currentUser;
    $this->time = $time;
    $this->collectService = $collectService;
    $this->userService = $userService;
    $this->apiService = $apiService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('current_user'),
      $container->get('datetime.time'),
      $container->get('spammaster.collect_service'),
      $container->get('spammaster.user_service'),
      $container->get('spammaster.api_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getSpamUser() {
    return ($this->currentUser && method_exists($this->currentUser, 'id')) ?
      $this->userService->spamMasterGetUser($this->currentUser->id()) : 0;
  }

  /**
   * {@inheritdoc}
   */
  protected function buildApiParams($license, $collection, $user, $type, $email, $content, $host = []) {
    return [
      'blog_license_key' => $license,
      'blog_threat_ip' => $collection['spammasterip'] ?? '',
      'blog_threat_user' => $user,
      'blog_threat_type' => $type,
      'blog_threat_email' => $email,
      'blog_threat_content' => $content,
      'blog_threat_agent' => $collection['spammasteragent'] ?? '',
      'blog_threat_refe' => $collection['spammasterreferer'] ?? '',
      'blog_threat_dest' => $collection['spammasterurl'] ?? '',
      'blog_web_adress' => $host['address'] ?? '',
      'blog_server_ip' => $host['ip'] ?? '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function getExemptValue($spamkey, $search) {
    return $this->connection->query(
      "SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :search) > :value",
      [
        ':key' => $spamkey,
        ':search' => $search,
        ':value' => 0,
      ]
    )->fetchField();
  }

  /**
   * {@inheritdoc}
   */
  protected function isExcludedUrl($url) {
    $exclude_substrings = [
      'autodiscover.xml',
      'recipebox',
      'api/recipe-box',
      '/take/2',
      '/take/3',
      '/take/4',
      '/take/5',
      '/take/6',
      '/take/7',
      '/take/8',
      '/take/9',
      '/take/10',
      '/take/11',
      '/take/12',
      '/take/13',
      '/take/14',
      '/take/15',
      '/take/16',
      '/take/17',
      '/take/18',
      '/take/19',
    ];
    foreach ($exclude_substrings as $exclusion) {
      if (stripos($url, $exclusion) !== FALSE) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  protected function incrementBlockCount($state_key = 'spammaster.total_block_count') {
    $count = $this->state->get($state_key);
    $count++;
    $this->state->set($state_key, $count);
    return $count;
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterElusiveCheck($spamelusive): string {
    $spammaster_date = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_license = $spammaster_settings->get('spammaster.license_key');
    $spammaster_subtype = $spammaster_settings->get('spammaster.subtype');
    $spammaster_status = $this->state->get('spammaster.license_status');
    $valid_statuses = ['VALID', 'MALFUNCTION_1', 'MALFUNCTION_2'];

    if (in_array($spammaster_status, $valid_statuses, TRUE) && $spammaster_subtype === 'prod') {
      $collectnow = $this->requestStack->getCurrentRequest()->getClientIp();
      $spammaster_collect_service = $this->collectService;
      $is_collected = $spammaster_collect_service->spamMasterGetCollect($collectnow);
      $spamCollection = Json::decode($is_collected);
      $is_collectedIpHost = $spammaster_collect_service->getIpAndHostname($spamCollection['spammasterurl']);
      $spamCollectionIpHost = Json::decode($is_collectedIpHost);

      $is_user = $this->getSpamUser();
      $spamUserData = Json::decode($is_user);

      $spammasterIdForm = $spamelusive->get('form_id') ?? FALSE;

      if (!empty($spamCollection['spammasterurl']) && $this->isExcludedUrl($spamCollection['spammasterurl'])) {
        return 'BAIL';
      }

      // phpcs:ignore DrupalPractice.Variables.GetRequestData.SuperglobalAccessed
      $spampost = $_POST;
      if (!empty($spampost) && is_array($spampost)) {
        if (
          preg_match('/[^a-zA-Z]system_/', $spammasterIdForm) !== 0 &&
          preg_match('/[^a-zA-Z]search_/', $spammasterIdForm) !== 0 &&
          preg_match('/[^a-zA-Z]views_exposed_form_/', $spammasterIdForm) !== 0
        ) {
          return 'BAIL';
        }
        if (count($spampost) === 1 && array_key_exists('js', $spampost) && $spampost['js'] === 'true') {
          return 'BAIL';
        }
        if ($this->getExemptValue('exempt-needle', $spammasterIdForm)) {
          return 'BAIL';
        }
        $spampoststr = str_replace('=', ' ', urldecode(http_build_query($spampost, '', ' ')));
        if ($this->getExemptValue('exempt-key', $spampoststr)) {
          return 'BAIL';
        }
        if ($this->getExemptValue('exempt-value', $spampoststr)) {
          return 'BAIL';
        }
        $blog_threat_type = 'HAF';
        $result_post_content_json  = Json::encode($spampost);
        $result_post_content_trim  = substr($result_post_content_json, 0, 963);
        $result_post_content_clean = strip_tags($result_post_content_trim);
        if (empty($result_post_content_clean)) {
          $result_post_content_clean = 'is_elusive_d';
        }
        $spammasteremail = $spammaster_collect_service->extractFirstEmail($spampoststr);

        $spam_master_leaning_url = 'https://www.spammaster.org/core/learn/get_learn_haf.php';
        $params = $this->buildApiParams($spammaster_license, $spamCollection, $is_user, $blog_threat_type, $spammasteremail, 'is_elusive_d - ' . $result_post_content_clean, $spamCollectionIpHost);

        $response = $this->apiService->postScanData($spam_master_leaning_url, $params, TRUE);

        if (empty($response['threat'])) {
          $spammasterWhiteTransient = $this->state->get('spammaster.white_transient');
          if ('1' === $spammasterWhiteTransient) {
            $this->connection->insert('spammaster_keys')->fields([
              'date' => $spammaster_date,
              'spamkey' => 'white-transient-haf',
              'spamvalue' => $spamCollection['spammasterip'],
            ])->execute();
          }
          return 'BAIL';
        }
        else {
          $this->connection->insert('spammaster_threats')->fields([
            'date' => $spammaster_date,
            'threat' => $spamCollection['spammasterip'],
          ])->execute();
          $this->incrementBlockCount();
          $this->connection->insert('spammaster_keys')->fields([
            'date' => $spammaster_date,
            'spamkey' => 'spammaster-firewall',
            'spamvalue' => 'Spam Master: firewall RBL BLOCK, Agent: ' . $spamCollection['spammasteragent'] . ', User ID: ' . $spamUserData['ID'] . ', Username: ' . $spamUserData['username'] . ', Ip: ' . $spamCollection['spammasterip'],
          ])->execute();
          return 'ELUSIVE';
        }
      }
      else {
        return 'BAIL';
      }
    }
    else {
      return 'BAIL';
    }
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterIsNeedle($spamFormId) {
    $exemptions = [
      ['exempt-needle-straw', 'straw', 'STRAW'],
      ['exempt-needle-sig-hide', 'sighide', 'SIGHIDDEN'],
      ['exempt-needle-sig-show', 'sigshow', 'SIGSHOW'],
      ['exempt-needle', 'exclude', 'NEEDLE'],
    ];
    foreach ($exemptions as $exemption) {
      $value = $this->connection->query("SELECT spamvalue FROM {spammaster_keys} WHERE spamkey = :key AND POSITION(spamvalue IN :str) > :value", [
        ':key' => $exemption[0],
        ':str' => $spamFormId,
        ':value' => 0,
      ])->fetchField();
      if (!empty($value)) {
        return $exemption[2];
      }
    }
    return "STRAW";
  }

  /**
   * {@inheritdoc}
   */
  public function spamMasterFloodCheck($spamelusive): string {
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_license = $spammaster_settings->get('spammaster.license_key');
    $spammaster_subtype = $spammaster_settings->get('spammaster.subtype');
    $spammaster_status = $this->state->get('spammaster.license_status');
    $valid_statuses = ['VALID', 'MALFUNCTION_1', 'MALFUNCTION_2'];

    if (in_array($spammaster_status, $valid_statuses, TRUE) && $spammaster_subtype === 'prod') {
      $collectnow = $this->requestStack->getCurrentRequest()->getClientIp();
      $spammaster_collect_service = $this->collectService;
      $is_collected = $spammaster_collect_service->spamMasterGetCollect($collectnow);
      $spamCollection = Json::decode($is_collected) ?: [];

      $is_collectedIpHost = $spammaster_collect_service->getIpAndHostname(!empty($spamCollection['spammasterurl']) ? $spamCollection['spammasterurl'] : '');
      $spamCollectionIpHost = Json::decode($is_collectedIpHost) ?: [];

      $is_user = $this->getSpamUser();

      // phpcs:ignore DrupalPractice.Variables.GetRequestData.SuperglobalAccessed
      $spampost = $_POST;

      if (!is_array($spampost) || empty($spampost)) {
        $result_post_content_clean = 'is_elusive_d';
      }
      else {
        $result_post_content_json  = Json::encode($spampost);
        $result_post_content_trim  = substr($result_post_content_json, 0, 963);
        $result_post_content_clean = strip_tags($result_post_content_trim);
        if (empty($result_post_content_clean)) {
          $result_post_content_clean = 'is_flood_d';
        }
      }
      $blog_threat_type = 'HAF';

      $this->incrementBlockCount();

      $spampoststr = str_replace('=', ' ', urldecode(http_build_query($spampost, '', ' ')));
      $spammasteremail = $spammaster_collect_service->extractFirstEmail($spampoststr);

      $spam_master_leaning_url = 'https://www.spammaster.org/core/learn/get_learn_haf.php';

      $params = $this->buildApiParams($spammaster_license, $spamCollection, $is_user, $blog_threat_type, $spammasteremail, 'is_flood_d - ' . $result_post_content_clean, $spamCollectionIpHost);

      $response = $this->apiService->postScanData($spam_master_leaning_url, $params, TRUE);

      if (empty($response['threat'])) {
        return 'BAIL';
      }
      else {
        return 'FLOOD';
      }
    }
    return 'BAIL';
  }

}
