<?php

namespace Drupal\spammaster;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Refactored SpamMasterMailService for Drupal PSR compliance.
 */
class SpamMasterMailService {
  use StringTranslationTrait;

  /**
   * The database connection object.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The AccountProxyInterface service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The Config Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The Time Service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The Mail Service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    Connection $connection,
    MessengerInterface $messenger,
    AccountProxyInterface $currentUser,
    ConfigFactoryInterface $configFactory,
    StateInterface $state,
    TimeInterface $time,
    MailManagerInterface $mailManager
  ) {
    $this->connection = $connection;
    $this->messenger = $messenger;
    $this->currentUser = $currentUser;
    $this->configFactory = $configFactory;
    $this->state = $state;
    $this->time = $time;
    $this->mailManager = $mailManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('messenger'),
      $container->get('current_user'),
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('datetime.time'),
      $container->get('plugin.manager.mail')
    );
  }

  /**
   * Helper to get recipient email.
   */
  protected function getRecipient($attached, $fallback_email) {
    if (empty($attached) || is_array($attached)) {
      return $fallback_email;
    }
    return $attached;
  }

  /**
   * Helper to send mail.
   */
  protected function sendMail($key, $to, $params) {
    $module = 'spammaster';
    $langcode = $this->currentUser->getPreferredLangcode();
    $send = TRUE;
    $this->mailManager->mail($module, $key, $to, $langcode, $params, NULL, $send);
  }

  /**
   * Helper to log mail actions.
   */
  protected function logMail($type, $to, $extra = '') {
    $date = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $this->connection->insert('spammaster_keys')->fields([
      'date' => $date,
      'spamkey' => 'spammaster-mail',
      'spamvalue' => "Spam Master: $type $to" . ($extra ? " $extra" : ''),
    ])->execute();
  }

  /**
   * Helper for license mail logs.
   */
  protected function logLicense($type, $extra = '') {
    $date = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $this->connection->insert('spammaster_keys')->fields([
      'date' => $date,
      'spamkey' => 'spammaster-license',
      'spamvalue' => "Spam Master: $type" . ($extra ? " $extra" : ''),
    ])->execute();
  }

  /**
   * Helper to map status to warning/signature.
   */
  protected function getWarningSignature($status) {
    $map = [
      'INACTIVE' => [
        'warning' => 'Warning: Spam Master is Inactive, you are not protected.',
        'signature' => 'Please correct the warnings.',
      ],
      'VALID' => [
        'warning' => 'Your license status is Valid & Online.',
        'signature' => 'All is good.',
      ],
      'MALFUNCTION_1' => [
        'warning' => 'Warnings: Malfunction 1, please update Spam Master to the latest version.',
        'signature' => 'Please correct the warnings.',
      ],
      'MALFUNCTION_2' => [
        'warning' => 'Warnings: Malfunction 2, you are using the same license key in more than one website.',
        'signature' => 'Please correct the warnings.',
      ],
      'MALFUNCTION_3' => [
        'warning' => 'Warning: Malfunction 3, get in touch with Spam Master support.',
        'signature' => 'Please correct the warnings.',
      ],
      'MALFUNCTION_4' => [
        'warning' => 'Warning: Not able to automatically Generate a FREE Connection Key on your server, email already in use.',
        'signature' => 'Please correct the warnings.',
      ],
      'MALFUNCTION_5' => [
        'warning' => 'Warning: Not able to Generate a FREE Connection Key because the daily limit of free keys was exceeded.',
        'signature' => 'Please correct the warnings.',
      ],
      'MALFUNCTION_6' => [
        'warning' => 'Warning: Not able to connect to the online RBL servers. Key already use in another website.',
        'signature' => 'Please correct the warnings.',
      ],
      'MALFUNCTION_7' => [
        'warning' => 'Warning: Not able to connect to the online RBL servers because your Spam Master installed version is out of date.',
        'signature' => 'Please correct the warnings.',
      ],
      'MALFUNCTION_8' => [
        'warning' => 'Warning: We have detected CDN, WAF that masks the end users IP addresses. Please read Spam Master online documentation to solve the issue.',
        'signature' => 'Please correct the warnings.',
      ],
      'UNSTABLE' => [
        'warning' => 'Warning: Free RBL Server connection is Unstable, please consider a Pro Key.',
        'signature' => 'Please correct the warnings.',
      ],
      'HIGH_VOLUME' => [
        'warning' => 'Warning: High Volume of spam checks, please consider a Pro Key.',
        'signature' => 'Please correct the warnings.',
      ],
      'DISCONNECTED' => [
        'warning' => 'Warning: Get in touch with Spam Master support.',
        'signature' => 'Please correct the warnings.',
      ],
      'EXPIRED' => [
        'warning' => 'Warning: your license is EXPIRED and you are not protected.',
        'signature' => 'Please correct the warnings.',
      ],
      'CODE_TAMPER_1' => [
        'warning' => 'Warning: Altered code detected. You are not protected… programmatic changes may affect Spam Master functionality and, in worse case scenarios may facilitate malicious activities like DoS attacks or injection exploits.',
        'signature' => 'Please correct the warnings.',
      ],
      'CODE_TAMPER_2' => [
        'warning' => 'Warning: Altered code detected. You are not protected… programmatic changes may affect Spam Master functionality and, in worse case scenarios may facilitate malicious activities like DoS attacks or injection exploits.',
        'signature' => 'Please correct the warnings.',
      ],
    ];
    return $map[$status] ?? ['warning' => '', 'signature' => ''];
  }

  /**
   * Helper to convert alert level to readable value.
   */
  protected function getAlertLevelDeconstructed($level) {
    if (!isset($level) || empty($level)) {
      return 'empty';
    }
    if (preg_match('/^ALERT_(\d)$/', $level, $matches)) {
      return $matches[1];
    }
    return $level;
  }

  /**
   * Helper to format block count results.
   */
  protected function getBlockCountResult($count, $label = 'Total Blocks') {
    if (!isset($count) || empty($count)) {
      return "$label: empty";
    }
    if ($count <= 10) {
      return "$label: good, less than 10";
    }
    if ($count > 10) {
      return "$label: " . number_format($count);
    }
    return "$label: $count";
  }

  /**
   * License trial creation mail.
   */
  public function spamMasterLicGenCreation() {
    $key = 'license_trial_create';
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_site_name = $site_settings->get('name');
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_license = $spammaster_settings->get('spammaster.license_key');
    $spammaster_status = $this->state->get('spammaster.license_status');
    $spammaster_license_protection = $this->state->get('spammaster.license_protection');
    $spammaster_attached = $this->state->get('spammaster.spam_master_attached');
    $to = $this->getRecipient($spammaster_attached, $this->currentUser->getEmail());

    if ('VALID' === $spammaster_status) {
      $spam_master_table_content = 'Congratulations, ' . $spammaster_site_name . ' is now protected by Spam Master against millions of threats.' . "\r\n\r\n";
      $spam_master_table_content .= 'Your License is: ' . $spammaster_license . '.' . "\r\n\r\n";
      $spam_master_table_content .= 'Protected Against: ' . number_format($spammaster_license_protection) . ' million threats.' . "\r\n\r\n";
      $spam_master_table_content .= 'Enjoy,' . "\r\n" . 'Spam Master Team' . "\r\n";
      $params['message'] = $spam_master_table_content;
      $this->sendMail($key, $to, $params);

      $this->messenger->addMessage($this->t('Remember to visit Spam Master configuration page.'));

      $this->logLicense('congratulations! free license created.');
      $this->logMail('free license created email sent To:', $to);
    }
    else {
      $this->messenger->addError($this->t('Spam Master Free license could not be created. License status is: @spammaster_status. Check Spam Master configuration page and read more about statuses.', ['@spammaster_status' => $spammaster_status]));
      $this->logLicense('free license not created, contains malfunction:', $spammaster_status);
      $this->logMail('email not sent, license contains malfunction', $to, $spammaster_status);
    }
  }

  /**
   * License expired mail.
   */
  public function spamMasterLicExpired() {
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_site_name = $site_settings->get('name');
    $spammaster_attached = $this->state->get('spammaster.spam_master_attached');
    $to = $this->getRecipient($spammaster_attached, $site_settings->get('mail'));
    $spammaster_type = $this->state->get('spammaster.type');

    if ('TRIAL' === $spammaster_type || 'FULL' === $spammaster_type) {
      $key = ('TRIAL' === $spammaster_type) ? 'license_trial_end' : 'license_full_end';
      $spam_master_table_content = $spammaster_site_name . ' is no longer protected by Spam Master against millions of threats.' . "\r\n\r\n";
      $spam_master_table_content .= ('TRIAL' === $spammaster_type) ?
        'If you enjoyed the protection you can quickly get a full license, it costs peanuts per year.' . "\r\n" .
        'Go to Spam Master settings page and click get full license.' . "\r\n\r\n" :
        'Hope you have enjoyed 1 year of bombastic protection. You can quickly get another license and get protected again, it costs peanuts per year.' . "\r\n" .
        'Go to Spam Master settings page and click get full license.' . "\r\n\r\n";
      $spam_master_table_content .= 'Thanks.' . "\r\n" . 'Spam Master Team' . "\r\n";
      $params['message'] = $spam_master_table_content;
      $this->sendMail($key, $to, $params);

      $this->logLicense(($spammaster_type === 'TRIAL') ? 'trial license expired.' : 'full license expired.');
      $this->logMail(($spammaster_type === 'TRIAL') ? 'mail trial license expired sent To:' : 'mail full license expired sent To:', $to);
    }
  }

  /**
   * License malfunction mail.
   */
  public function spamMasterLicMalfunctions() {
    $key = 'license_malfunction';
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_site_name = $site_settings->get('name');
    $spammaster_attached = $this->state->get('spammaster.spam_master_attached');
    $to = $this->getRecipient($spammaster_attached, $site_settings->get('mail'));
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_license = $spammaster_settings->get('spammaster.license_key');
    $spammaster_status = $this->state->get('spammaster.license_status');

    $spam_master_table_content = 'Warning, your ' . $spammaster_site_name . ' might not be 100% protected.' . "\r\n\r\n";
    $spam_master_table_content .= 'Your License: ' . $spammaster_license . ' status is: ' . $spammaster_status . '.' . "\r\n\r\n";
    $spam_master_table_content .= 'Some license status are easy to fix, example Malfunction_1 just means you need to update the module to the latest version and the status will automatically fix itself.' . "\r\n";
    $spam_master_table_content .= 'All statuses are explained in www.spammaster.org, About menu, Documentation page and, in case of trouble get in touch with our support.' . "\r\n\r\n";
    $spam_master_table_content .= 'Spam Master Team' . "\r\n";
    $params['message'] = $spam_master_table_content;
    $this->sendMail($key, $to, $params);

    $this->logLicense('license malfunction detected.');
    $this->logMail('license malfunction email sent To:', $to);
  }

  /**
   * Alert level 3 mail.
   */
  public function spamMasterLicAlertLevel3() {
    $key = 'lic_alert_level_3';
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_site_name = $site_settings->get('name');
    $spammaster_attached = $this->state->get('spammaster.spam_master_attached');
    $to = $this->getRecipient($spammaster_attached, $site_settings->get('mail'));
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_license = $spammaster_settings->get('spammaster.license_key');
    $spammaster_status = $this->state->get('spammaster.license_status');
    $spammaster_license_protection = $this->state->get('spammaster.license_protection');

    $spam_master_table_content = 'Warning!!! Spam Master Alert 3 detected for ' . $spammaster_site_name . '.' . "\r\n\r\n";
    $spam_master_table_content .= 'Your License: ' . $spammaster_license . ' status is: ' . $spammaster_status . ' and you are protected against: ' . number_format($spammaster_license_protection) . ' threats.' . "\r\n\r\n";
    $spam_master_table_content .= 'The daily Alert 3 email will automatically stop when your website alert level drops to safer levels.' . "\r\n";
    $spam_master_table_content .= 'All alert levels are explained in our website documentation section and, in case of trouble get in touch with our support.' . "\r\n\r\n";
    $spam_master_table_content .= 'Spam Master Team' . "\r\n";
    $params['message'] = $spam_master_table_content;
    $this->sendMail($key, $to, $params);

    $this->logLicense('alert level 3 detected.');
    $this->logMail('alert level 3 email sent To:', $to);
  }

  /**
   * Daily report mail.
   */
  public function spamMasterMailDailyReport() {
    $key = 'mail_daily_report';
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_site_name = $site_settings->get('name');
    $spammaster_attached = $this->state->get('spammaster.spam_master_attached');
    $to = $this->getRecipient($spammaster_attached, $site_settings->get('mail'));
    $spammaster_status = $this->state->get('spammaster.license_status');
    $sig = $this->getWarningSignature($spammaster_status);
    $spammaster_license_protection = $this->state->get('spammaster.license_protection');
    $spammaster_license_probability = $this->state->get('spammaster.license_probability');
    $spammaster_license_alert_level = $this->state->get('spammaster.license_alert_level');
    $alert_level = $this->getAlertLevelDeconstructed($spammaster_license_alert_level);
    $spammaster_total_block_count = $this->state->get('spammaster.total_block_count');
    $total_block_count_result = $this->getBlockCountResult($spammaster_total_block_count, 'Firewall Total Triggers');
    // Get count last 1 day of blocks from spammaster_keys.
    $time = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $time_expires = date('Y-m-d H:i:s', strtotime($time . '-1 days'));
    $spammaster_spam_watch_query = $this->connection->select('spammaster_keys', 'u');
    $spammaster_spam_watch_query->fields('u', ['spamkey']);
    $spammaster_spam_watch_query->where('(date BETWEEN :time_expires AND :time)', [
      ':time_expires' => $time_expires,
      ':time' => $time,
    ]);
    $spammaster_spam_watch_query->where('(spamkey = :registration OR spamkey = :comment OR spamkey = :contact OR spamkey = :firewall)', [
      ':registration' => 'spammaster-registration',
      ':comment' => 'spammaster-comment',
      'contact' => 'spammaster-contact',
      ':firewall' => 'spammaster-firewall',
    ]);
    $spammaster_spam_watch_result = $spammaster_spam_watch_query->countQuery()->execute()->fetchField();
    $block_count_result = empty($spammaster_spam_watch_result)
      ? 'Firewall Weekly Triggers: good, nothing to report'
      : 'Firewall Weekly Triggers: ' . number_format($spammaster_spam_watch_result);

    $spam_master_table_content = 'Spam Master Daily Report for ' . $spammaster_site_name . '.' . "\r\n\r\n";
    $spam_master_table_content .= $spammaster_status . ' - ' . $sig['warning'] . "\r\n\r\n";
    $spam_master_table_content .= 'Alert Level: ' . $alert_level . "\r\n";
    $spam_master_table_content .= 'Spam Probability: ' . $spammaster_license_probability . '%' . "\r\n";
    $spam_master_table_content .= 'Protected Against: ' . number_format($spammaster_license_protection) . ' million threats' . "\r\n";
    $spam_master_table_content .= $total_block_count_result . "\r\n";
    $spam_master_table_content .= $block_count_result . "\r\n\r\n";
    $spam_master_table_content .= $sig['signature'] . "\r\n\r\n";
    $spam_master_table_content .= 'The daily report email can be turned off in Spam Master module settings page, Emails & Reporting section.' . "\r\n\r\n";
    $spam_master_table_content .= 'Spam Master Team';
    $params['message'] = $spam_master_table_content;
    $this->sendMail($key, $to, $params);

    $this->logMail('daily email sent To:', $to);
  }

  /**
   * Weekly report mail.
   */
  public function spamMasterMailWeeklyReport() {
    $key = 'mail_weekly_report';
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_site_name = $site_settings->get('name');
    $spammaster_attached = $this->state->get('spammaster.spam_master_attached');
    $to = $this->getRecipient($spammaster_attached, $site_settings->get('mail'));
    $spammaster_status = $this->state->get('spammaster.license_status');
    $sig = $this->getWarningSignature($spammaster_status);
    $spammaster_license_protection = $this->state->get('spammaster.license_protection');
    $spammaster_license_probability = $this->state->get('spammaster.license_probability');
    $spammaster_license_alert_level = $this->state->get('spammaster.license_alert_level');
    $alert_level = $this->getAlertLevelDeconstructed($spammaster_license_alert_level);
    $spammaster_total_block_count = $this->state->get('spammaster.total_block_count');
    $total_block_count_result = $this->getBlockCountResult($spammaster_total_block_count, 'Total Blocks');
    // Set 7 days time.
    $time = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $time_expires = date('Y-m-d H:i:s', strtotime($time . '-7 days'));
    // Get count last 7 days of blocks from spammaster_keys.
    $spammaster_spam_watch_query = $this->connection->select('spammaster_keys', 'u');
    $spammaster_spam_watch_query->fields('u', ['spamkey']);
    $spammaster_spam_watch_query->where('(date BETWEEN :time_expires AND :time)', [
      ':time_expires' => $time_expires,
      ':time' => $time,
    ]);
    $spammaster_spam_watch_query->where('(spamkey = :firewall OR spamkey = :honeypot)', [
      ':firewall' => 'spammaster-firewall',
      ':honeypot' => 'spammaster-honeypot',
    ]);
    $spammaster_spam_watch_result = $spammaster_spam_watch_query->countQuery()->execute()->fetchField();
    $block_count_result = empty($spammaster_spam_watch_result)
      ? 'Combined Weekly Triggers: good, nothing to report'
      : 'Combined Weekly Triggers: ' . number_format($spammaster_spam_watch_result);

    $spammaster_buffer_size = $this->connection->select('spammaster_threats', 'u');
    $spammaster_buffer_size->fields('u', ['threat']);
    $spammaster_buffer_size_result = $spammaster_buffer_size->countQuery()->execute()->fetchField();
    $spammaster_buffer_size_result_count = empty($spammaster_buffer_size_result) ? '1' : $spammaster_buffer_size_result;
    $spammaster_white_size = $this->connection->select('spammaster_white', 'u');
    $spammaster_white_size->fields('u', ['white_']);
    $spammaster_white_size_result = $spammaster_white_size->countQuery()->execute()->fetchField();
    $spammaster_white_size_result_count = empty($spammaster_white_size_result) ? '0' : $spammaster_white_size_result;
    // Get count last 7 days of firewall from spammaster_keys.
    $spammaster_firewall_size = $this->connection->select('spammaster_keys', 'u');
    $spammaster_firewall_size->fields('u', ['spamkey']);
    $spammaster_firewall_size->where('(date BETWEEN :time_expires AND :time)', [
      ':time_expires' => $time_expires,
      ':time' => $time,
    ]);
    $spammaster_firewall_size->where('(spamkey = :firewall)', [':firewall' => 'spammaster-firewall']);
    $spammaster_firewall_size_result = $spammaster_firewall_size->countQuery()->execute()->fetchField();
    $firewall_count_result = empty($spammaster_firewall_size_result)
      ? 'Firewall Weekly Triggers: 0'
      : 'Firewall Weekly Triggers: ' . number_format($spammaster_firewall_size_result);
    // Get count last 7 days of honeypot from spammaster_keys.
    $spammaster_honeypot_size = $this->connection->select('spammaster_keys', 'u');
    $spammaster_honeypot_size->fields('u', ['spamkey']);
    $spammaster_honeypot_size->where('(date BETWEEN :time_expires AND :time)', [
      ':time_expires' => $time_expires,
      ':time' => $time,
    ]);
    $spammaster_honeypot_size->where('(spamkey = :honeypot)', [':honeypot' => 'spammaster-honeypot']);
    $spammaster_honeypot_size_result = $spammaster_honeypot_size->countQuery()->execute()->fetchField();
    $honeypot_count_result = empty($spammaster_honeypot_size_result)
      ? 'Honeypot Weekly Trigers: 0'
      : 'Honeypot Weekly Trigers: ' . number_format($spammaster_honeypot_size_result);

    $spam_master_table_content = 'Spam Master weekly report for ' . $spammaster_site_name . '.' . "\r\n\r\n";
    $spam_master_table_content .= $spammaster_status . ' - ' . $sig['warning'] . "\r\n";
    $spam_master_table_content .= 'Alert Level: ' . $alert_level . "\r\n";
    $spam_master_table_content .= 'Spam Probability: ' . $spammaster_license_probability . '%' . "\r\n";
    $spam_master_table_content .= 'Protected Against: ' . number_format($spammaster_license_protection) . ' million threats' . "\r\n";
    $spam_master_table_content .= 'Spam Buffer Size: ' . number_format($spammaster_buffer_size_result_count) . "\r\n";
    $spam_master_table_content .= 'Spam Whitelist Size: ' . number_format($spammaster_white_size_result_count) . "\r\n";
    $spam_master_table_content .= $total_block_count_result . "\r\n";
    $spam_master_table_content .= $block_count_result . "\r\n";
    $spam_master_table_content .= $firewall_count_result . "\r\n";
    $spam_master_table_content .= $honeypot_count_result . "\r\n\r\n";
    $spam_master_table_content .= $sig['signature'] . "\r\n\r\n";
    $spam_master_table_content .= 'The weekly report email can be turned off in Spam Master module settings page, Emails & Reporting section.' . "\r\n\r\n";
    $spam_master_table_content .= 'See you next week!' . "\r\n";
    $spam_master_table_content .= 'Spam Master Team' . "\r\n";
    $params['message'] = $spam_master_table_content;
    $this->sendMail($key, $to, $params);

    $this->logMail('weekly email sent To:', $to);
  }

  /**
   * Help report mail.
   */
  public function spamMasterMailHelpReport() {
    $key = 'mail_help_report';
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_site_name = $site_settings->get('name');
    $to = 'stats@spammaster.org';
    $spammaster_version = $this->state->get('spammaster.version');
    $spammaster_platform_version = \Drupal::VERSION;
    $spammaster_license_protection = $this->state->get('spammaster.license_protection');
    $spammaster_license_probability = $this->state->get('spammaster.license_probability');
    $spammaster_license_alert_level = $this->state->get('spammaster.license_alert_level');
    $spammaster_status = $this->state->get('spammaster.license_status');
    $spammaster_settings = $this->configFactory->get('spammaster.settings');
    $spammaster_license = $spammaster_settings->get('spammaster.license_key');
    $alert_level = $this->getAlertLevelDeconstructed($spammaster_license_alert_level);
    $spammaster_total_block_count = $this->state->get('spammaster.total_block_count');
    $total_block_count_result = $this->getBlockCountResult($spammaster_total_block_count, 'Total Blocks');
    $time = date('Y-m-d H:i:s', $this->time->getCurrentTime());
    $time_expires = date('Y-m-d H:i:s', strtotime($time . '-7 days'));

    // Weekly triggers.
    $spammaster_spam_watch_query = $this->connection->select('spammaster_keys', 'u');
    $spammaster_spam_watch_query->fields('u', ['spamkey']);
    $spammaster_spam_watch_query->where('(date BETWEEN :time_expires AND :time)', [
      ':time_expires' => $time_expires,
      ':time' => $time,
    ]);
    $spammaster_spam_watch_query->where('(spamkey = :firewall OR spamkey = :honeypot)', [
      ':firewall' => 'spammaster-firewall',
      ':honeypot' => 'spammaster-honeypot',
    ]);
    $spammaster_spam_watch_result = $spammaster_spam_watch_query->countQuery()->execute()->fetchField();
    $block_count_result = empty($spammaster_spam_watch_result)
      ? 'Combined Weekly Triggers: good, nothing to report'
      : 'Combined Weekly Triggers: ' . number_format($spammaster_spam_watch_result);

    $spammaster_buffer_size = $this->connection->select('spammaster_threats', 'u');
    $spammaster_buffer_size->fields('u', ['threat']);
    $spammaster_buffer_size_result = $spammaster_buffer_size->countQuery()->execute()->fetchField();
    $spammaster_buffer_size_result_count = empty($spammaster_buffer_size_result) ? '1' : $spammaster_buffer_size_result;
    $spammaster_white_size = $this->connection->select('spammaster_white', 'u');
    $spammaster_white_size->fields('u', ['white_']);
    $spammaster_white_size_result = $spammaster_white_size->countQuery()->execute()->fetchField();
    $spammaster_white_size_result_count = empty($spammaster_white_size_result) ? '0' : $spammaster_white_size_result;
    $spammaster_firewall_size = $this->connection->select('spammaster_keys', 'u');
    $spammaster_firewall_size->fields('u', ['spamkey']);
    $spammaster_firewall_size->where('(date BETWEEN :time_expires AND :time)', [
      ':time_expires' => $time_expires,
      ':time' => $time,
    ]);
    $spammaster_firewall_size->where('(spamkey = :firewall)', [':firewall' => 'spammaster-firewall']);
    $spammaster_firewall_size_result = $spammaster_firewall_size->countQuery()->execute()->fetchField();
    $firewall_count_result = empty($spammaster_firewall_size_result)
      ? 'Firewall Weekly Triggers: 0'
      : 'Firewall Weekly Triggers: ' . number_format($spammaster_firewall_size_result);
    $spammaster_honeypot_size = $this->connection->select('spammaster_keys', 'u');
    $spammaster_honeypot_size->fields('u', ['spamkey']);
    $spammaster_honeypot_size->where('(date BETWEEN :time_expires AND :time)', [
      ':time_expires' => $time_expires,
      ':time' => $time,
    ]);
    $spammaster_honeypot_size->where('(spamkey = :honeypot)', [':honeypot' => 'spammaster-honeypot']);
    $spammaster_honeypot_size_result = $spammaster_honeypot_size->countQuery()->execute()->fetchField();
    $honeypot_count_result = empty($spammaster_honeypot_size_result)
      ? 'Honeypot Weekly Trigers: 0'
      : 'Honeypot Weekly Trigers: ' . number_format($spammaster_honeypot_size_result);
    $spammaster_exempt_count = $this->connection->select('spammaster_keys', 'u');
    $spammaster_exempt_count->condition('u.spamkey', '%' . $this->connection->escapeLike('exempt') . '%', 'LIKE');
    $spammaster_exempt_count->fields('u', ['spamkey']);
    $spammaster_exempt_query = $spammaster_exempt_count->countQuery()->execute()->fetchField();
    $spammaster_exempt_query = empty($spammaster_exempt_query) ? '0' : $spammaster_exempt_query;

    $spam_master_table_content = 'Spam Master weekly report for ' . $spammaster_site_name . '.' . "\r\n";
    $spam_master_table_content .= 'Drupal Version: ' . $spammaster_platform_version . "\r\n";
    $spam_master_table_content .= 'Spam Master Version: ' . $spammaster_version . ' - Status: ' . $spammaster_status . "\r\n";
    $spam_master_table_content .= 'Spam Master Key: ' . $spammaster_license . "\r\n";
    $spam_master_table_content .= 'Alert Level: ' . $alert_level . "\r\n";
    $spam_master_table_content .= 'Spam Probability: ' . $spammaster_license_probability . '%' . "\r\n";
    $spam_master_table_content .= 'Protected Against: ' . number_format($spammaster_license_protection) . ' million threats' . "\r\n";
    $spam_master_table_content .= 'Spam Buffer Size: ' . number_format($spammaster_buffer_size_result_count) . "\r\n";
    $spam_master_table_content .= 'Spam Whitelist Size: ' . number_format($spammaster_white_size_result_count) . "\r\n";
    $spam_master_table_content .= 'Spam Exempt Size: ' . number_format($spammaster_exempt_query) . "\r\n";
    $spam_master_table_content .= $total_block_count_result . "\r\n";
    $spam_master_table_content .= $block_count_result . "\r\n";
    $spam_master_table_content .= $firewall_count_result . "\r\n";
    $spam_master_table_content .= $honeypot_count_result . "\r\n";
    $spam_master_table_content .= 'Spam Master Statistics powered by Spam Master Drupal.' . "\r\n";
    $params['message'] = $spam_master_table_content;
    $this->sendMail($key, $to, $params);

    $this->logMail('help us improve email was successfully sent To:', $to);
  }

  /**
   * Discount notification mail.
   */
  public function spamMasterDiscNotify($spammaster_discdate, $spamdesc, $spamdescper, $spamcode, $spammaster_send) {
    $key = 'disc_notification';
    $site_settings = $this->configFactory->get('system.site');
    $spammaster_site_name = $site_settings->get('name');
    $admin_email = $site_settings->get('mail');
    $spammaster_attached = $this->state->get('spammaster.spam_master_attached');
    $spamsenddb = $this->state->get('spammaster.spam_master_disc_not');
    $spamsenddbdatepre = $this->state->get('spammaster.spam_master_disc_not_date');
    $spammaster_date_short = date('Y-m-d', $this->time->getCurrentTime());

    if ($spamsenddbdatepre !== $spammaster_discdate && '0' === $spammaster_send && '0' === $spamsenddb) {
      // Update state values.
      $updated_values = [
        'spammaster.spam_master_disc_not_date' => $spammaster_date_short,
        'spammaster.spam_master_disc_not' => '1',
      ];
      $this->state->setMultiple($updated_values);
      $to = $this->getRecipient($spammaster_attached, $admin_email);
      $spam_master_table_content = 'Your website, ' . $spammaster_site_name . ' is protected by Spam Master for some time now against millions of threats with a Free license key.' . "\r\n";
      $spam_master_table_content .= 'If you are enjoying the protection you can quickly get a PRO key with a' . $spamdesc . ' ' . $spamdescper . ' DISCOUNT CODE: ' . $spamcode . '.' . "\r\n\r\n";
      $spam_master_table_content .= 'Grab it Now... The offer is only valid today, if you decide to get it use the below coupon during checkout and afterwards insert your new PRO license key in the module settings page in order to connect to our Business Class Servers and enjoy bombastic scan speeds.' . "\r\n\r\n";
      $spam_master_table_content .= 'CHECKOUT: https://www.techgasp.com/downloads/spam-master-license/' . "\r\n";
      $spam_master_table_content .= 'DISCOUNT CODE: ' . $spamcode . '' . "\r\n\r\n";
      $spam_master_table_content .= 'Thanks.' . "\r\n" . 'Spam Master Team' . "\r\n";
      $params['message'] = $spam_master_table_content;
      $this->sendMail($key, $to, $params);

      $this->logMail('Discount notification email sent To:', $to);
    }
  }

}
