<?php

namespace Drupal\spamspan;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Security\Attribute\TrustedCallback;

/**
 * Provides a common Settings form for Spamspan plugins.
 */
trait SpamspanSettingsFormTrait {

  /**
   * {@inheritDoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element = [];
    // Spamspan '@' replacement.
    $element['spamspan_at'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Replacement for "@"'),
      '#default_value' => $this->getSetting('spamspan_at'),
      '#required' => TRUE,
      '#description' => $this->t('Replace "@" with this text when javascript is disabled.'),
    ];
    $element['spamspan_use_graphic'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use a graphical replacement for "@"'),
      '#default_value' => $this->getSetting('spamspan_use_graphic'),
      '#description' => $this->t('Replace "@" with a graphical representation when javascript is disabled (and ignore the setting "Replacement for @" above).'),
    ];
    $element['spamspan_dot_enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Replace dots in email with text'),
      '#default_value' => $this->getSetting('spamspan_dot_enable'),
      '#description' => $this->t('Switch on dot replacement.'),
    ];
    $element['spamspan_dot'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Replacement for "."'),
      '#default_value' => $this->getSetting('spamspan_dot'),
      '#required' => TRUE,
      '#description' => $this->t('Replace "." with this text.'),
    ];

    // No trees, see https://www.drupal.org/node/2378437.
    // Add a details on pre-render.
    $element['#pre_render'][] = [static::class, 'preRenderSettingsForm'];
    $element['use_form'] = [
      '#type' => 'details',
      '#title' => $this->t('Use a form instead of a link'),
      '#open' => TRUE,
    ];
    $element['spamspan_use_form'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use a form instead of a link'),
      '#default_value' => $this->getSetting('spamspan_use_form'),
      '#description' => $this->t('Link to a contact form instead of an email address. The following settings are used only if you select this option.'),
      '#details' => 'use_form',
    ];
    $element['spamspan_form_pattern'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Replacement string for the email address'),
      '#default_value' => $this->getSetting('spamspan_form_pattern'),
      '#required' => TRUE,
      '#description' => $this->t('Replace the email link with this string and substitute the following: <br />%url = the url where the form resides,<br />%email = the email address (base64 and urlencoded),<br />%displaytext = text to display instead of the email address.'),
      '#details' => 'use_form',
    ];
    $element['spamspan_form_default_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default url'),
      '#default_value' => $this->getSetting('spamspan_form_default_url'),
      '#required' => TRUE,
      '#description' => $this->t('Default url to form to use if none specified (e.g. me@example.com[custom_url_to_form])'),
      '#details' => 'use_form',
    ];
    $element['spamspan_form_default_displaytext'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default displaytext'),
      '#default_value' => $this->getSetting('spamspan_form_default_displaytext'),
      '#required' => TRUE,
      '#description' => $this->t('Default displaytext to use if none specified (e.g. me@example.com[custom_url_to_form|custom_displaytext])'),
      '#details' => 'use_form',
    ];

    return $element;
  }

  /**
   * Element '#pre_render' callback.
   *
   * In our form, we added an arbitrary #details property to any element that
   * belongs in a details. Move form elements having that property accordingly.
   */
  #[TrustedCallback]
  public static function preRenderSettingsForm(array $form): array {
    foreach (Element::children($form) as $key) {
      $element = $form[$key];
      if (isset($element['#details'], $form[$element['#details']])) {
        $form[$element['#details']][$key] = $element;
        unset($form[$key]);
      }
    }
    return $form;
  }

  /**
   * Returns the value of a setting, or its default value if absent.
   *
   * We need to define this method because EmailSpamspanFormatter and
   * FilterSpamspan have different interfaces and FilterSpamspan is missing
   * getSetting() definition.
   * Also for what ever reason because this is a Trait method overloading does
   * not work.
   *
   * @param string $key
   *   The setting name.
   *
   * @return mixed
   *   The setting value.
   *
   * @see PluginSettingsBase::getSetting()
   */
  public function getSetting($key) {
    // Merge defaults if we have no value for the key.
    if (method_exists($this, 'mergeDefaults') && !$this->defaultSettingsMerged && !array_key_exists($key, $this->settings)) {
      $this->mergeDefaults();
    }
    return $this->settings[$key] ?? NULL;
  }

}
