<?php

namespace Drupal\speakeasy\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Configure SpeakEasy settings for this site.
 */
class SpeakEasySettingsForm extends ConfigFormBase {

  /**
   * Logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a SpeakEasySettingsForm object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, RequestStack $request_stack, LoggerChannelFactoryInterface $logger_factory) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->requestStack = $request_stack;
    $this->logger = $logger_factory->get('speakeasy');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('request_stack'),
      $container->get('logger.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'speakeasy_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['speakeasy.settings'];
  }

  /**
   * Detect the current browser based on the User-Agent header.
   */
  private function getCurrentBrowser(): string {
    $ua = strtolower($this->requestStack->getCurrentRequest()->headers->get('User-Agent', ''));
    if (str_contains($ua, 'chrome') && !str_contains($ua, 'edg')) {
      return 'chrome';
    }
    elseif (str_contains($ua, 'edg')) {
      return 'edge';
    }
    elseif (str_contains($ua, 'firefox')) {
      return 'firefox';
    }
    elseif (str_contains($ua, 'safari') && !str_contains($ua, 'chrome')) {
      return 'safari';
    }
    return 'other';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('speakeasy.settings');
    $default_voices = $config->get('default_voices_by_browser') ?: [];
    $current_browser = $this->getCurrentBrowser();

    $this->logger->debug('Loaded configuration on form build', [
      'config' => $default_voices,
      'current_browser' => $current_browser,
    ]);

    $form['#attached']['library'][] = 'speakeasy/speakeasy.settings';

    $form['default_voices_by_browser'] = [
      '#type' => 'details',
      '#title' => $this->t('Default Voices by Browser'),
      '#description' => $this->t('Select default voices for each browser. Leave blank for browser default. Options for the current browser are loaded from your system and fallbacks are provided for others.'),
      '#open' => TRUE,
    ];

    $browsers = [
      'chrome' => $this->t('Google Chrome'),
      'edge' => $this->t('Microsoft Edge'),
      'firefox' => $this->t('Mozilla Firefox'),
      'safari' => $this->t('Apple Safari'),
      'other' => $this->t('Other Browsers'),
    ];

    foreach ($browsers as $key => $label) {
      $saved_voice = $default_voices[$key] ?? '';
      $form['default_voices_by_browser'][$key] = [
        '#type' => 'textfield',
        '#title' => $label,
        '#default_value' => $saved_voice,
        '#description' => $this->t('Select the voice for @browser. Options for the current browser come from available voices.', [
          '@browser' => $label,
        ]),
        '#autocomplete_route_name' => 'speakeasy.voice_autocomplete',
        '#autocomplete_route_parameters' => ['browser' => $key],
      ];
    }

    $form['default_speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Speed'),
      '#description' => $this->t('Default speech speed (0.1 to 10).'),
      '#default_value' => $config->get('default_speed') ?: 1,
      '#step' => 0.1,
      '#min' => 0.1,
      '#max' => 10,
    ];

    $form['theme'] = [
      '#type' => 'select',
      '#title' => $this->t('Theme'),
      '#description' => $this->t('Select the style theme for SpeakEasy output.'),
      '#options' => [
        'default' => $this->t('Default'),
        'olivero' => $this->t('Olivero'),
        'gin' => $this->t('Gin'),
      ],
      '#default_value' => $config->get('theme') ?: 'default',
    ];

    $form['block_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Block options'),
      '#open' => TRUE,
    ];

    $form['block_options']['allow_voice_selection'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow voice selection'),
      '#description' => $this->t('Allow SpeakEasy blocks to display the voice selection dropdown.'),
      '#default_value' => $config->get('allow_voice_selection') ?? TRUE,
    ];

    $form['block_options']['allow_highlighting'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow highlighting'),
      '#description' => $this->t('Allow SpeakEasy blocks to highlight text as it is spoken.'),
      '#default_value' => $config->get('allow_highlighting') ?? TRUE,
    ];

    $overrides = $config->get('language_voice_overrides') ?: [];
    $languages = \Drupal::languageManager()->getLanguages();

    $form['language_voice_overrides'] = [
      '#type' => 'table',
      '#caption' => $this->t('Language voice overrides'),
      '#header' => [
        $this->t('Language'),
        $this->t('Voice'),
      ],
    ];

    foreach ($languages as $langcode => $language) {
      $form['language_voice_overrides'][$langcode]['language'] = [
        '#plain_text' => $language->getName(),
      ];
      $form['language_voice_overrides'][$langcode]['voice'] = [
        '#type' => 'textfield',
        '#default_value' => $overrides[$langcode] ?? '',
      ];
    }

    // Hidden field to store real-time voices from JavaScript.
    $form['current_voices'] = [
      '#type' => 'hidden',
      '#default_value' => '',
      '#attributes' => ['id' => 'speakeasy-current-voices'],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $voices_by_browser = [];
    $browsers = ['chrome', 'edge', 'firefox', 'safari', 'other'];

    $all_values = $form_state->getValues();
    $this->logger->debug('All submitted form state values', [
      'values' => $all_values,
    ]);

    foreach ($browsers as $browser) {
      $voice = $form_state->getValue($browser) ?? '';
      $trimmedVoice = trim($voice);
      if (!empty($trimmedVoice)) {
        $voices_by_browser[$browser] = $trimmedVoice;
      }
    }

    $this->logger->debug('Processed voices to save', [
      'voices' => $voices_by_browser,
    ]);

    $config = $this->config('speakeasy.settings');
    $config
      ->set('default_voices_by_browser', $voices_by_browser)
      ->set('default_speed', $form_state->getValue('default_speed'))
      ->set('theme', $form_state->getValue('theme'))
      ->set('allow_voice_selection', (bool) $form_state->getValue('allow_voice_selection'))
      ->set('allow_highlighting', (bool) $form_state->getValue('allow_highlighting'));
    
    $language_overrides = [];
    $languages = \Drupal::languageManager()->getLanguages();
    $submitted_overrides = $form_state->getValue('language_voice_overrides') ?? [];
    foreach ($submitted_overrides as $langcode => $row) {
      $voice = trim($row['voice'] ?? '');
      if ($voice !== '' && isset($languages[$langcode])) {
        $language_overrides[$langcode] = $voice;
      }
    }
    $config->set('language_voice_overrides', $language_overrides)->save();

    $saved_config = $config->get();
    $this->logger->debug('Saved configuration', [
      'config' => $saved_config,
    ]);

    $this->messenger()->addStatus($this->t('SpeakEasy settings have been saved.'));

    parent::submitForm($form, $form_state);
  }

}
