<?php

namespace Drupal\speakeasy\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\UserDataInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for users to set SpeakEasy preferences.
 */
class SpeakEasyUserPreferencesForm extends FormBase {

  /**
   * The user data service.
   */
  protected UserDataInterface $userData;

  /**
   * The current user.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructs a new SpeakEasyUserPreferencesForm object.
   */
  public function __construct(UserDataInterface $user_data, AccountProxyInterface $current_user) {
    $this->userData = $user_data;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('user.data'),
      $container->get('current_user'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'speakeasy_user_preferences_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $uid = $this->currentUser->id();
    $voice_name = $this->userData->get('speakeasy', $uid, 'voice_name') ?? '';
    $speed = $this->userData->get('speakeasy', $uid, 'speed') ?? 1;

    $form['voice_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Preferred voice name'),
      '#description' => $this->t('Leave blank for the browser default.'),
      '#default_value' => $voice_name,
    ];

    $form['speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Speed'),
      '#description' => $this->t('Speech speed (0.1 to 10).'),
      '#default_value' => $speed,
      '#step' => 0.1,
      '#min' => 0.1,
      '#max' => 10,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save preferences'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $uid = $this->currentUser->id();
    $voice_name = trim($form_state->getValue('voice_name'));
    $speed = (float) $form_state->getValue('speed');

    $this->userData->set('speakeasy', $uid, 'voice_name', $voice_name);
    $this->userData->set('speakeasy', $uid, 'speed', $speed);

    $this->messenger()->addStatus($this->t('Your SpeakEasy preferences have been saved.'));
  }

}

