<?php

namespace Drupal\speakeasy\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\Config;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Html;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Component\Utility\Xss;
use Drupal\user\UserDataInterface;

/**
 * Provides a 'SpeakEasy' block.
 *
 * @Block(
 *   id = "speakeasy_block",
 *   admin_label = @Translation("SpeakEasy Block"),
 * )
 */
class SpeakEasyBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new SpeakEasyBlock instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\Config $config
   *   The configuration for this module.
   * @param \Drupal\node\NodeInterface|null $node
   *   The node whose page is being viewed, if any.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected Config $config,
    protected ?NodeInterface $node,
    protected AccountInterface $currentUser,
    protected UserDataInterface $userData,
    protected LanguageManagerInterface $languageManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory')->get('speakeasy.settings'),
      $container->get('current_route_match')->getParameter('node'),
      $container->get('current_user'),
      $container->get('user.data'),
      $container->get('language_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'output_style' => 'default',
      'button_text' => $this->t('Listen to this page'),
      'stop_button_text' => $this->t('Stop'),
      'voice_name' => '',
      'speed' => 1,
      'show_voice_select' => TRUE,
      'highlight' => FALSE,
      'fields' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Collect all text content from the node fields if a node is present.
    $content = '';

    // Selected fields from block configuration.
    $selected_fields = array_filter($this->configuration['fields'] ?? []);
    $content_selectors = [];
    if (!empty($selected_fields)) {
      foreach ($selected_fields as $field_name) {
        $class_selector = '.' . Html::getClass('field--name-' . $field_name);
        $data_selector = '[data-speakeasy-field="' . $field_name . '"]';
        $content_selectors[] = $class_selector;
        $content_selectors[] = $data_selector;
      }
    }

    if ($this->node) {
      // Loop through all fields on the node and collect text content.
      foreach ($this->node->getFieldDefinitions() as $field_name => $field_definition) {
        // Skip fields not chosen in configuration if any were selected.
        if (!empty($selected_fields) && !in_array($field_name, $selected_fields, TRUE)) {
          continue;
        }
        if ($this->node->hasField($field_name)) {
          $field = $this->node->get($field_name);

          // Ensure the current user can view the field before accessing it.
          if (!$field->access('view', $this->currentUser)) {
            continue;
          }

          if (!$field->isEmpty() && in_array($field_definition->getType(), ['string', 'string_long', 'text', 'text_long', 'text_with_summary', 'text_plain', 'email', 'telephone'])) {
            // For multi-valued fields, concatenate all values.
            foreach ($field as $item) {
              $text = $item->value;
              // Strip HTML tags and decode HTML entities to ensure plain text.
              $text = Html::decodeEntities(strip_tags($text));
              $content .= (strlen($content) > 0 ? ' ' : '') . $text;
            }
          }
        }
      }
    }

    // Pass the combined plain text content to drupalSettings.
    $uid = $this->currentUser->id();
    $user_voice = $this->userData->get('speakeasy', $uid, 'voice_name') ?: '';
    $user_speed = $this->userData->get('speakeasy', $uid, 'speed');

    $language_voice_map = $this->config->get('language_voice_overrides') ?: [];
    $interface_langcode = $this->languageManager->getCurrentLanguage()->getId();
    $default_langcode = $this->languageManager->getDefaultLanguage()->getId();
    $node_langcode = $this->node ? $this->node->language()->getId() : NULL;

    // Determine the language to use, falling back to interface or default.
    $resolved_langcode = $node_langcode ?? $interface_langcode ?? $default_langcode;

    // Determine the voice name based on user preference or language overrides.
    $voice_name = $user_voice;
    if (!$voice_name) {
      if ($node_langcode && isset($language_voice_map[$node_langcode])) {
        $voice_name = $language_voice_map[$node_langcode];
        $resolved_langcode = $node_langcode;
      }
      elseif (isset($language_voice_map[$interface_langcode])) {
        $voice_name = $language_voice_map[$interface_langcode];
        $resolved_langcode = $interface_langcode;
      }
      elseif (isset($language_voice_map[$default_langcode])) {
        $voice_name = $language_voice_map[$default_langcode];
        $resolved_langcode = $default_langcode;
      }
      else {
        $voice_name = $this->configuration['voice_name'] ?: $this->config->get('default_voice_name');
      }
    }

    $speed = $user_speed ?? ($this->configuration['speed'] ?: $this->config->get('default_speed') ?: 1);
    $output_style = $this->configuration['output_style'] ?? 'default';
    $allow_voice_selection = $this->config->get('allow_voice_selection');
    $allow_highlighting = $this->config->get('allow_highlighting');
    $show_voice_select = ($allow_voice_selection === NULL || $allow_voice_selection) && ($this->configuration['show_voice_select'] ?? TRUE);
    $highlight = ($allow_highlighting === NULL || $allow_highlighting) && !empty($this->configuration['highlight']);

    // Build the block.
    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['speakeasy-container']],
      '#attached' => [
        'library' => ['speakeasy/speakeasy'],
        'drupalSettings' => [
          'speakeasy' => [
            'content' => $content, // Pass combined plain text content to JS.
            'contentSelectors' => $content_selectors,
            'fieldNames' => $selected_fields,
            'voiceName' => $voice_name,
            'speed' => $speed,
            'outputStyle' => $output_style,
            'language' => $resolved_langcode,
            'languageVoiceMap' => $language_voice_map,
            'highlight' => $highlight,
            'userPreferences' => [
              'voiceName' => $user_voice,
              'speed' => $user_speed,
            ],
          ],
        ],
      ],
    ];

    $theme = $this->config->get('theme') ?: 'default';
    $theme_libraries = [
      'default' => 'speakeasy/speakeasy_theme_default',
      'olivero' => 'speakeasy/speakeasy_theme_olivero',
      'gin' => 'speakeasy/speakeasy_theme_gin',
    ];
    if (isset($theme_libraries[$theme])) {
      $build['#attached']['library'][] = $theme_libraries[$theme];
    }

    // Add cache contexts and tags to ensure the block is unique per node and
    // responds to configuration changes.
    $build['#cache']['contexts'][] = 'route';
    if ($this->node && ($nid = $this->node->id())) {
      $build['#cache']['tags'][] = "node:$nid";
    }
    $build['#cache']['tags'][] = 'config:speakeasy.settings';

    switch ($output_style) {
      case 'simple_link':
        // Include the voice select for this style as well.
        $build['voice_select'] = [
          '#type' => 'html_tag',
          '#tag' => 'select',
          '#attributes' => [
            'id' => 'speakeasy-voice-select',
            'class' => ['speakeasy-voice-select'],
          ],
        ];
        if (!$show_voice_select) {
          $build['voice_select']['#attributes']['style'] = 'display:none';
        }

        $build['simple_link'] = [
          '#type' => 'link',
          '#title' => $this->t('Listen'),
          '#url' => Url::fromRoute('<none>'),
          '#attributes' => [
            'id' => 'speakeasy-link',
            'class' => ['speakeasy-link'],
            'href' => '#',
            // Indicate it's a button-like link.
            'role' => 'button',
            // Indicate toggle state.
            'aria-pressed' => 'false',
          ],
        ];
        break;

      case 'media_player':
        $build['media_player'] = [
          '#theme' => 'speakeasy_media_player',
          '#show_voice_select' => $show_voice_select,
          '#attached' => ['library' => ['speakeasy/speakeasy']],
        ];
        break;

      default:
        $build['voice_select'] = [
          '#type' => 'html_tag',
          '#tag' => 'select',
          '#attributes' => [
            'id' => 'speakeasy-voice-select',
            'class' => ['speakeasy-voice-select'],
          ],
        ];
        if (!$show_voice_select) {
          $build['voice_select']['#attributes']['style'] = 'display:none';
        }

        $build['listen_button'] = [
          '#type' => 'html_tag',
          '#tag' => 'button',
          '#value' => $this->configuration['button_text'],
          '#attributes' => [
            'id' => 'speakeasy-tts-button',
            'class' => ['speakeasy-button'],
            'aria-label' => $this->t('Listen to the content on this page'),
          ],
        ];


        $build['stop_button'] = [
          '#type' => 'html_tag',
          '#tag' => 'button',
          '#value' => $this->configuration['stop_button_text'],
          '#attributes' => ['id' => 'speakeasy-stop-button', 'class' => ['speakeasy-button']],
        ];
        break;
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    // Add a link to the settings page.
    if ($this->currentUser->hasPermission('access speakeasy settings')) {
      $link = Link::fromTextAndUrl(
        $this->t('SpeakEasy settings page'),
        Url::fromRoute('speakeasy.settings', [], [
          'attributes' => [
            'target' => '_blank',
            'rel' => 'noopener noreferrer',
          ],
        ])
      )->toString();

      $text = $this->t('To configure default settings, visit the :link.', [':link' => $link]);

      $form['speakeasy_settings_link'] = [
        '#type' => 'markup',
        '#markup' => Markup::create(Xss::filter('<p>' . $text . '</p>', ['p', 'a'])),
        '#weight' => -10,
      ];
    }

    // Build options for available text fields.
    $field_options = [];
    $bundle_info = \Drupal::service('entity_type.bundle.info')->getBundleInfo('node');
    $entity_field_manager = \Drupal::service('entity_field.manager');
    foreach (array_keys($bundle_info) as $bundle) {
      $definitions = $entity_field_manager->getFieldDefinitions('node', $bundle);
      foreach ($definitions as $field_name => $definition) {
        $type = $definition->getType();
        if (in_array($type, ['string', 'string_long', 'text', 'text_long', 'text_with_summary', 'text_plain', 'email', 'telephone'])) {
          $field_options[$field_name] = $definition->getLabel() . ' (' . $field_name . ')';
        }
      }
    }
    ksort($field_options);
    $allow_voice_selection = $this->config->get('allow_voice_selection');
    $allow_highlighting = $this->config->get('allow_highlighting');
    if (!$allow_voice_selection) {
      $this->configuration['show_voice_select'] = FALSE;
    }
    if (!$allow_highlighting) {
      $this->configuration['highlight'] = FALSE;
    }
    if ($allow_voice_selection === NULL || $allow_voice_selection) {
      $form['show_voice_select'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Show Voice Selection Dropdown'),
        '#description' => $this->t('Allow users to select from available voices.'),
        '#default_value' => $this->configuration['show_voice_select'],
      ];
    }

    if ($allow_highlighting === NULL || $allow_highlighting) {
      $form['highlight'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Highlight spoken text'),
        '#description' => $this->t('Highlight text on the page as it is spoken.'),
        '#default_value' => $this->configuration['highlight'],
      ];
    }
    
    $form['output_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Output Style'),
      '#description' => $this->t('Select the display style for the SpeakEasy block.'),
      '#options' => [
        'simple_link' => $this->t('Simple Link'),
        'media_player' => $this->t('Media Player'),
        'default' => $this->t('Default'),
      ],
      '#default_value' => $this->configuration['output_style'],
    ];

    $form['button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Listen Button Text'),
      '#description' => $this->t('Text displayed on the listen button.'),
      '#default_value' => $this->configuration['button_text'],
      '#states' => [
        'visible' => [
          ':input[name="settings[output_style]"]' => ['value' => 'default'],
        ],
      ],
    ];

    $form['stop_button_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Stop Button Text'),
      '#description' => $this->t('Text displayed on the stop button.'),
      '#default_value' => $this->configuration['stop_button_text'],
      '#states' => [
        'visible' => [
          ':input[name="settings[output_style]"]' => ['value' => 'default'],
        ],
      ],
    ];
    $form['fields'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Fields to include'),
      '#description' => $this->t('Select the text fields that should be read.'),
      '#options' => $field_options,
      '#default_value' => $this->configuration['fields'],
    ];
    $form['voice_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default Voice Name'),
      '#description' => $this->t('Name of the default voice to use. Leave blank for browser default.'),
      '#default_value' => $this->configuration['voice_name'],
    ];

    $form['speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Speed'),
      '#description' => $this->t('Speech speed (0.1 to 10). Default is 1.'),
      '#default_value' => $this->configuration['speed'],
      '#step' => 0.1,
      '#min' => 0.1,
      '#max' => 10,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    foreach (array_keys($this->defaultConfiguration()) as $config_field) {
      if ($config_field === 'fields') {
        $this->configuration[$config_field] = array_filter($form_state->getValue($config_field) ?? []);
      }
      else {
        $this->configuration[$config_field] = $form_state->getValue($config_field);
      }
    }
    $allow_voice_selection = $this->config->get('allow_voice_selection');
    $allow_highlighting = $this->config->get('allow_highlighting');
    if (!$allow_voice_selection) {
      $this->configuration['show_voice_select'] = FALSE;
    }
    if (!$allow_highlighting) {
      $this->configuration['highlight'] = FALSE;
    }
  }

}
