(function (Drupal, drupalSettings, once) {
  Drupal.behaviors.speakeasy = {
    attach: function (context, settings) {
      once('speakeasy', '.speakeasy-container', context).forEach(function (element) {
        const outputStyle = drupalSettings.speakeasy.outputStyle || 'default';
        const highlight = !!drupalSettings.speakeasy.highlight;
        function normalizeLang(lang) {
          return (lang || '').replace(/_/g, '-').toLowerCase();
        }

        if (!window.speechSynthesis) {
          alert('Sorry, your browser does not support speech synthesis.');
          return;
        }

        const synth = window.speechSynthesis;
        let voices = [];
        let utteranceQueue = [];
        let currentUtteranceIndex = 0;
        let isSpeaking = false;
        let sentenceSpans = [];
        let autoScroll = true;
        let isAutoScrolling = false;
        let progressSlider;
        let updateProgressSlider = () => {};

        const onUserScroll = () => {
          if (!isAutoScrolling) {
            autoScroll = false;
            window.removeEventListener('scroll', onUserScroll);
          }
        };
        window.addEventListener('scroll', onUserScroll, { passive: true });

        const fieldNames = Array.isArray(drupalSettings.speakeasy.fieldNames)
          ? drupalSettings.speakeasy.fieldNames
          : [];
        fieldNames.forEach((field) => {
          const classSelector = '.field--name-' + field.replace(/_/g, '-');
          document.querySelectorAll(classSelector).forEach((el) => {
            el.setAttribute('data-speakeasy-field', field);
          });
        });

        const highlightSentences = (container) => {
          const spans = [];
          const sentences = [];
          const sentenceEnd = /[.!?]+[\])'"`’”]*\s*/;
          const walker = document.createTreeWalker(container, NodeFilter.SHOW_TEXT, {
            acceptNode: (node) => node.textContent.trim() ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_REJECT
          });

          let startNode = null;
          let startOffset = 0;
          let buffer = '';
          let node;
          let lastNode = null;
          const ranges = [];

          while ((node = walker.nextNode())) {
            lastNode = node;
            const text = node.textContent;
            for (let i = 0; i < text.length; i++) {
              if (buffer === '') {
                startNode = node;
                startOffset = i;
              }
              buffer += text[i];
              if (sentenceEnd.test(buffer)) {
                const range = document.createRange();
                range.setStart(startNode, startOffset);
                range.setEnd(node, i + 1);
                ranges.push(range);
                sentences.push(buffer.trim());
                buffer = '';
              }
            }
          }

          if (buffer.trim() !== '' && startNode && lastNode) {
            const range = document.createRange();
            range.setStart(startNode, startOffset);
            range.setEnd(lastNode, lastNode.textContent.length);
            ranges.push(range);
            sentences.push(buffer.trim());
          }

          ranges.forEach((range, index) => {
            const span = document.createElement('span');
            span.classList.add('speakeasy-sentence');
            span.dataset.index = index;
            try {
              range.surroundContents(span);
              spans.push(span);
            } catch (e) {
              // If we cannot surround contents, skip highlighting for this sentence.
            }
          });

          return { spans, sentences };
        };
        const clearHighlight = () => {
          if (!highlight) {
            return;
          }
          sentenceSpans.forEach(span => span.classList.remove('speakeasy-current'));
        };

        const loadVoices = () => {
          voices = synth.getVoices();
          const voiceSelect = element.querySelector('#speakeasy-voice-select');
          if (voiceSelect && voices.length > 0) {
            voiceSelect.innerHTML = '';
            voices.forEach((voice) => {
              const option = document.createElement('option');
              option.value = voice.name;
              option.textContent = `${voice.name} (${voice.lang})`;
              voiceSelect.appendChild(option);
            });
          }

          let langCode = drupalSettings.speakeasy.language ||
            document.documentElement.lang ||
            navigator.language;
          langCode = langCode ? langCode.toLowerCase() : '';

          const map = drupalSettings.speakeasy.languageVoiceMap || {};
          let defaultVoiceName = '';

          const normalizedLangCode = normalizeLang(langCode);
          const baseLangCode = normalizedLangCode.split('-')[0];

          if (normalizedLangCode && map[normalizedLangCode]) {
            defaultVoiceName = map[normalizedLangCode];
          } else if (normalizedLangCode) {
            const match = voices.find(v => {
              const voiceLang = normalizeLang(v.lang);
              return voiceLang && (voiceLang === normalizedLangCode || voiceLang.split('-')[0] === baseLangCode);
            });
            if (match) {
              defaultVoiceName = match.name;
            }
          }

          if (!defaultVoiceName) {
            defaultVoiceName = drupalSettings.speakeasy.voiceName;
          }

          if (defaultVoiceName && voices.some(v => v.name === defaultVoiceName)) {
            if (voiceSelect) {
              voiceSelect.value = defaultVoiceName;
            }
            drupalSettings.speakeasy.voiceName = defaultVoiceName;
          }
        };

        if (synth.onvoiceschanged !== undefined) {
          synth.onvoiceschanged = loadVoices;
        }
        loadVoices();

        const getSelectedVoice = () => {
          const voiceSelect = element.querySelector('#speakeasy-voice-select');
          const selectedVoiceName = voiceSelect ? voiceSelect.value : drupalSettings.speakeasy.voiceName;
          return selectedVoiceName ? voices.find(voice => voice.name === selectedVoiceName) : null;
        };

        const getSpeed = () => {
          const speedInput = element.querySelector('#speakeasy-speed');
          return speedInput ? parseFloat(speedInput.value) || 1 : parseFloat(drupalSettings.speakeasy.speed) || 1;
        };

        const createUtterances = () => {
          let sentences = [];
          sentenceSpans = [];
          let containers = [];
          if (highlight) {
            const selectors = Array.isArray(drupalSettings.speakeasy.contentSelectors)
              ? drupalSettings.speakeasy.contentSelectors
              : [];
            const containerSet = new Set();
            selectors.forEach((sel) => {
              try {
                const found = document.querySelectorAll(sel);
                if (found.length) {
                  found.forEach(el => containerSet.add(el));
                }
              } catch (e) {
                // Ignore invalid selectors.
              }
            });
            containers = Array.from(containerSet);
          }
          if (highlight && containers.length) {
            sentences = [];
            containers.forEach((container) => {
              const result = highlightSentences(container);
              sentenceSpans = sentenceSpans.concat(result.spans);
              sentences = sentences.concat(result.sentences);
            });
          } else {
            const text = drupalSettings.speakeasy.content || '';
            sentences = text.match(/[^.!?]+[.!?]+[\])'"`’”]*|.+$/g) || [];
          }

          utteranceQueue = sentences.map((sentence) => {
            const trimmed = sentence.trim();
            const utterance = new SpeechSynthesisUtterance(trimmed);
            utterance.rate = getSpeed();
            utterance.voice = getSelectedVoice();
            return utterance;
          });
        };

        const speakQueue = () => {
          updateProgressSlider();
          if (currentUtteranceIndex < utteranceQueue.length) {
            const utterance = utteranceQueue[currentUtteranceIndex];

            utterance.voice = getSelectedVoice();
            utterance.rate = getSpeed();

            if (highlight && sentenceSpans[currentUtteranceIndex]) {
              sentenceSpans.forEach(span => span.classList.remove('speakeasy-current'));
              const currentSpan = sentenceSpans[currentUtteranceIndex];
              currentSpan.classList.add('speakeasy-current');
              if (autoScroll) {
                isAutoScrolling = true;
                currentSpan.scrollIntoView({ behavior: 'smooth', block: 'center' });
                setTimeout(() => {
                  isAutoScrolling = false;
                }, 1000);
              }
            }

            utterance.onend = () => {
              if (highlight && sentenceSpans[currentUtteranceIndex]) {
                sentenceSpans[currentUtteranceIndex].classList.remove('speakeasy-current');
              }
              currentUtteranceIndex++;
              speakQueue();
            };

            synth.speak(utterance);
          } else {
            isSpeaking = false;
            currentUtteranceIndex = 0;
            utteranceQueue = [];
            clearHighlight();
            updateProgressSlider();
          }
        };

        const updateLinkState = () => {
          const link = document.getElementById('speakeasy-link');
          if (link) {
            if (isSpeaking) {
              link.textContent = Drupal.t('Stop Listening');
              link.setAttribute('aria-pressed', 'true');
            } else {
              link.textContent = Drupal.t('Listen');
              link.setAttribute('aria-pressed', 'false');
            }
            link.classList.toggle('speaking', isSpeaking);
          }
        };

        if (outputStyle === 'media_player') {
          const playButton = element.querySelector('#speakeasy-play-button');
          const pauseButton = element.querySelector('#speakeasy-pause-button');
          const stopButton = element.querySelector('#speakeasy-stop-button');
          const settingsToggle = element.querySelector('#speakeasy-settings-toggle');
          const settingsContainer = element.querySelector('#speakeasy-settings');
          progressSlider = element.querySelector('#speakeasy-progress');

          updateProgressSlider = () => {
            if (progressSlider && utteranceQueue.length > 0) {
              const progress = (currentUtteranceIndex / utteranceQueue.length) * 100;
              progressSlider.value = progress;
            }
          };

          if (settingsToggle && settingsContainer) {
            settingsToggle.addEventListener('click', function () {
              settingsContainer.style.display = (settingsContainer.style.display === 'none') ? 'block' : 'none';
            });
          }

          if (playButton) {
            playButton.addEventListener('click', function () {
              if (isSpeaking && synth.paused) {
                synth.resume();
              } else if (!isSpeaking) {
                const contentText = drupalSettings.speakeasy.content;
                if (!contentText && !highlight) {
                  alert('No content available to read.');
                  return;
                }

                createUtterances();
                isSpeaking = true;
                currentUtteranceIndex = 0;
                synth.cancel();
                if (progressSlider) {
                  progressSlider.value = 0;
                }
                speakQueue();
              }
            });
          }

          if (pauseButton) {
            pauseButton.addEventListener('click', function () {
              if (synth.speaking && !synth.paused) {
                synth.pause();
              } else if (synth.paused) {
                synth.resume();
              }
            });
          }

          if (stopButton) {
            stopButton.addEventListener('click', function () {
              if (synth.speaking || isSpeaking) {
                synth.cancel();
                isSpeaking = false;
                currentUtteranceIndex = 0;
                utteranceQueue = [];
                clearHighlight();
                if (progressSlider) {
                  progressSlider.value = 0;
                }
              }
            });
          }

          if (progressSlider) {
            progressSlider.addEventListener('input', function () {
              if (utteranceQueue.length > 0) {
                const newIndex = Math.round((progressSlider.value / 100) * utteranceQueue.length);
                currentUtteranceIndex = Math.min(newIndex, utteranceQueue.length - 1);
                if (isSpeaking) {
                  synth.cancel();
                  speakQueue();
                }
              }
            });
          }
        } else if (outputStyle === 'default') {
          const ttsButton = element.querySelector('#speakeasy-tts-button');
          const stopButton = element.querySelector('#speakeasy-stop-button');

          if (ttsButton) {
            ttsButton.addEventListener('click', function () {
              if (!isSpeaking) {
                const contentText = drupalSettings.speakeasy.content;
                if (!contentText && !highlight) {
                  alert('No content available to read.');
                  return;
                }

                createUtterances();
                isSpeaking = true;
                currentUtteranceIndex = 0;
                synth.cancel();
                speakQueue();
              }
            });
          }

          if (stopButton) {
            stopButton.addEventListener('click', function () {
              if (synth.speaking || isSpeaking) {
                synth.cancel();
                isSpeaking = false;
                currentUtteranceIndex = 0;
                utteranceQueue = [];
                clearHighlight();
              }
            });
          }
          } else if (outputStyle === 'simple_link') {
            if (outputStyle === 'simple_link') {
              const link = document.getElementById('speakeasy-link');
              if (link) {
                link.addEventListener('click', function (e) {
                  e.preventDefault();
                  if (isSpeaking) {
                    synth.cancel();
                    isSpeaking = false;
                    currentUtteranceIndex = 0;
                    utteranceQueue = [];
                    clearHighlight();
                    updateLinkState();
                  } else {
                    const contentText = drupalSettings.speakeasy.content;
                    if (!contentText && !highlight) {
                      alert('No content available to read.');
                      return;
                    }

                    createUtterances();
                    isSpeaking = true;
                    currentUtteranceIndex = 0;
                    synth.cancel();
                    speakQueue();
                    updateLinkState();
                  }
                });

                // Allow triggering via Enter or Space keys.
                link.addEventListener('keydown', function (e) {
                  if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    link.click();
                  }
                });
              }
            }
          }

          once('speakeasy-keyboard-shortcuts', 'body').forEach(function () {
            document.addEventListener('keydown', function (e) {
              const active = document.activeElement;
              if (active && (active.tagName === 'INPUT' || active.tagName === 'TEXTAREA' || active.tagName === 'SELECT' || active.isContentEditable)) {
                return;
              }

              if (e.code === 'Space' || e.key === ' ') {
                e.preventDefault();
                if (isSpeaking) {
                  synth.cancel();
                  isSpeaking = false;
                  currentUtteranceIndex = 0;
                  utteranceQueue = [];
                  clearHighlight();
                  updateLinkState();
                } else {
                  const contentText = drupalSettings.speakeasy.content;
                  if (!contentText && !highlight) {
                    alert('No content available to read.');
                    return;
                  }
                  createUtterances();
                  isSpeaking = true;
                  currentUtteranceIndex = 0;
                  synth.cancel();
                  speakQueue();
                  updateLinkState();
                }
              } else if (e.key && e.key.toLowerCase() === 's') {
                e.preventDefault();
                if (synth.speaking || isSpeaking) {
                  synth.cancel();
                  isSpeaking = false;
                  currentUtteranceIndex = 0;
                  utteranceQueue = [];
                  clearHighlight();
                  updateLinkState();
                }
              }
            });
          });

          // Stop speech on page unload
          window.addEventListener('pagehide', function () {
            if (synth.speaking) {
              synth.cancel();
            }
            clearHighlight();
          });

          window.addEventListener('beforeunload', function () {
            if (synth.speaking) {
              synth.cancel();
            }
            clearHighlight();
          });
        });
      }
    };
})(Drupal, drupalSettings, once);
