<?php

namespace Drupal\speakeasy\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Psr\Log\LoggerInterface;

/**
 * Controller for voice autocomplete functionality.
 */
class VoiceAutocompleteController extends ControllerBase {

  /**
   * The request stack service.
   */
  protected RequestStack $requestStack;

  /**
   * Logger channel.
   */
  protected LoggerInterface $logger;

  /**
   * State service.
   */
  protected StateInterface $state;

  /**
   * Constructs a VoiceAutocompleteController object.
   */
  public function __construct(RequestStack $request_stack, LoggerChannelFactoryInterface $logger_factory, StateInterface $state) {
    $this->requestStack = $request_stack;
    $this->logger = $logger_factory->get('speakeasy');
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('logger.factory'),
      $container->get('state'),
    );
  }

  /**
   * Detect the current browser based on the User-Agent header.
   */
  private function getCurrentBrowser() {
    $ua = strtolower($this->requestStack->getCurrentRequest()->headers->get('User-Agent', ''));
    $this->logger->debug('User-Agent', ['ua' => $ua]);
    if (strpos($ua, 'chrome') !== FALSE && strpos($ua, 'edg') === FALSE) {
      return 'chrome';
    }
    elseif (strpos($ua, 'edg') !== FALSE) {
      return 'edge';
    }
    elseif (strpos($ua, 'firefox') !== FALSE) {
      return 'firefox';
    }
    elseif (strpos($ua, 'safari') !== FALSE && strpos($ua, 'chrome') === FALSE) {
      return 'safari';
    }
    return 'other';
  }

  /**
   * Autocomplete handler for voice names.
   */
  public function autocomplete(string $browser) {
    $request = $this->requestStack->getCurrentRequest();
    $q = $request->query->get('q', '');
    $matches = [];
    $current_browser = $this->getCurrentBrowser();

    $this->logger->debug('Autocomplete called', [
      'browser' => $browser,
      'current_browser' => $current_browser,
      'q' => $q,
    ]);

    // Get voices from state for the current browser.
    $available_voices = [];
    if ($browser === $current_browser) {
      $stored_voices = $this->state->get('speakeasy_available_voices', []);
      $this->logger->debug('Stored voices from state', [
        'voices' => $stored_voices,
      ]);
      if (!empty($stored_voices) && is_array($stored_voices)) {
        $available_voices = $stored_voices;
      }
    }

    // Fallback lists for non-active browsers or if no voices are stored.
    if (empty($available_voices)) {
      $this->logger->debug('Using fallback voices', ['browser' => $browser]);
      switch ($browser) {
        case 'chrome':
          $available_voices = [
            ['name' => 'Google US English', 'lang' => 'en-US'],
            ['name' => 'Google UK English Female', 'lang' => 'en-GB'],
            ['name' => 'Google español', 'lang' => 'es-ES'],
          ];
          break;

        case 'edge':
          $available_voices = [
            ['name' => 'Microsoft David Desktop', 'lang' => 'en-US'],
            ['name' => 'Microsoft Zira Desktop', 'lang' => 'en-US'],
            ['name' => 'Microsoft Mark Mobile', 'lang' => 'en-US'],
          ];
          break;

        case 'firefox':
          $available_voices = [
            ['name' => 'Mozilla TTS Voice', 'lang' => 'en-US'],
            ['name' => 'Mozilla TTS UK', 'lang' => 'en-GB'],
          ];
          break;

        case 'safari':
          $available_voices = [
            ['name' => 'Samantha', 'lang' => 'en-US'],
            ['name' => 'Alex', 'lang' => 'en-US'],
            ['name' => 'Victoria', 'lang' => 'en-GB'],
          ];
          break;

        case 'other':
        default:
          $available_voices = [
            ['name' => 'Google US English', 'lang' => 'en-US'],
            ['name' => 'Microsoft David Desktop', 'lang' => 'en-US'],
            ['name' => 'Mozilla TTS Voice', 'lang' => 'en-US'],
          ];
          break;
      }
    }

    if ($q) {
      foreach ($available_voices as $voice) {
        if (stripos($voice['name'], $q) !== FALSE) {
          $matches[] = [
            'value' => $voice['name'],
            'label' => $voice['name'] . ' (' . $voice['lang'] . ')',
          ];
        }
      }
    }

    $this->logger->debug('Autocomplete matches', [
      'matches' => $matches,
    ]);

    return new JsonResponse($matches);
  }
}
