<?php

namespace Drupal\speakeasy\Controller;

use Drupal\Core\Access\CsrfTokenValidatorInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Psr\Log\LoggerInterface;

class VoiceStorageController extends ControllerBase {

  /**
   * The CSRF token validator.
   *
   * @var \Drupal\Core\Access\CsrfTokenValidatorInterface
   */
  protected CsrfTokenValidatorInterface $csrfTokenValidator;

  /**
   * Logger channel.
   */
  protected LoggerInterface $logger;

  /**
   * State service.
   */
  protected StateInterface $state;

  /**
   * Constructs a VoiceStorageController object.
   */
  public function __construct(CsrfTokenValidatorInterface $csrf_token_validator, LoggerChannelFactoryInterface $logger_factory, StateInterface $state) {
    $this->csrfTokenValidator = $csrf_token_validator;
    $this->logger = $logger_factory->get('speakeasy');
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('csrf_token'),
      $container->get('logger.factory'),
      $container->get('state'),
    );
  }

  /**
   * Stores voices sent via AJAX request.
   */
  public function storeVoices(Request $request) {
    // Validate the CSRF token from the X-CSRF-Token header.
    $token = $request->headers->get('X-CSRF-Token');
    if (!$this->csrfTokenValidator->validate($token, 'speakeasy/store-voices')) {
      throw new AccessDeniedHttpException();
    }

    // Handle JSON payload if Content-Type is application/json.
    if ($request->headers->get('Content-Type') === 'application/json') {
      $data = json_decode($request->getContent(), TRUE);
      $voices = $data['voices'] ?? null;
    }
    else {
      // Fallback to form-encoded data (e.g., from Drupal.ajax).
      $voices = $request->request->get('voices', null);
    }

    // Validate that voices is an array and not empty.
    if (is_array($voices) && !empty($voices)) {
      $this->state->set('speakeasy_available_voices', $voices);
      $this->logger->debug('Stored voices', ['voices' => $voices]);
      return new JsonResponse(['status' => 'success']);
    }

    // Return an error response if the data is invalid.
    return new JsonResponse(['status' => 'error', 'message' => 'Invalid or empty voices data'], 400);
  }
}
