(function (Drupal, drupalSettings, once) {
  const { EVENTS } = Drupal.speakeasyConstants;
  const bus = Drupal.speakeasyBus;
  Drupal.behaviors.speakeasyHighlight = {
    attach: function (context, settings) {
      if (!drupalSettings.speakeasy || !drupalSettings.speakeasy.highlight) {
        return;
      }
      once('speakeasy-highlight', 'body', context).forEach(function () {
        const selectors = Array.isArray(drupalSettings.speakeasy.contentSelectors)
          ? drupalSettings.speakeasy.contentSelectors
          : [];
        if (selectors.length === 0) {
          return;
        }

        if (Drupal.speakeasyUtils && Drupal.speakeasyUtils.markSpeakeasyFields) {
          Drupal.speakeasyUtils.markSpeakeasyFields();
        }

        let sentenceSpans = [];
        let autoScroll = true;
        let isAutoScrolling = false;
        let currentIndex = -1;

        const onUserScroll = () => {
          if (!isAutoScrolling) {
            autoScroll = false;
            window.removeEventListener('scroll', onUserScroll);
          }
        };
        window.addEventListener('scroll', onUserScroll, { passive: true });

        const highlightSentences = (container) => {
          const spans = [];
          const sentenceEnd = /[.!?]+[\])'"`’”]*\s*/;
          const walker = document.createTreeWalker(container, NodeFilter.SHOW_TEXT, {
            acceptNode: (node) => node.textContent.trim() ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_REJECT,
          });

          let startNode = null;
          let startOffset = 0;
          let buffer = '';
          let node;
          let lastNode = null;
          const ranges = [];

          while ((node = walker.nextNode())) {
            lastNode = node;
            const text = node.textContent;
            for (let i = 0; i < text.length; i++) {
              if (buffer === '') {
                startNode = node;
                startOffset = i;
              }
              buffer += text[i];
              if (sentenceEnd.test(buffer)) {
                const range = document.createRange();
                range.setStart(startNode, startOffset);
                range.setEnd(node, i + 1);
                ranges.push(range);
                buffer = '';
              }
            }
          }

          if (buffer.trim() !== '' && startNode && lastNode) {
            const range = document.createRange();
            range.setStart(startNode, startOffset);
            range.setEnd(lastNode, lastNode.textContent.length);
            ranges.push(range);
          }

          ranges.forEach((range, index) => {
            const span = document.createElement('span');
            span.classList.add('speakeasy-sentence');
            span.dataset.index = index;
            try {
              range.surroundContents(span);
              spans.push(span);
            }
            catch (e) {
              // If we cannot surround contents, skip highlighting for this sentence.
            }
          });

          return spans;
        };

        const clearHighlight = () => {
          sentenceSpans.forEach((span) => span.classList.remove('speakeasy-current'));
        };

        const containerSet = new Set();
        selectors.forEach((sel) => {
          try {
            const found = document.querySelectorAll(sel);
            if (found.length) {
              found.forEach((el) => containerSet.add(el));
            }
          }
          catch (e) {
            // Ignore invalid selectors.
          }
        });
        const containers = Array.from(containerSet);
        containers.forEach((container) => {
          sentenceSpans = sentenceSpans.concat(highlightSentences(container));
        });

        bus.subscribe(EVENTS.SENTENCE_START, (e) => {
          currentIndex = e.detail.index;
          sentenceSpans.forEach((span) => span.classList.remove('speakeasy-current'));
          const current = sentenceSpans[currentIndex];
          if (current) {
            current.classList.add('speakeasy-current');
            if (autoScroll) {
              isAutoScrolling = true;
              current.scrollIntoView({ behavior: 'smooth', block: 'center' });
              setTimeout(() => {
                isAutoScrolling = false;
              }, 1000);
            }
          }
        });

        bus.subscribe(EVENTS.PAUSE, () => {
          clearHighlight();
        });

        bus.subscribe(EVENTS.RESUME, () => {
          if (currentIndex >= 0) {
            const current = sentenceSpans[currentIndex];
            if (current) {
              current.classList.add('speakeasy-current');
              if (autoScroll) {
                isAutoScrolling = true;
                current.scrollIntoView({ behavior: 'smooth', block: 'center' });
                setTimeout(() => {
                  isAutoScrolling = false;
                }, 1000);
              }
            }
          }
        });

        const stopHandler = () => {
          clearHighlight();
          currentIndex = -1;
        };

        bus.subscribe(EVENTS.STOP, stopHandler);
        window.addEventListener('pagehide', stopHandler);
        window.addEventListener('beforeunload', stopHandler);
      });
    },
  };
})(Drupal, drupalSettings, once);
