<?php

namespace Drupal\speakeasy\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\UserDataInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for users to set Speakeasy preferences.
 */
class SpeakeasyUserPreferencesForm extends FormBase {

  /**
   * The user data service.
   */
  protected UserDataInterface $userData;

  /**
   * The current user.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructs a new SpeakeasyUserPreferencesForm object.
   */
  public function __construct(UserDataInterface $user_data, AccountProxyInterface $current_user) {
    $this->userData = $user_data;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('user.data'),
      $container->get('current_user'),
    );
  }

  /**
   * {@inheritdoc}
   *
   * @return string
   *   The unique form identifier.
   */
  public function getFormId(): string {
    return 'speakeasy_user_preferences_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The built form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $uid = $this->currentUser->id();
    $voice_name = $this->userData->get('speakeasy', $uid, 'voice_name') ?? '';
    $speed = $this->userData->get('speakeasy', $uid, 'speed') ?? 1;
    $settings_config = \Drupal::config('speakeasy.settings');
    $allowed_voices = $settings_config->get('allowed_voices_by_browser') ?: [];
    $allowed_languages = $settings_config->get('allowed_languages') ?: [];

    $form['voice_name'] = [
      '#type' => 'select',
      '#title' => $this->t('Preferred voice'),
      '#description' => $this->t('Select from available voices. Leave blank for the browser default.'),
      '#options' => [],
      '#empty_option' => $this->t('- Browser default -'),
      '#default_value' => $voice_name,
      '#validated' => TRUE,
      '#attributes' => [
        'id' => 'speakeasy-user-voice-select',
        'data-default-voice' => $voice_name,
      ],
    ];

    $form['#attached']['library'][] = 'speakeasy/speakeasy.user_preferences';
    $form['#attached']['drupalSettings']['speakeasy']['allowedVoices'] = $allowed_voices;
    $form['#attached']['drupalSettings']['speakeasy']['allowedLanguages'] = $allowed_languages;

    $form['speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Speed'),
      '#description' => $this->t('Speech speed (0.1 to 2).'),
      '#default_value' => $speed,
      '#step' => 0.1,
      '#min' => 0.1,
      '#max' => 2,
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save preferences'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $uid = $this->currentUser->id();
    $voice_name = trim($form_state->getValue('voice_name'));
    $speed = (float) $form_state->getValue('speed');

    $this->userData->set('speakeasy', $uid, 'voice_name', $voice_name);
    $this->userData->set('speakeasy', $uid, 'speed', $speed);

    $this->messenger()->addStatus($this->t('Your Speakeasy preferences have been saved.'));
  }

}

