(function (Drupal, drupalSettings, once) {
  const { EVENTS, PLAYER_STATES } = Drupal.speakeasyConstants;
  const bus = Drupal.speakeasyBus;
  Drupal.behaviors.speakeasyTts = {
    attach: function (context, settings) {
      once('speakeasy-tts', '.speakeasy-container', context).forEach(function (element) {
        const outputStyle = drupalSettings.speakeasy.outputStyle || 'default';
        const contentText = drupalSettings.speakeasy.content;
        const selectors = Array.isArray(drupalSettings.speakeasy.contentSelectors) ? drupalSettings.speakeasy.contentSelectors : [];

        if (!contentText && selectors.length === 0) {
          element.style.display = 'none';
          return;
        }


        if (!('speechSynthesis' in window) || typeof window.SpeechSynthesisUtterance === 'undefined') {
          Drupal.announce(Drupal.t('Sorry, your browser does not support speech synthesis.'));
          return;
        }

        const synth = window.speechSynthesis;
        const asap = (fn) => (window.queueMicrotask ? queueMicrotask(fn) : setTimeout(fn, 0));


        // ------- State -------
        let voices = [];
        let utteranceQueue = [];
        let currentUtteranceIndex = 0;
        let playerState = PLAYER_STATES.IDLE;
        let cachedSentences = [];
        let lastContent = '';
        let progressSlider;
        let updateProgressSlider = () => {};
        let keepAliveTimer = null; // Chrome auto-pause workaround
        let engineUnlocked = false;

        // Mark Drupal fields so you can target them later if desired.
        if (Drupal.speakeasyUtils && Drupal.speakeasyUtils.markSpeakeasyFields) {
          Drupal.speakeasyUtils.markSpeakeasyFields();
        }

        // ------- Helpers -------
        const sentenceEnd = /[.!?]+[\])'"`’”]*\s*/;

        function splitSentencesFromContainer(container) {
          const sentences = [];
          const walker = document.createTreeWalker(container, NodeFilter.SHOW_TEXT, {
            acceptNode: (node) => node.textContent.trim() ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_REJECT,
          });
          let buffer = '';
          let node;
          while ((node = walker.nextNode())) {
            const text = node.textContent;
            for (let i = 0; i < text.length; i++) {
              buffer += text[i];
              if (sentenceEnd.test(buffer)) {
                sentences.push(buffer.trim());
                buffer = '';
              }
            }
          }
          if (buffer.trim() !== '') sentences.push(buffer.trim());
          return sentences;
        }

        function getSentencesFromSelectors() {
          const containerSet = new Set();
          selectors.forEach((sel) => {
            try {
              document.querySelectorAll(sel).forEach((el) => containerSet.add(el));
            } catch (e) { /* ignore invalid selectors */ }
          });
          const containers = Array.from(containerSet);
          let sentences = [];
          containers.forEach((c) => { sentences = sentences.concat(splitSentencesFromContainer(c)); });
          return sentences;
        }

        // Break long sentences into Chrome-safe chunks (~220 chars).
        function chunkForChrome(text, maxLen = 220) {
          const chunks = [];
          const softBreak = /[,;:—-]\s/g;
          const hardBreak = /\s/g;
          let remaining = text.trim();
          while (remaining.length > maxLen) {
            let splitAt = -1;
            const windowText = remaining.slice(0, maxLen + 1);
            // Prefer a soft break near the end
            const softIdx = windowText.lastIndexOf(','); // quick path
            const punctMatch = windowText.match(/.*([,;:—-])\s.*$/);
            if (punctMatch) {
              splitAt = windowText.lastIndexOf(punctMatch[1] + ' ');
            } else {
              softBreak.lastIndex = 0;
              if (softBreak.test(windowText)) {
                softBreak.lastIndex = 0;
                // fallback if regex not caught above
                const softAll = [...windowText.matchAll(softBreak)];
                if (softAll.length) splitAt = softAll[softAll.length - 1].index + 2;
              }
            }
            if (splitAt < 0) {
              // then any whitespace
              hardBreak.lastIndex = 0;
              const wsAll = [...windowText.matchAll(hardBreak)];
              if (wsAll.length) splitAt = wsAll[wsAll.length - 1].index + 1;
            }
            if (splitAt < 0) splitAt = maxLen; // hard cut
            chunks.push(remaining.slice(0, splitAt).trim());
            remaining = remaining.slice(splitAt).trim();
          }
          if (remaining) chunks.push(remaining);
          return chunks;
        }

        function prepareSentences() {
          let sentences = [];
          if (contentText) {
            sentences = contentText.match(/[^.!?]+[.!?]+[\])'"`’”]*|.+$/g) || [];
          } else {
            sentences = getSentencesFromSelectors();
          }
          const combined = sentences.join(' ');
          if (combined === lastContent && cachedSentences.length) return;
          lastContent = combined;
          cachedSentences = sentences;
          utteranceQueue = [];
        }

        const schedulePrepare = (() => {
          let timeout;
          return () => {
            clearTimeout(timeout);
            timeout = setTimeout(prepareSentences, 50);
          };
        })();

        // Load voices using shared service and refresh selector when new voices appear.
        async function loadVoices() {
          const { voices: loadedVoices, defaultVoice } = await Drupal.speakeasyVoice.loadVoices();
          const changed = loadedVoices.length !== voices.length || loadedVoices.some((v, i) => v.name !== (voices[i] && voices[i].name));
          voices = loadedVoices;
          const voiceSelect = element.querySelector('#speakeasy-voice-select');
          if (voiceSelect && voices.length > 0 && changed) {
            voiceSelect.innerHTML = '';
            voices.forEach((voice) => {
              const option = document.createElement('option');
              option.value = voice.name;
              option.textContent = `${voice.name} (${voice.lang})`;
              voiceSelect.appendChild(option);
            });
            if (defaultVoice && voices.some((v) => v.name === defaultVoice)) {
              voiceSelect.value = defaultVoice;
            }
            voiceSelect.onchange = () => {
              Drupal.speakeasyVoice.applyVoice(voiceSelect.value);
            };
          }
          if (defaultVoice && voices.some((v) => v.name === defaultVoice)) {
            drupalSettings.speakeasy.voiceName = defaultVoice;
          }
        }

        function getSelectedVoice() {
          const voiceSelect = element.querySelector('#speakeasy-voice-select');
          const selectedVoiceName = voiceSelect ? voiceSelect.value : drupalSettings.speakeasy.voiceName;
          return selectedVoiceName ? voices.find((voice) => voice.name === selectedVoiceName) : null;
        }

        function getSpeed() {
          const speedInput = element.querySelector('#speakeasy-speed');
          return speedInput ? (parseFloat(speedInput.value) || 1) : (parseFloat(drupalSettings.speakeasy.speed) || 1);
        }

        function createUtterances() {
          prepareSentences();
          if (utteranceQueue.length) return;
          // Expand each sentence into Chrome-safe chunks.
          utteranceQueue = cachedSentences.flatMap(s => chunkForChrome(s.trim())).filter(Boolean);
        }

        // Keep-alive: Chrome may auto-pause on long reads / background tabs.
        function startKeepAlive() {
          stopKeepAlive();
          keepAliveTimer = setInterval(() => {
            try {
              if (playerState === PLAYER_STATES.PLAYING) {
                // Only resume when actively playing to respect user pauses.
                synth.resume();
              }
            } catch (e) {}
          }, 1000);
        }
        function stopKeepAlive() {
          if (keepAliveTimer) {
            clearInterval(keepAliveTimer);
            keepAliveTimer = null;
          }
        }

        // ------- Playback control -------
        const voicesReady = loadVoices();
        // Prepare immediately so the first chunk is ready at tap.
       prepareSentences();
        schedulePrepare();

        bus.subscribe('speakeasy:voiceChanged', (e) => {
          const voiceName = e.detail && e.detail.voiceName;
          drupalSettings.speakeasy.voiceName = voiceName;
          const voiceSelect = element.querySelector('#speakeasy-voice-select');
          if (voiceSelect) {
            voiceSelect.value = voiceName;
          }
        });

        function startSpeech() {
          if (playerState !== PLAYER_STATES.IDLE) return;
          
          // Some browsers only expose voices after a user gesture.
          voicesReady.then(loadVoices);
          playerState = PLAYER_STATES.PLAYING;
          currentUtteranceIndex = 0;
          // Start now; don't await voices. Apply voice when available.
          createUtterances();
          try { synth.cancel(); } catch (e) {}
          if (progressSlider) progressSlider.value = 0;
          try { synth.resume(); } catch (e) {}
          asap(speakQueue);
          updateLinkState();
        }

        function speakQueue() {
          if (playerState !== PLAYER_STATES.PLAYING) {
            return;
          }
          updateProgressSlider();
          if (currentUtteranceIndex < utteranceQueue.length) {
            const utteranceText = utteranceQueue[currentUtteranceIndex];
            const utterance = new SpeechSynthesisUtterance(utteranceText);
            const v = getSelectedVoice();
            if (v) utterance.voice = v; // otherwise use browser default for instant start
             
            utterance.rate = getSpeed();

            // Update progress more smoothly where supported
            utterance.onboundary = () => updateProgressSlider();
            utterance.onstart = () => {
              bus.dispatch(EVENTS.SENTENCE_START, { index: currentUtteranceIndex });
            };
            utterance.onpause = () => {
              playerState = PLAYER_STATES.PAUSED;
              bus.dispatch(EVENTS.PAUSE);
              updateLinkState();
            };
            utterance.onresume = () => {
              playerState = PLAYER_STATES.PLAYING;
              bus.dispatch(EVENTS.RESUME);
              updateLinkState();
            };
            utterance.onend = () => {
              if (playerState === PLAYER_STATES.PLAYING) {
                currentUtteranceIndex++;
                speakQueue();
              }
            };
            utterance.onerror = () => {
              // Skip bad chunk and continue if still playing.
              currentUtteranceIndex++;
              if (playerState === PLAYER_STATES.PLAYING) {
                speakQueue();
              }
            };

            // Start keep-alive as soon as we speak the first chunk
            if (currentUtteranceIndex === 0) startKeepAlive();

            synth.speak(utterance);
          } else {
            playerState = PLAYER_STATES.IDLE;
            currentUtteranceIndex = 0;
            stopKeepAlive();
            bus.dispatch(EVENTS.STOP);
            updateProgressSlider();
            updateLinkState();
          }
          updateLinkState();
        }

        function hardStop() {
          if (playerState === PLAYER_STATES.IDLE && !synth.speaking) return;
          // Update UI instantly for perceived latency win.
          playerState = PLAYER_STATES.IDLE;
          currentUtteranceIndex = 0;
          stopKeepAlive();
          bus.dispatch(EVENTS.STOP);
          if (progressSlider) progressSlider.value = 0;
          updateLinkState();
          // Engine hammer: cancel → (pause+cancel) → silent flush.
          try { synth.cancel(); } catch (e) {}
          asap(() => {
            if (synth.speaking) {
              try { synth.pause(); synth.cancel(); } catch (e) {}
            }
            try {
              const u = new SpeechSynthesisUtterance(' ');
              u.volume = 0; u.rate = 10;
              synth.speak(u);
              synth.cancel();
            } catch (e) {}
          });
        }

        function updateLinkState() {
          const link = element.querySelector('#speakeasy-link');
          if (link) {
            if (playerState === PLAYER_STATES.IDLE) {
              link.textContent = Drupal.t('Listen');
              link.setAttribute('aria-pressed', 'false');
            } else {
              link.textContent = Drupal.t('Stop Listening');
              link.setAttribute('aria-pressed', 'true');
            }
            link.classList.toggle('speaking', playerState !== PLAYER_STATES.IDLE);
          }
          const ttsBtn = element.querySelector('#speakeasy-tts-button');
          if (ttsBtn) {
            const listenText = ttsBtn.dataset.listenText || Drupal.t('Listen');
            if (playerState === PLAYER_STATES.PLAYING) {
              ttsBtn.textContent = Drupal.t('Pause');
              ttsBtn.setAttribute('aria-pressed', 'true');
            } else if (playerState === PLAYER_STATES.PAUSED) {
              ttsBtn.textContent = Drupal.t('Resume');
              ttsBtn.setAttribute('aria-pressed', 'true');
            } else {
              ttsBtn.textContent = listenText;
              ttsBtn.setAttribute('aria-pressed', 'false');
            }
            ttsBtn.classList.toggle('speaking', playerState !== PLAYER_STATES.IDLE);
          }
          const playBtn = element.querySelector('#speakeasy-play-button');
          const pauseBtn = element.querySelector('#speakeasy-pause-button');
          const stopBtn = element.querySelector('#speakeasy-stop-button');
          if (playBtn) {
            playBtn.setAttribute('aria-label', playerState === PLAYER_STATES.PAUSED ? Drupal.t('Resume') : Drupal.t('Play'));
            playBtn.classList.toggle('speakeasy-playing', playerState === PLAYER_STATES.PLAYING);
            playBtn.setAttribute('aria-pressed', playerState === PLAYER_STATES.PLAYING ? 'true' : 'false');
          }
          if (pauseBtn) {
            pauseBtn.setAttribute('aria-label', playerState === PLAYER_STATES.PAUSED ? Drupal.t('Resume') : Drupal.t('Pause'));
            pauseBtn.classList.toggle('speakeasy-paused', playerState === PLAYER_STATES.PAUSED);
            pauseBtn.disabled = playerState === PLAYER_STATES.IDLE;
            pauseBtn.setAttribute('aria-pressed', playerState === PLAYER_STATES.PAUSED ? 'true' : 'false');
          }
          if (stopBtn) {
            stopBtn.disabled = playerState === PLAYER_STATES.IDLE;
          }
        }

        // ------- UI wiring -------
        function addPressHandler(button, handler) {
          if (!button) return;
          const wrapped = (e) => {
            e.preventDefault();
            handler(e);
          };
          // Trigger on earliest gesture; avoid duplicate click.
          if (window.PointerEvent) {
            button.addEventListener('pointerdown', wrapped, { passive: false });
          } else if ('ontouchstart' in window) {
            button.addEventListener('touchstart', wrapped, { passive: false });
            button.addEventListener('mousedown', wrapped);
          } else {
            button.addEventListener('mousedown', wrapped);
          }
        }

        if (outputStyle === 'media_player') {
          const playButton = element.querySelector('#speakeasy-play-button');
          const pauseButton = element.querySelector('#speakeasy-pause-button');
          const stopButton = element.querySelector('#speakeasy-stop-button');
          const settingsToggle = element.querySelector('#speakeasy-settings-toggle');
          const settingsContainer = element.querySelector('#speakeasy-settings');
          progressSlider = element.querySelector('#speakeasy-progress');

          updateProgressSlider = () => {
            if (progressSlider && utteranceQueue.length > 0) {
              const progress = (currentUtteranceIndex / utteranceQueue.length) * 100;
              progressSlider.value = progress;
            }
          };

          if (settingsToggle && settingsContainer) {
            settingsToggle.addEventListener('click', function () {
              settingsContainer.style.display = (settingsContainer.style.display === 'none') ? 'flex' : 'none';
            });
          }

          addPressHandler(playButton, function () {
            
            if (playerState === PLAYER_STATES.PAUSED) {
              synth.resume();
              playerState = PLAYER_STATES.PLAYING;
              bus.dispatch(EVENTS.RESUME);
              updateLinkState();
            } else if (playerState === PLAYER_STATES.IDLE) {
              startSpeech();
            }
          });

          addPressHandler(pauseButton, function () {
            
            if (playerState === PLAYER_STATES.PLAYING) {
              synth.pause();
              playerState = PLAYER_STATES.PAUSED;
              bus.dispatch(EVENTS.PAUSE);
              updateLinkState();
            } else if (playerState === PLAYER_STATES.PAUSED) {
              synth.resume();
              playerState = PLAYER_STATES.PLAYING;
              bus.dispatch(EVENTS.RESUME);
              updateLinkState();
            }
          });

          addPressHandler(stopButton, function () {
            if (playerState !== PLAYER_STATES.IDLE) hardStop();
          });

          if (progressSlider) {
            progressSlider.addEventListener('input', function () {
              if (utteranceQueue.length > 0) {
                const newIndex = Math.round((progressSlider.value / 100) * utteranceQueue.length);
                currentUtteranceIndex = Math.min(newIndex, utteranceQueue.length - 1);
                if (playerState !== PLAYER_STATES.IDLE && !synth.paused) {
                  synth.cancel();
                  requestAnimationFrame(speakQueue);
                }
              }
            });
          }
        }
        else if (outputStyle === 'default') {
          const ttsButton = element.querySelector('#speakeasy-tts-button');
          const stopButton = element.querySelector('#speakeasy-stop-button');

          if (ttsButton) {
            ttsButton.dataset.listenText = ttsButton.textContent;
            addPressHandler(ttsButton, function () {
              
              if (playerState === PLAYER_STATES.PLAYING) {
                synth.pause();
                playerState = PLAYER_STATES.PAUSED;
                bus.dispatch(EVENTS.PAUSE);
                updateLinkState();
              } else if (playerState === PLAYER_STATES.PAUSED) {
                synth.resume();
                playerState = PLAYER_STATES.PLAYING;
                bus.dispatch(EVENTS.RESUME);
                updateLinkState();
              } else if (playerState === PLAYER_STATES.IDLE) {
                startSpeech();
              }
            });
          }
          addPressHandler(stopButton, function () {
            if (playerState !== PLAYER_STATES.IDLE) hardStop();
          });
        }
        else if (outputStyle === 'simple_link') {
          const link = element.querySelector('#speakeasy-link');
          if (link) {
            link.addEventListener('click', function (e) {
              e.preventDefault();
              
              if (playerState === PLAYER_STATES.IDLE) {
                startSpeech();
              } else {
                hardStop();
              }
            });
            link.addEventListener('keydown', function (e) {
              if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                link.click();
              }
            });
          }
        }

        updateLinkState();

        // Keyboard shortcuts (space = toggle, s = stop) – ignore when typing
        once('speakeasy-keyboard-shortcuts', 'body').forEach(function () {
          document.addEventListener('keydown', function (e) {
            const a = document.activeElement;
            if (a && (a.tagName === 'INPUT' || a.tagName === 'TEXTAREA' || a.tagName === 'SELECT' || a.isContentEditable)) return;

            if (e.code === 'Space' || e.key === ' ') {
              e.preventDefault();
              
              if (playerState === PLAYER_STATES.IDLE) { startSpeech(); } else { hardStop(); }
            } else if (e.key && e.key.toLowerCase() === 's') {
              e.preventDefault();
              hardStop();
            }
          });
        });

        // Handle tab visibility changes (Chrome can pause on hide)
        document.addEventListener('visibilitychange', function () {
          if (document.visibilityState === 'visible' && playerState === PLAYER_STATES.PLAYING && synth.paused) {
            try {
              synth.resume();
              bus.dispatch(EVENTS.RESUME);
            } catch (e) {}
          }
        });

        // Stop on page unload/navigation
        window.addEventListener('pagehide', hardStop);
        window.addEventListener('beforeunload', hardStop);
      });
    },
  };
})(Drupal, drupalSettings, once);
