(function (Drupal, drupalSettings) {
  'use strict';

  function normalizeLang(lang) {
    return (lang || '').replace(/_/g, '-').toLowerCase();
  }

  function getBrowser() {
    const ua = navigator.userAgent.toLowerCase();
    if (ua.includes('edg')) return 'edge';
    if (ua.includes('chrome')) return 'chrome';
    if (ua.includes('firefox')) return 'firefox';
    if (ua.includes('safari')) return 'safari';
    return 'other';
  }

  function waitForVoices(pollMs = 50) {
    return new Promise((resolve) => {
      const synth = window.speechSynthesis;
      let resolved = false;
      const tryResolve = () => {
        if (resolved) return;
        const v = synth.getVoices();
        if (v && v.length) {
          resolved = true;
          synth.removeEventListener('voiceschanged', onVoices);
          clearInterval(poll);
          resolve(v);
        }
      };
      const onVoices = () => tryResolve();
      synth.addEventListener('voiceschanged', onVoices);

      const poll = setInterval(() => {
        tryResolve();
      }, pollMs);

      tryResolve();
    });
  }

  let _voicesCache = null;
  let _voicesChangedListener = null;

  function rebuildVoiceSelector() {
    if (!_voicesCache) return;
    const select = document.querySelector('#speakeasy-user-voice-select');
    if (!select) return;
    const { voices, defaultVoice } = _voicesCache;
    select.innerHTML = '';
    const defaultOption = document.createElement('option');
    defaultOption.value = '';
    defaultOption.textContent = Drupal.t('- Browser default -');
    select.appendChild(defaultOption);
    voices.forEach((voice) => {
      const option = document.createElement('option');
      option.value = voice.name;
      option.textContent = `${voice.name} (${voice.lang})`;
      select.appendChild(option);
    });
    if (defaultVoice && voices.some(v => v.name === defaultVoice)) {
      select.value = defaultVoice;
    }
  }

  Drupal.speakeasyVoice = {
    async loadVoices() {
      if (_voicesCache) return _voicesCache;
      if (!('speechSynthesis' in window)) {
        _voicesCache = { voices: [], defaultVoice: '' };
        return _voicesCache;
      }
      let voices = await waitForVoices();
      if (!voices.length) {
        if (!_voicesChangedListener) {
          const synth = window.speechSynthesis;
          _voicesChangedListener = async () => {
            await Drupal.speakeasyVoice.loadVoices();
            if (_voicesCache && _voicesCache.voices.length) {
              synth.removeEventListener('voiceschanged', _voicesChangedListener);
              _voicesChangedListener = null;
            }
          };
          synth.addEventListener('voiceschanged', _voicesChangedListener);
        }
        return { voices: [], defaultVoice: '' };
      }
      // Limit to allowed languages first.
      const allowedLangs = Array.isArray(drupalSettings.speakeasy.allowedLanguages)
        ? drupalSettings.speakeasy.allowedLanguages
        : [];
      if (allowedLangs.length > 0) {
        voices = voices.filter(v => {
          const voiceLang = normalizeLang(v.lang);
          const base = voiceLang.split('-')[0];
          return allowedLangs.includes(base) || allowedLangs.includes(voiceLang);
        });
      }

      // Apply voice whitelisting per browser and integrate with allowed languages.
      const allowedMap = drupalSettings.speakeasy.allowedVoices || {};
      const browser = getBrowser();
      const allowedList = Array.isArray(allowedMap[browser]) ? allowedMap[browser] : [];
      if (allowedList.length > 0) {
        voices = voices.filter(v => allowedList.includes(v.name));
      }

      const langCode = drupalSettings.speakeasy.language || document.documentElement.lang || navigator.language || '';
      const normalizedLangCode = normalizeLang(langCode);
      const baseLangCode = normalizedLangCode.split('-')[0];

      let defaultVoiceName = '';
      const userPrefVoice = drupalSettings.speakeasy.userPreferences && drupalSettings.speakeasy.userPreferences.voiceName;
      if (userPrefVoice && voices.some(v => v.name === userPrefVoice)) {
        defaultVoiceName = userPrefVoice;
      } else if (normalizedLangCode) {
        const match = voices.find((v) => {
          const voiceLang = normalizeLang(v.lang);
          return voiceLang && (voiceLang === normalizedLangCode || voiceLang.split('-')[0] === baseLangCode);
        });
        if (match) defaultVoiceName = match.name;
      }
      if (!defaultVoiceName) defaultVoiceName = drupalSettings.speakeasy.voiceName;
      _voicesCache = { voices, defaultVoice: defaultVoiceName };
      rebuildVoiceSelector();
      return _voicesCache;
    },

    applyVoice(voiceName) {
      drupalSettings.speakeasy.voiceName = voiceName;
      Drupal.speakeasyBus.dispatch('speakeasy:voiceChanged', { voiceName });
    }
  };
})(Drupal, drupalSettings);
