<?php

namespace Drupal\speakeasy\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Speakeasy settings for this site.
 */
class SpeakeasySettingsForm extends ConfigFormBase {

  /**
   * Logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * The language manager service.
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs a SpeakeasySettingsForm object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, LoggerChannelFactoryInterface $logger_factory, LanguageManagerInterface $language_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->logger = $logger_factory->get('speakeasy');
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('logger.factory'),
      $container->get('language_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'speakeasy_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['speakeasy.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('speakeasy.settings');
    $allowed_voices = $config->get('allowed_voices_by_browser') ?: [];
    $allowed_languages = $config->get('allowed_languages') ?: [];

    $languages = $this->languageManager->getLanguages();
    $language_options = [];
    foreach ($languages as $langcode => $language) {
      $language_options[$langcode] = $language->getName();
    }

    $this->logger->debug('Loaded configuration on form build', [
      'allowed_voices' => $allowed_voices,
    ]);
    $form['settings_tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Speakeasy settings'),
    ];

    $browsers = [
      'chrome' => $this->t('Google Chrome'),
      'edge' => $this->t('Microsoft Edge'),
      'firefox' => $this->t('Mozilla Firefox'),
      'safari' => $this->t('Apple Safari'),
      'other' => $this->t('Other Browsers'),
    ];

    // Block specific options.
    $form['block_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Block options'),
      '#group' => 'settings_tabs',
    ];

    $form['block_options']['theme'] = [
      '#type' => 'select',
      '#title' => $this->t('Theme'),
      '#description' => $this->t('Select the style theme for Speakeasy output.'),
      '#options' => [
        'none' => $this->t('None'),
        'default' => $this->t('Default'),
        'olivero' => $this->t('Olivero'),
      ],
      '#default_value' => $config->get('theme') ?: 'default',
    ];

    $form['block_options']['allow_voice_selection'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow voice selection'),
      '#description' => $this->t('Allow Speakeasy blocks to display the voice selection dropdown.'),
      '#default_value' => $config->get('allow_voice_selection') ?? TRUE,
    ];

    $form['block_options']['allow_highlighting'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow highlighting'),
      '#description' => $this->t('Allow Speakeasy blocks to highlight text as it is spoken.'),
      '#default_value' => $config->get('allow_highlighting') ?? TRUE,
    ];

    $form['block_options']['default_speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Speed'),
      '#description' => $this->t('Default speech speed (0.1 to 2).'),
      '#default_value' => $config->get('default_speed') ?: 1,
      '#step' => 0.1,
      '#min' => 0.1,
      '#max' => 2,
    ];
    
      // Allowed voices grouped by browser.
    $form['allowed_voices_by_browser'] = [
      '#type' => 'details',
      '#title' => $this->t('Voice Whitelisting by Browser'),
      '#description' => $this->t('Enter comma-separated voice names allowed for each browser. Leave blank to allow all voices.'),
      '#group' => 'settings_tabs',
    ];

    foreach ($browsers as $key => $label) {
      $allowed_list = isset($allowed_voices[$key]) ? implode(', ', $allowed_voices[$key]) : '';
      $form['allowed_voices_by_browser']['allowed_' . $key] = [
        '#type' => 'textfield',
        '#title' => $label,
        '#default_value' => $allowed_list,
        '#description' => $this->t('Comma-separated list of allowed voices for @browser.', [
          '@browser' => $label,
        ]),
      ];
    }

    $form['language_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Language options'),
      '#description' => $this->t('Select languages that are allowed for voices. Leave all unchecked to allow all languages.'),
      '#group' => 'settings_tabs',
    ];

    $form['language_options']['allowed_languages'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Languages'),
      '#options' => $language_options,
      '#default_value' => $allowed_languages,
      '#attributes' => ['class' => ['speakeasy-language-checkboxes']],
    ];

    $form['language_options']['select_all_languages'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Select all languages'),
      '#attributes' => ['class' => ['speakeasy-select-all-languages']],
    ];

    $form['language_options']['#attached']['library'][] = 'speakeasy/speakeasy.admin';

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $allowed_by_browser = [];
    $browsers = ['chrome', 'edge', 'firefox', 'safari', 'other'];
    $allowed_languages = array_filter($form_state->getValue('allowed_languages') ?? []);

    $all_values = $form_state->getValues();
    $this->logger->debug('All submitted form state values', [
      'values' => $all_values,
    ]);

    foreach ($browsers as $browser) {
      $allowed_input = $form_state->getValue('allowed_' . $browser) ?? '';
      $allowed_list = array_filter(array_map('trim', explode(',', $allowed_input)));
      if (!empty($allowed_list)) {
        $allowed_by_browser[$browser] = array_values($allowed_list);
      }
    }

    $this->logger->debug('Processed allowed voices to save', [
      'allowed' => $allowed_by_browser,
    ]);

    $config = $this->config('speakeasy.settings');
    $config
      ->set('allowed_voices_by_browser', $allowed_by_browser)
      ->set('allowed_languages', array_values($allowed_languages))
      ->set('default_speed', $form_state->getValue('default_speed'))
      ->set('theme', $form_state->getValue('theme'))
      ->set('allow_voice_selection', (bool) $form_state->getValue('allow_voice_selection'))
      ->set('allow_highlighting', (bool) $form_state->getValue('allow_highlighting'))
      ->save();

    $saved_config = $config->get();
    $this->logger->debug('Saved configuration', [
      'config' => $saved_config,
    ]);

    $this->messenger()->addStatus($this->t('Speakeasy settings have been saved.'));

    parent::submitForm($form, $form_state);
  }

}
