# SPhoenix AI Content Assistant

AI-powered content generation and optimization module for Drupal 10+ with advanced chatbot interface, field mapping, and comprehensive analytics.

## Table of Contents

- [Overview](#overview)
- [Features](#features)
- [Requirements](#requirements)
- [Installation](#installation)
- [Configuration](#configuration)
- [API Documentation](#api-documentation)
- [Permissions](#permissions)
- [Usage Examples](#usage-examples)
- [Services](#services)
- [Troubleshooting](#troubleshooting)
- [Contributing](#contributing)
- [License](#license)

## Overview

The SPhoenix AI Content Assistant module integrates powerful AI tools directly into Drupal's content editing workflow. It provides a chatbot interface that helps content creators generate, optimize, and analyze content using advanced AI models.

## Features

- **AI-Powered Content Generation**: Create articles, blog posts, and other content types
- **Content Optimization**: SEO analysis and content improvement suggestions
- **Image Generation**: AI-generated thumbnails and graphics
- **Data Visualization**: Charts, graphs, and tables
- **Multi-language Support**: Generate content in multiple languages
- **Field Mapping**: Intelligent mapping of AI responses to Drupal fields
- **Usage Analytics**: Comprehensive tracking and reporting
- **CKEditor Integration**: Seamless integration with Drupal's rich text editor
- **Subscription Management**: Built-in usage tracking and plan management

## Requirements

- **Drupal**: 10.x or 11.x
- **PHP**: 8.0 or higher
- **Dependencies**: 
  - `drupal:node`
  - `drupal:field`
  - `drupal:user`
  - `drupal:system`
- **SPhoenix AI Account**: Active API subscription

## Installation

### Composer Installation (Recommended)

```bash
composer require drupal/sphoenix_ai
drush en sphoenix_ai
drush cr
```

### Manual Installation

1. Download the module to `modules/contrib/sphoenix_ai`
2. Enable the module: `drush en sphoenix_ai`
3. Clear cache: `drush cr`

### Post-Installation Setup

1. Configure permissions at `/admin/people/permissions#module-sphoenix_ai`
2. Configure module settings at `/admin/config/content/sphoenix-ai`
3. Enable content types at `/admin/config/content/sphoenix-ai/content-types`
4. Configure field mappings at `/admin/config/content/sphoenix-ai/field-mapping`

## Configuration

### Configuration Routes

| Route | URL | Purpose |
|-------|-----|---------|
| **Main Settings** | `/admin/config/content/sphoenix-ai` | Core module configuration |
| **Content Types** | `/admin/config/content/sphoenix-ai/content-types` | Enable/disable for content types |
| **Field Mapping** | `/admin/config/content/sphoenix-ai/field-mapping` | Configure field mappings |
| **Usage Statistics** | `/admin/reports/sphoenix-ai` | View usage analytics |

### Main Settings Configuration

#### Default AI Parameters
```yaml
default_temperature: 0.7          # AI response randomness (0.0-1.0)
default_tone: 'conversational'    # Content tone
default_word_length: 600          # Target word count
default_paragraphs: 4            # Number of paragraphs
include_summary: true            # Include TL;DR sections
include_faq: false              # Include FAQ sections
```

#### Interface Settings
```yaml
chatbot_position: 'right'        # Chatbot position: right|left|bottom
chatbot_theme: 'default'         # Theme: default|dark|light|minimal
auto_expand_chatbot: false       # Auto-expand on page load
auto_suggest_tasks: true         # Show task suggestions
```

#### SEO Configuration
```yaml
default_keywords:               # Default keywords (line-separated)
  - 'content marketing'
  - 'SEO optimization'
competitor_urls:               # Competitor analysis URLs
  - 'https://competitor1.com'
  - 'https://competitor2.com'
```

### Content Types Configuration

Enable AI assistant for specific content types:

```php
// Example configuration
$config['sphoenix_ai.settings']['enabled_content_types'] = [
  'article',
  'blog_post',
  'page'
];
```

### Field Mapping

Configure how AI responses map to Drupal fields:

| AI Field | Description | Example Drupal Field |
|----------|-------------|---------------------|
| `title` | Generated titles | `title` |
| `html_content` | Rich text content | `body` |
| `meta_title` | SEO titles | `field_meta_title` |
| `meta_description` | Meta descriptions | `field_meta_description` |
| `keywords` | Tags/keywords | `field_tags` |
| `url_alias` | URL aliases | `path` |

## API Documentation

### Authentication Endpoints

#### Get Authentication Status
```http
GET /ai-assistant/auth/status
Authorization: Bearer {drupal_session}
```

**Response:**
```json
{
  "success": true,
  "data": {
    "isAuthenticated": true,
    "user": {
      "id": "user123",
      "email": "user@example.com",
      "firstname": "John"
    },
    "subscription": {
      "plan_name": "Pro",
      "remaining_usage": 450
    }
  }
}
```

#### Refresh Authentication
```http
POST /ai-assistant/auth/refresh
Content-Type: application/json
X-CSRF-Token: {csrf_token}
```

### Main Chat Endpoint

```http
POST /ai-assistant/chat
Content-Type: application/json
X-CSRF-Token: {csrf_token}
```

**Request Body:**
```json
{
  "task_type": "content_generation",
  "prompt": "Write an article about sustainable energy",
  "content_type": "article",
  "existing_content": "Optional existing content...",
  "field_data": {
    "title[0][value]": "Current title",
    "body[0][value]": "Current body content"
  }
}
```

### Available Task Types

| Task Type | Description | Use Case |
|-----------|-------------|----------|
| `content_generation` | Create new content | Blog posts, articles |
| `content_regeneration` | Improve existing content | Content updates |
| `content_optimization` | SEO optimization | Search ranking improvement |
| `thumbnail_generation` | Generate images | Article thumbnails |
| `seo_analysis` | Analyze SEO performance | Content audits |
| `keyword_extraction` | Extract keywords | SEO research |
| `meta_generation` | Generate meta tags | Search snippets |
| `table_generation` | Create data tables | Structured data |
| `graph_generation` | Create charts | Data visualization |

### Response Format

```json
{
  "success": true,
  "state": "chat_ready",
  "data": {
    "task_type": "content_generation",
    "content": {
      "title": "Generated title",
      "html_content": "<p>Generated content...</p>",
      "meta_title": "SEO title",
      "meta_description": "SEO description",
      "keywords": ["keyword1", "keyword2"]
    },
    "suggested_actions": [
      "insert_content",
      "copy_content", 
      "regenerate"
    ],
    "processing_time": 2.345
  },
  "usage_info": {
    "remaining_usage": 449,
    "usage_percentage": 10.2
  }
}
```

### File Upload Endpoints

#### Upload Image
```http
POST /ai-assistant/upload-image
Content-Type: multipart/form-data

image: {file}           # Image file (PNG, JPG, GIF, WebP, SVG)
type: "chart"           # Upload type
```

#### Upload from URL
```http
POST /ai-assistant/upload-image-from-url
Content-Type: application/json

{
  "imageUrl": "https://example.com/image.png",
  "type": "ai-image"
}
```

### Usage Statistics

#### Get Usage Data
```http
GET /ai-assistant/usage/data?days=30&type=user
```

#### Export Usage
```http
GET /ai-assistant/usage/export?days=30&format=csv
```

## Permissions

Configure permissions at `/admin/people/permissions#module-sphoenix_ai`:

| Permission | Description | Recommended Roles |
|------------|-------------|-------------------|
| `use ai content assistant` | Access AI features | Content Creator, Editor |
| `administer sphoenix ai` | Configure settings | Site Administrator |
| `view ai content assistant usage` | View analytics | Content Manager, Administrator |

## Usage Examples

### JavaScript Integration

```javascript
// Basic content generation
async function generateContent(prompt, contentType) {
  const response = await fetch('/ai-assistant/chat', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'X-CSRF-Token': drupalSettings.user.csrf_token
    },
    body: JSON.stringify({
      task_type: 'content_generation',
      prompt: prompt,
      content_type: contentType
    })
  });
  
  const result = await response.json();
  if (result.success) {
    return result.data.content;
  }
  throw new Error(result.error);
}

// Usage
generateContent('Write about renewable energy', 'article')
  .then(content => {
    console.log('Generated:', content.title);
    // Insert into form fields
  });
```

### Content Optimization Example

```javascript
// Optimize existing content
async function optimizeContent(existingContent, contentType) {
  const response = await fetch('/ai-assistant/chat', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'X-CSRF-Token': drupalSettings.user.csrf_token
    },
    body: JSON.stringify({
      task_type: 'content_optimization',
      prompt: 'Optimize this content for SEO and readability',
      existing_content: existingContent,
      content_type: contentType
    })
  });
  
  return await response.json();
}
```

### PHP Service Usage

```php
<?php

// Custom module integration
function mymodule_generate_content($prompt) {
  $api_client = \Drupal::service('sphoenix_ai.api_client');
  $auth_service = \Drupal::service('sphoenix_ai.authentication');
  
  if (!$auth_service->isAuthenticated()) {
    throw new \Exception('User not authenticated');
  }
  
  $response = $api_client->makeApiRequest([
    'task_type' => 'content_generation',
    'prompt' => $prompt,
    'content_type' => 'article'
  ]);
  
  return $response;
}
```

## Services

### Available Services

| Service ID | Class | Purpose |
|------------|-------|---------|
| `sphoenix_ai.api_client` | `ApiClientService` | API communication |
| `sphoenix_ai.authentication` | `AuthenticationService` | User authentication |
| `sphoenix_ai.subscription` | `SubscriptionService` | Subscription management |
| `sphoenix_ai.field_mapper` | `FieldMapperService` | Field mapping |
| `sphoenix_ai.usage_tracker` | `UsageTrackerService` | Usage tracking |
| `sphoenix_ai.content_analyzer` | `ContentAnalyzerService` | Content analysis |

### Service Integration

```php
<?php

// Dependency injection in custom service
class MyCustomService {
  
  protected $aiClient;
  protected $fieldMapper;
  
  public function __construct(
    ApiClientService $ai_client,
    FieldMapperService $field_mapper
  ) {
    $this->aiClient = $ai_client;
    $this->fieldMapper = $field_mapper;
  }
  
  public function processContent($content_type, $data) {
    // Use AI services
    $mapping = $this->fieldMapper->getFieldMappingSuggestions($content_type, $data);
    $response = $this->aiClient->makeApiRequest($data);
    
    return $response;
  }
}
```

## Troubleshooting

### Common Issues

#### Chatbot Not Appearing
1. Check content type configuration at `/admin/config/content/sphoenix-ai/content-types`
2. Verify user permissions
3. Clear cache: `drush cr`
4. Check browser console for JavaScript errors

#### Authentication Problems
```bash
# Clear Drupal cache
drush cr

# Check module status
drush pml | grep sphoenix_ai

# Rebuild registry
drush rr
```

#### Content Not Inserting
1. Configure field mappings at `/admin/config/content/sphoenix-ai/field-mapping`
2. Check CKEditor compatibility
3. Verify field permissions

### Debug Mode

Enable in `settings.local.php`:
```php
$config['sphoenix_ai.settings']['debug_mode'] = TRUE;
$config['sphoenix_ai.settings']['log_api_requests'] = TRUE;
```

### Error Codes

| Code | Description | Solution |
|------|-------------|----------|
| `NOT_AUTHENTICATED` | User not logged in | Redirect to login |
| `SUBSCRIPTION_REQUIRED` | No active subscription | Show upgrade options |
| `USAGE_LIMIT_EXCEEDED` | Monthly limit reached | Plan upgrade needed |
| `INVALID_REQUEST` | Malformed request | Check request format |
| `VALIDATION_ERROR` | Missing parameters | Verify required fields |

### Logs

Check logs for debugging:
- Drupal logs: `/admin/reports/dblog`
- Browser console for JavaScript errors
- Server logs: `/var/log/apache2/error.log` or `/var/log/nginx/error.log`

## Contributing

We welcome contributions! Please:

1. Fork the repository
2. Create a feature branch
3. Write tests for new functionality
4. Follow Drupal coding standards
5. Submit a pull request

### Development Setup

```bash
# Clone repository
git clone https://github.com/yourusername/sphoenix_ai.git

# Install dependencies
composer install

# Run tests
./vendor/bin/phpunit

# Code standards check
./vendor/bin/phpcs --standard=Drupal,DrupalPractice .
```

### Coding Standards

- Follow [Drupal coding standards](https://www.drupal.org/docs/develop/standards)
- Use PHP 8+ features appropriately
- Write comprehensive documentation
- Include unit tests for new features

## License

This project is licensed under the GPL-2.0+ License - see the [LICENSE](LICENSE) file for details.

## Support

- **Issue Queue**: [Drupal.org Issue Queue](https://www.drupal.org/project/issues/sphoenix_ai)
- **Documentation**: [Module Documentation](https://www.drupal.org/docs/contributed-modules/sphoenix-ai)
- **Community**: [Drupal Slack #ai-content-assistant](https://drupal.slack.com)

## Changelog

### Version 1.0.0
- Initial release
- Core AI chat functionality
- Field mapping system
- Usage tracking
- CKEditor 5 integration
- Multi-language support

---

**Developed by the SPhoenix AI team** | [Website](https://www.sphoenixai.com) | [Support](https://www.sphoenixai.com/contact)
