/**
 * @file
 * SPhoenix AI Content Assistant chatbot with CKEditor 5 support for Drupal 11.
 */

(function ($, Drupal, drupalSettings) {
  'use strict';

  // Task types enum
  const TaskType = {
    CONTENT_GENERATION: 'content_generation',
    CONTENT_REGENERATION: 'content_regeneration',
    CONTENT_OPTIMIZATION: 'content_optimization',
    THUMBNAIL_GENERATION: 'thumbnail_generation',
    GRAPH_GENERATION: 'graph_generation',
    TABLE_GENERATION: 'table_generation',
    META_GENERATION: 'meta_generation',
    SEO_ANALYSIS: 'seo_analysis',
    CONTENT_ANALYSIS: 'content_analysis',
    KEYWORD_EXTRACTION: 'keyword_extraction',
    CONTENT_SUMMARY: 'content_summary',
    TITLE_GENERATION: 'title_generation',
    DESCRIPTION_GENERATION: 'description_generation',
  };

  // Task configurations
  const taskConfigs = {
    [TaskType.CONTENT_GENERATION]: {
      label: 'Content Generation',
      description: 'Generate new articles and content from scratch',
      icon: '✏️',
      prompt_placeholder: 'Describe what content you want to generate...',
      requires_existing: false,
    },
    [TaskType.CONTENT_REGENERATION]: {
      label: 'Content Regeneration',
      description: 'Regenerate and improve existing content',
      icon: '🔄',
      prompt_placeholder: 'How should the content be improved?',
      requires_existing: true,
    },
    [TaskType.CONTENT_OPTIMIZATION]: {
      label: 'Content Optimization',
      description: 'Optimize content for SEO and readability',
      icon: '📈',
      prompt_placeholder: 'What aspects should be optimized?',
      requires_existing: true,
    },
    [TaskType.THUMBNAIL_GENERATION]: {
      label: 'Thumbnail Generation',
      description: 'Generate images and thumbnails',
      icon: '🖼️',
      prompt_placeholder: 'Describe the image you want to generate...',
      requires_existing: false,
    },
    [TaskType.GRAPH_GENERATION]: {
      label: 'Graph Generation',
      description: 'Create data visualizations and charts',
      icon: '📊',
      prompt_placeholder: 'Describe the chart or graph needed...',
      requires_existing: false,
    },
    [TaskType.TABLE_GENERATION]: {
      label: 'Table Generation',
      description: 'Generate structured data tables',
      icon: '📋',
      prompt_placeholder: 'Describe the table data needed...',
      requires_existing: false,
    },
    [TaskType.META_GENERATION]: {
      label: 'Meta Generation',
      description: 'Generate SEO meta tags and descriptions',
      icon: '🏷️',
      prompt_placeholder: 'Content topic for meta generation...',
      requires_existing: false,
    },
    [TaskType.SEO_ANALYSIS]: {
      label: 'SEO Analysis',
      description: 'Analyze content for SEO performance',
      icon: '🔍',
      prompt_placeholder: 'Any specific SEO aspects to analyze?',
      requires_existing: true,
    },
    [TaskType.CONTENT_ANALYSIS]: {
      label: 'Content Analysis',
      description: 'Analyze content quality and readability',
      icon: '📖',
      prompt_placeholder: 'What aspects to analyze?',
      requires_existing: true,
    },
    [TaskType.KEYWORD_EXTRACTION]: {
      label: 'Keyword Extraction',
      description: 'Extract and suggest relevant keywords',
      icon: '🔑',
      prompt_placeholder: 'Optional: specify keyword focus...',
      requires_existing: true,
    },
    [TaskType.CONTENT_SUMMARY]: {
      label: 'Content Summary',
      description: 'Create content summaries and abstracts',
      icon: '📄',
      prompt_placeholder: 'Any specific summary requirements?',
      requires_existing: true,
    },
    [TaskType.TITLE_GENERATION]: {
      label: 'Title Generation',
      description: 'Generate compelling titles and headlines',
      icon: '📰',
      prompt_placeholder: 'Topic or content for title generation...',
      requires_existing: false,
    },
    [TaskType.DESCRIPTION_GENERATION]: {
      label: 'Description Generation',
      description: 'Create descriptions for various purposes',
      icon: '📝',
      prompt_placeholder: 'What needs a description?',
      requires_existing: false,
    },
  };

  /**
   * AI Content Assistant chatbot class.
   */
  class AIContentAssistant {
    constructor() {
      this.isInitialized = false;
      this.isExpanded = false;
      this.isProcessing = false;
      this.selectedTaskType = null;
      this.contentType = null;
      this.chatHistory = [];
      this.fieldMappings = {};
      this.authState = {
        authenticated: false,
        user: null,
        subscription: null,
      };
      this.subscriptionInfo = null;

      this.init();
    }

    async init() {
      if (this.isInitialized) return;

      if (!this.isEnabledPage()) {
        return;
      }

      this.detectContentType();
      await this.checkAuthenticationStatus();
      this.createChatbotInterface();
      this.attachEventListeners();
      this.isInitialized = true;

      if (drupalSettings.aiContentAssistant?.autoExpand) {
        setTimeout(() => this.expandChatbot(), 500);
      }

      this.checkInitialState();
    }

    isEnabledPage() {
      const enabledTypes = drupalSettings.aiContentAssistant?.enabledContentTypes || [];
      const currentType = this.detectContentType();
      return enabledTypes.includes(currentType) && $('form.node-form').length > 0;
    }

    detectContentType() {
      if (this.contentType) return this.contentType;

      const $form = $('form.node-form');
      if ($form.length) {
        const formId = $form.attr('id');
        const matches = formId.match(/node-(\w+)-(?:form|edit-form)/);
        if (matches) {
          this.contentType = matches[1].replace(/-/g, '_');
          return this.contentType;
        }
      }

      const path = window.location.pathname;
      const urlMatches = path.match(/node\/add\/(\w+)|node\/(\d+)\/edit/);
      if (urlMatches && urlMatches[1]) {
        this.contentType = urlMatches[1].replace(/-/g, '_');
      }

      return this.contentType;
    }

    async checkAuthenticationStatus() {
      try {
        const response = await fetch('/ai-assistant/auth/status', {
          method: 'GET',
          headers: { 'Content-Type': 'application/json' }
        });

        if (!response.ok) {
          this.authState = { authenticated: false, user: null, subscription: null };
          return;
        }

        const result = await response.json();

        if (result.success && result.data) {
          this.authState = {
            authenticated: result.data.isAuthenticated || false,
            user: result.data.user || null,
            subscription: result.data.subscription || null,
          };

          if (this.authState.authenticated && this.authState.subscription) {
            const hasCmsSubscription = this.checkCmsSubscription(this.authState.subscription);
            if (!hasCmsSubscription) {
              this.authState.subscription = null;
            }
          }

          if (this.authState.authenticated && this.authState.subscription) {
            await this.getSubscriptionStatus();
          }
        } else {
          this.authState = { authenticated: false, user: null, subscription: null };
        }
      } catch (error) {
        console.warn('Failed to check authentication status:', error);
        this.authState = { authenticated: false, user: null, subscription: null };
      }
    }

    checkCmsSubscription(subscription) {
      if (!subscription || !subscription.all_subscriptions) return false;
      return subscription.all_subscriptions.some(sub =>
        sub.status === 'active' && sub.plan_id && sub.plan_id.category === 'cms'
      );
    }

    async getSubscriptionStatus() {
      try {
        const response = await fetch('/ai-assistant/subscription/status');
        const result = await response.json();
        if (result.success) {
          this.subscriptionInfo = result.data;
        }
      } catch (error) {
        console.warn('Failed to get subscription status:', error);
      }
    }

    createChatbotInterface() {
      const position = drupalSettings.aiContentAssistant?.chatbotPosition || 'right';
      const theme = drupalSettings.aiContentAssistant?.chatbotTheme || 'default';
      const logoUrl = drupalSettings.aiContentAssistant?.logoUrl;
      const logoHtml = logoUrl
        ? `<img src="${logoUrl}" alt="Logo" class="ai-chatbot__logo-image" style="width: 24px; height: 24px;">`
        : '<span class="ai-chatbot__logo-icon">🚀</span>';
      const toggleIcon = logoUrl
        ? `<img src="${logoUrl}" alt="Logo" class="ai-chatbot__toggle-icon">`
        : '<span class="ai-chatbot__toggle-icon">🤖</span>';


      const chatbotHtml = `
        <div id="ai-content-assistant" class="ai-chatbot ai-chatbot--${position} ai-chatbot--${theme} ai-chatbot--collapsed">
          <div class="ai-chatbot__toggle" id="ai-chatbot-toggle">
            ${toggleIcon}
            <span class="ai-chatbot__toggle-text">SPhoenix AI</span>
            ${this.renderUsageBadge()}
          </div>

          <div class="ai-chatbot__container">
            <div class="ai-chatbot__header">
              <div class="ai-chatbot__header-left">
                <div class="ai-chatbot__logo">
                  ${logoHtml}
                  <h3 class="ai-chatbot__title">SPhoenix AI</h3>
                </div>
                <div class="ai-chatbot__status-indicator ${this.authState.authenticated ? 'ai-chatbot__status--online' : 'ai-chatbot__status--offline'}"></div>
              </div>

              <div class="ai-chatbot__header-right">
                <div class="ai-chatbot__user-info">${this.renderUserInfo()}</div>
                <div class="ai-chatbot__header-actions">
                  <button class="ai-chatbot__action-btn ai-chatbot__sync-btn" id="ai-chatbot-sync" title="Refresh subscription status">
                    <span class="ai-chatbot__sync-icon">🔄</span>
                  </button>
                  <button class="ai-chatbot__action-btn ai-chatbot__close" id="ai-chatbot-close">
                    <span class="ai-chatbot__close-icon">×</span>
                  </button>
                </div>
              </div>
            </div>

            <div class="ai-chatbot__auth-section" id="ai-chatbot-auth" style="display: none;">
              ${this.createAuthInterface()}
            </div>

            <div class="ai-chatbot__subscription-section" id="ai-chatbot-subscription" style="display: none;">
              ${this.createSubscriptionInterface()}
            </div>

            <div class="ai-chatbot__main-content" id="ai-chatbot-main">
              <div class="ai-chatbot__messages" id="ai-chatbot-messages"></div>
              <div class="ai-chatbot__task-selector" id="ai-chatbot-task-selector" style="display: none;">
                ${this.createTaskSelector()}
              </div>
              <div class="ai-chatbot__input-area">
                <div class="ai-chatbot__input-wrapper">
                  <textarea id="ai-chatbot-input" class="ai-chatbot__input" placeholder="Type your message or '/' for commands..." rows="2"></textarea>
                  <button id="ai-chatbot-send" class="ai-chatbot__send-button" disabled>
                    <span class="ai-chatbot__send-icon">➤</span>
                  </button>
                </div>
                <div class="ai-chatbot__status" id="ai-chatbot-status"></div>
              </div>
            </div>
          </div>
        </div>
      `;

      $('body').append(chatbotHtml);
    }

    createAuthInterface() {
      return `
        <div class="ai-auth-interface">
          <div class="ai-auth-header">
            <h4>Login to SPhoenix AI</h4>
            <p>Access 100+ AI-powered CMS tools</p>
          </div>
          <div class="ai-auth-actions">
            <button type="button" class="ai-btn ai-btn--primary" id="ai-redirect-login">
              <span class="ai-btn-icon">🔐</span>
              Login with SPhoenix AI
            </button>
            <p class="ai-auth-note">You'll be redirected to our secure login page</p>
          </div>
        </div>
      `;
    }

    createSubscriptionInterface() {
      return `
        <div class="ai-subscription-interface">
          <div class="ai-subscription-header">
            <h4>CMS Subscription Required</h4>
            <p>You need a CMS plan to use AI tools in Drupal</p>
          </div>
          <div class="ai-subscription-actions">
            <button class="ai-btn ai-btn--primary" id="ai-view-cms-plans">View CMS Plans</button>
          </div>
        </div>
      `;
    }

    renderUsageBadge() {
      if (!this.authState.authenticated || !this.subscriptionInfo?.usage_info) {
        return '';
      }

      const usage = this.subscriptionInfo.usage_info;
      const remaining = usage.remaining_usage || 0;
      const percentage = Math.round(usage.usage_percentage || 0);

      let badgeClass = 'usage-badge';
      if (percentage >= 90) badgeClass += ' usage-badge--critical';
      else if (percentage >= 75) badgeClass += ' usage-badge--warning';
      else badgeClass += ' usage-badge--normal';

      return `<div class="${badgeClass}">${remaining}</div>`;
    }

    renderUserInfo() {
      if (!this.authState.authenticated) {
        return '<span class="ai-user-status">Not logged in</span>';
      }

      const user = this.authState.user;
      const planName = this.subscriptionInfo?.subscription?.primary_subscription?.plan_id?.plan_name || 'Free';
      const remainingUsage = this.subscriptionInfo?.subscription?.primary_subscription?.remaining_usage || 0;

      return `
        <div class="ai-user-info">
          <span class="ai-user-name">${user?.firstname || user?.email || 'User'} (${planName}) (${remainingUsage})</span>
        </div>
      `;
    }

    createTaskSelector() {
      let selectorHtml = '<div class="ai-task-selector">';
      selectorHtml += '<div class="ai-task-selector__header">';
      selectorHtml += '<h4 class="ai-task-selector__title">Select AI Task:</h4>';
      selectorHtml += '<button class="ai-task-selector__close" id="ai-task-selector-close">×</button>';
      selectorHtml += '</div>';
      selectorHtml += '<div class="ai-task-selector__grid">';

      Object.entries(taskConfigs).forEach(([taskType, config]) => {
        selectorHtml += `
          <div class="ai-task-option" data-task-type="${taskType}">
            <span class="ai-task-option__icon">${config.icon}</span>
            <div class="ai-task-option__content">
              <h5 class="ai-task-option__label">${config.label}</h5>
              <p class="ai-task-option__description">${config.description}</p>
            </div>
          </div>
        `;
      });

      selectorHtml += '</div></div>';
      return selectorHtml;
    }

    attachEventListeners() {
      const self = this;

      // Main chatbot controls
      $(document).on('click', '#ai-chatbot-toggle', (e) => {
        e.preventDefault();
        this.toggleChatbot();
      });

      $(document).on('click', '#ai-chatbot-close', (e) => {
        e.preventDefault();
        this.collapseChatbot();
      });

      // Authentication
      $(document).on('click', '#ai-redirect-login', (e) => this.redirectToLogin(e));
      $(document).on('click', '#ai-view-cms-plans', () => this.viewCmsPlans());

      // Chat input
      $(document).on('input', '#ai-chatbot-input', (e) => this.handleInput(e));
      $(document).on('keydown', '#ai-chatbot-input', (e) => this.handleKeyDown(e));
      $(document).on('click', '#ai-chatbot-send', () => this.sendMessage());

      // Task selection
      $(document).on('click', '.ai-task-option', function () {
        const taskType = $(this).data('task-type');
        self.selectTask(taskType);
      });

      $(document).on('click', '#ai-task-selector-close', (e) => {
        e.preventDefault();
        $('#ai-chatbot-task-selector').hide();
      });

      $(document).on('click', '.ai-action-btn, .ai-image-action-btn, .ai-field-action-btn, .ai-preview-btn', function (e) {
        e.preventDefault();
        e.stopImmediatePropagation();
        const action = $(this).data('action');
        self.handleAction(action, $(this));
      });

      //Action buttons
      // $(document).on('click', '.ai-action-btn', function () {
      //   const action = $(this).data('action');
      //   self.handleAction(action, $(this));
      // });

      // $(document).on('click', '.ai-field-action-btn', function (e) {
      //   e.preventDefault();
      //   const action = $(this).data('action');
      //   self.handleAction(action, $(this));
      // });

      $(document).on('click', '#ai-chatbot-sync', (e) => {
        e.preventDefault();
        this.handleSync();
      });
    }

    redirectToLogin(e) {
      e.preventDefault();
      const loginUrl = drupalSettings.aiContentAssistant?.apiEndpoints.loginUrl;
      window.location.href = loginUrl;
    }

    viewCmsPlans() {
      const plansUrl = drupalSettings.aiContentAssistant?.apiEndpoints?.plansUrl ||
        'https://www.sphoenixai.com/plans-catalog?category=cms';
      window.open(plansUrl, '_blank');
    }

    checkInitialState() {
      if (!this.authState.authenticated) {
        this.showAuthenticationInterface();
        this.addMessage('system', 'Please login to access AI-powered content tools.');
      } else if (!this.authState.subscription || !this.checkCmsSubscription(this.authState.subscription)) {
        this.showSubscriptionInterface();
        this.addMessage('system', 'CMS subscription required to use AI features.');
      } else if (!this.subscriptionInfo?.has_usage_remaining) {
        this.showSubscriptionInterface();
        this.addMessage('system', 'Usage limit reached. Please upgrade your plan.');
      } else {
        this.showChatInterface();
        this.addMessage('system', 'Welcome! I\'m your AI assistant. How can I help you today?');
      }
    }

    showAuthenticationInterface() {
      $('#ai-chatbot-auth').show();
      $('#ai-chatbot-subscription').hide();
      $('#ai-chatbot-main').hide();
    }

    showSubscriptionInterface() {
      $('#ai-chatbot-auth').hide();
      $('#ai-chatbot-subscription').show();
      $('#ai-chatbot-main').hide();
    }

    showChatInterface() {
      $('#ai-chatbot-auth').hide();
      $('#ai-chatbot-subscription').hide();
      $('#ai-chatbot-main').show();
    }

    handleInput(e) {
      const input = e.target.value;
      const $sendButton = $('#ai-chatbot-send');
      const $taskSelector = $('#ai-chatbot-task-selector');

      $sendButton.prop('disabled', input.trim().length === 0);

      if (input === '/') {
        $taskSelector.show();
        e.target.value = '';
      } else {
        $taskSelector.hide();
      }

      this.autoResizeTextarea(e.target);
    }

    handleKeyDown(e) {
      if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
        e.preventDefault();
        this.sendMessage();
      }

      if (e.key === '/' && e.target.value === '') {
        setTimeout(() => $('#ai-chatbot-task-selector').show(), 50);
      }
    }

    autoResizeTextarea(textarea) {
      textarea.style.height = 'auto';
      textarea.style.height = Math.min(textarea.scrollHeight, 150) + 'px';
    }

    toggleChatbot() {
      if (this.isExpanded) {
        this.collapseChatbot();
      } else {
        this.expandChatbot();
      }
    }

    expandChatbot() {
      const $chatbot = $('#ai-content-assistant');
      $chatbot.removeClass('ai-chatbot--collapsed').addClass('ai-chatbot--expanded');
      this.isExpanded = true;
      setTimeout(() => $('#ai-chatbot-input').focus(), 300);
    }

    collapseChatbot() {
      const $chatbot = $('#ai-content-assistant');
      $chatbot.removeClass('ai-chatbot--expanded').addClass('ai-chatbot--collapsed');
      $('#ai-chatbot-task-selector').hide();
      this.isExpanded = false;
    }

    selectTask(taskType) {
      this.selectedTaskType = taskType;
      const config = taskConfigs[taskType];

      $('#ai-chatbot-task-selector').hide();
      $('#ai-chatbot-input').attr('placeholder', config.prompt_placeholder).focus();

      this.addMessage('system', `Selected: ${config.label}. ${config.description}`);

      if (config.requires_existing) {
        this.checkExistingContent();
      }
    }

    checkExistingContent() {
      const fieldData = this.extractFieldData();

      // Check for CKEditor 5 content first
      let hasBodyContent = false;
      if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
        try {
          Drupal.CKEditor5Instances.forEach((editor, key) => {
            if (editor.sourceElement.id === 'edit-body-0-value') {
              const editorInstance = Drupal.CKEditor5Instances.get(key);
              if (editorInstance) {
                const content = editorInstance.getData();
                if (content && content.trim().length > 0) {
                  hasBodyContent = true;
                }
              }
            }
          });
        } catch (e) {
          console.warn('Failed to check CKEditor 5 content:', e);
        }
      }

      // Check traditional field data
      const hasFieldData = Object.keys(fieldData).length > 0;

      if (!hasBodyContent && !hasFieldData) {
        this.addMessage('system', 'This task requires existing content. Please add some content to your fields first.');
        return false;
      }

      return true;
    }

    extractFieldData() {
      const fieldData = {};
      const fieldSelectors = [
        'input[name="title[0][value]"]',
        'textarea[name="body[0][value]"]',  // Add body content
        'textarea[name*="[field_"]',
        'input[name*="[field_"]'
      ];

      fieldSelectors.forEach(selector => {
        $(selector).each(function () {
          const $field = $(this);
          const name = $field.attr('name');
          let value = $field.val();

          // For body fields, try to get content from CKEditor 5 first
          if (name && name.includes('body') && typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
            try {
              let editorContent = '';
              Drupal.CKEditor5Instances.forEach((editor, key) => {
                if (editor.sourceElement.id === 'edit-body-0-value') {
                  const editorInstance = Drupal.CKEditor5Instances.get(key);
                  if (editorInstance) {
                    editorContent = editorInstance.getData();
                  }
                }
              });

              // Use editor content if available, otherwise fallback to textarea value
              if (editorContent && editorContent.trim().length > 0) {
                value = editorContent;
              }
            } catch (e) {
              console.warn('Failed to get CKEditor 5 content, using textarea value:', e);
              // Use textarea value as fallback
            }
          }

          if (value && value.trim().length > 0) {
            fieldData[name] = value;
          }
        });
      });

      return fieldData;
    }

    async sendMessage() {
      if (!this.authState.authenticated) {
        this.showNotification('Please login first', 'error');
        this.showAuthenticationInterface();
        return;
      }

      if (!this.subscriptionInfo?.has_usage_remaining) {
        this.showNotification('No usage remaining. Please upgrade your plan.', 'error');
        this.showSubscriptionInterface();
        return;
      }

      const input = $('#ai-chatbot-input').val().trim();
      if (!input || this.isProcessing) return;

      if (!this.selectedTaskType) {
        this.selectedTaskType = this.autoDetectTaskType(input);
      }

      this.addMessage('user', input);
      $('#ai-chatbot-input').val('');
      this.setProcessingState(true);

      try {
        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
      }
    }

    autoDetectTaskType(input) {
      const lowerInput = input.toLowerCase();

      let hasBodyContent = false;
      if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
        try {
          Drupal.CKEditor5Instances.forEach((editor, key) => {
            if (editor.sourceElement.id === 'edit-body-0-value') {
              const editorInstance = Drupal.CKEditor5Instances.get(key);
              if (editorInstance) {
                const content = editorInstance.getData();
                if (content && content.trim().length > 0) {
                  hasBodyContent = true;
                }
              }
            }
          });
        } catch (e) {
          console.warn('Failed to check body content for task detection:', e);
        }
      }

      if (lowerInput.includes('generate') || lowerInput.includes('create') || lowerInput.includes('write')) {
        return TaskType.CONTENT_GENERATION;
      } else if (lowerInput.includes('improve') || lowerInput.includes('rewrite')) {
        return TaskType.CONTENT_REGENERATION;
      } else if (lowerInput.includes('optimize') || lowerInput.includes('seo')) {
        return TaskType.CONTENT_OPTIMIZATION;
      } else if (lowerInput.includes('analyze')) {
        return TaskType.SEO_ANALYSIS;
      } else if (lowerInput.includes('keywords')) {
        return TaskType.KEYWORD_EXTRACTION;  // Added
      } else if (lowerInput.includes('image') || lowerInput.includes('thumbnail')) {
        return TaskType.THUMBNAIL_GENERATION;
      } else if (lowerInput.includes('meta')) {
        return TaskType.META_GENERATION;
      }

      return hasBodyContent ? TaskType.CONTENT_REGENERATION : TaskType.CONTENT_GENERATION;
    }

    preparePayload(input) {
      const config = taskConfigs[this.selectedTaskType];
      const payload = {
        task_type: this.selectedTaskType,
        prompt: input,
        content_type: this.contentType,
      };

      // Always extract field data for better context
      const fieldData = this.extractFieldData();

      // For tasks that require existing content OR for better AI context
      if (config.requires_existing || Object.keys(fieldData).length > 0) {
        // Priority order: body content > title > all fields combined
        let existingContent = '';
        let bodyContent = '';

        // Try to get body content using CKEditor 5 first
        if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
          try {
            Drupal.CKEditor5Instances.forEach((editor, key) => {
              if (editor.sourceElement.id === 'edit-body-0-value') {
                const editorInstance = Drupal.CKEditor5Instances.get(key);
                if (editorInstance) {
                  bodyContent = editorInstance.getData();
                }
              }
            });
          } catch (e) {
            console.warn('Failed to get CKEditor 5 content in preparePayload:', e);
          }
        }

        // Use body content if available from CKEditor or fieldData
        if (bodyContent && bodyContent.trim().length > 0) {
          existingContent = bodyContent;
        } else if (fieldData['body[0][value]'] && fieldData['body[0][value]'].trim().length > 0) {
          existingContent = fieldData['body[0][value]'];
        } else if (fieldData['title[0][value]'] && fieldData['title[0][value]'].trim().length > 0) {
          existingContent = fieldData['title[0][value]'];
        } else {
          existingContent = Object.values(fieldData).join('\n\n');
        }

        // Always include existing content if available for better AI context
        if (existingContent && existingContent.trim().length > 0) {
          payload.existing_content = existingContent;
          payload.field_data = fieldData;
        }
      }

      return payload;
    }

    async makeApiRequest(payload) {
      const response = await fetch('/ai-assistant/chat', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-Token': drupalSettings.user.csrf_token || ''
        },
        body: JSON.stringify(payload)
      });

      return await response.json();
    }

    handleApiResponse(responseData) {
      this.fieldMappings = responseData.field_mappings || {};

      if (responseData.task_type == 'content_generation' || responseData.task_type == 'content_regeneration') {
        this.displayContentResponse(responseData.content, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'thumbnail_generation') {
        this.displayImageResponse(responseData.image, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'seo_analysis') {
        this.displayAnalysisResponse(responseData.analysis, responseData.suggested_actions || []);
      } else if (responseData.graph) {
        this.displayGraphResponse(responseData.graph, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'content_optimization') {
        this.displayOptimizedContentResponse(responseData.content, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'table_generation') {
        this.displayTableResponse(responseData.content, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'meta_generation') {
        this.displayMetaResponse(responseData.content, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'content_summary') {
        this.displaySummaryResponse(responseData.content, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'content_analysis') {
        this.displayContentAnalysisResponse(responseData.analysis, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'keyword_extraction') {
        this.displayKeywordExtractionResponse(responseData.content, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'title_generation') {
        this.displayTitleGenerationResponse(responseData.content, responseData.suggested_actions || []);
      } else if (responseData.task_type == 'description_generation') {
        this.displayDescriptionResponse(responseData.content, responseData.suggested_actions || []);
      } else {
        this.addMessage('system', 'Response received but no displayable content found.');
      }

      this.selectedTaskType = null;
      $('#ai-chatbot-input').attr('placeholder', 'Type your message or \'/\' for commands...');
    }

    displayDescriptionResponse(content, actions) {
      let messageHtml = '<div class="ai-response ai-response--description-generation">';

      // Add description generation header
      messageHtml += '<div class="ai-description-header">';
      messageHtml += '<h5 class="ai-description-title">📝 Descriptions Generated</h5>';
      messageHtml += '<div class="ai-description-stats">';

      // Show humanization score if available
      if (content.humanization_score) {
        const humanScore = Math.round(content.humanization_score * 100);
        const scoreClass = humanScore >= 80 ? 'excellent' : humanScore >= 70 ? 'good' : humanScore >= 50 ? 'medium' : 'poor';
        messageHtml += `<span class="ai-stat-item ai-humanization-score ai-score--${scoreClass}">🤖 ${humanScore}% Human-like</span>`;
      }

      messageHtml += '</div>';
      messageHtml += '</div>';

      // Define field mappings for descriptions
      const descriptionFieldMappings = {
        'meta_description': { label: 'Meta Description', icon: '🔍', priority: 1 },
        'plain_text': { label: 'Full Description', icon: '📄', priority: 2 }
      };

      // Sort fields by priority and only show fields that have content
      const sortedFields = Object.entries(descriptionFieldMappings)
        .filter(([fieldKey]) => content[fieldKey] && content[fieldKey] !== '')
        .sort(([, a], [, b]) => a.priority - b.priority);

      // Process each description field
      sortedFields.forEach(([fieldKey, fieldInfo]) => {
        const fieldValue = content[fieldKey];

        messageHtml += `<div class="ai-field-item ai-field-item--description" data-field="${fieldKey}">`;
        messageHtml += `<div class="ai-field-header">`;
        messageHtml += `<div class="ai-field-title-section">`;
        messageHtml += `<span class="ai-field-icon">${fieldInfo.icon}</span>`;
        messageHtml += `<h5 class="ai-field-title">${fieldInfo.label}</h5>`;
        messageHtml += `</div>`;

        // Action buttons for each field
        messageHtml += `<div class="ai-field-actions-header">`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                      data-action="insert_field"
                      data-field="${fieldKey}"
                      data-value='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                      title="Insert ${fieldInfo.label}">
                      ➕
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                      data-action="copy_field"
                      data-field="${fieldKey}"
                      data-value='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                      title="Copy ${fieldInfo.label}">
                      📋
                    </button>`;
        messageHtml += `</div></div>`;

        // Display content with appropriate formatting and analysis
        if (fieldKey === 'meta_description') {
          // Show character count and SEO analysis for meta description
          const charCount = fieldValue.length;
          const isOptimal = charCount >= 120 && charCount <= 160;
          const charClass = isOptimal ? 'optimal' : 'warning';

          messageHtml += `<div class="ai-field-content ai-field-content--meta-description">`;
          messageHtml += `<div class="ai-description-text">${this.escapeHtml(fieldValue)}</div>`;
          messageHtml += `<div class="ai-description-analysis">`;
          messageHtml += `<span class="ai-char-count ai-char--${charClass}">📏 ${charCount} characters</span>`;
          if (isOptimal) {
            messageHtml += '<span class="ai-description-feedback ai-feedback--positive">✅ Optimal length for SEO</span>';
          } else if (charCount < 120) {
            messageHtml += '<span class="ai-description-feedback ai-feedback--warning">⚠️ Consider making longer (120-160 chars recommended)</span>';
          } else {
            messageHtml += '<span class="ai-description-feedback ai-feedback--warning">⚠️ Consider making shorter (120-160 chars recommended)</span>';
          }
          messageHtml += `</div></div>`;
        } else if (fieldKey === 'plain_text') {
          // Show word count for full description
          const wordCount = fieldValue.trim().split(/\s+/).length;
          const truncatedText = fieldValue.length > 200 ? fieldValue.substring(0, 200) + '...' : fieldValue;

          messageHtml += `<div class="ai-field-content ai-field-content--full-description">`;
          messageHtml += `<div class="ai-description-text">${this.escapeHtml(truncatedText)}</div>`;
          if (fieldValue.length > 200) {
            messageHtml += `<div class="ai-content-indicator">Description truncated for preview...</div>`;
          }
          messageHtml += `<div class="ai-description-analysis">`;
          messageHtml += `<span class="ai-word-count">📊 ${wordCount} words</span>`;
          messageHtml += `</div></div>`;
        } else {
          messageHtml += `<div class="ai-field-content">`;
          messageHtml += `<span class="field-value">${this.escapeHtml(fieldValue)}</span>`;
          messageHtml += `</div>`;
        }

        messageHtml += `</div>`;
      });

      // Description generation specific actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__description-actions">';
        messageHtml += '<h6>Description Actions:</h6>';
        messageHtml += '<div class="ai-description-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-description-action-btn"
                        data-action="${action}"
                        data-description-content='${JSON.stringify(content).replace(/'/g, "&#39;")}'
                        title="${this.getActionLabel(action)}">
                        ${this.getActionLabel(action)}
                      </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { content, actions, type: 'description_generation' });
    }

    displayTitleGenerationResponse(content, actions) {
      let messageHtml = '<div class="ai-response ai-response--title-generation">';

      // Add title generation header
      messageHtml += '<div class="ai-title-header">';
      messageHtml += '<h5 class="ai-title-title">📰 Title Generated</h5>';
      messageHtml += '<div class="ai-title-stats">';

      // Show humanization score if available
      if (content.humanization_score) {
        const humanScore = Math.round(content.humanization_score * 100);
        const scoreClass = humanScore >= 80 ? 'excellent' : humanScore >= 70 ? 'good' : humanScore >= 50 ? 'medium' : 'poor';
        messageHtml += `<span class="ai-stat-item ai-humanization-score ai-score--${scoreClass}">🤖 ${humanScore}% Human-like</span>`;
      }

      // Show character count for the title
      if (content.title) {
        const charCount = content.title.length;
        const isOptimal = charCount >= 30 && charCount <= 70;
        const charClass = isOptimal ? 'optimal' : 'warning';
        messageHtml += `<span class="ai-stat-item ai-char-count ai-char--${charClass}">📏 ${charCount} characters</span>`;
      }

      messageHtml += '</div>';
      messageHtml += '</div>';

      // Generated title field
      if (content.title) {
        messageHtml += '<div class="ai-field-item ai-field-item--generated-title" data-field="title">';
        messageHtml += '<div class="ai-field-header">';
        messageHtml += '<div class="ai-field-title-section">';
        messageHtml += '<span class="ai-field-icon">📰</span>';
        messageHtml += '<h5 class="ai-field-title">Generated Title</h5>';
        messageHtml += '</div>';

        // Action buttons for title
        messageHtml += '<div class="ai-field-actions-header">';
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                      data-action="insert_field"
                      data-field="title"
                      data-value='${JSON.stringify(content.title).replace(/'/g, "&#39;")}'
                      title="Insert Title">
                      ➕
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                      data-action="copy_field"
                      data-field="title"
                      data-value='${JSON.stringify(content.title).replace(/'/g, "&#39;")}'
                      title="Copy Title">
                      📋
                    </button>`;
        messageHtml += '</div></div>';

        // Display the title prominently
        messageHtml += '<div class="ai-field-content ai-field-content--title-display">';
        messageHtml += `<div class="ai-generated-title">${this.escapeHtml(content.title)}</div>`;

        // Show character analysis
        const charCount = content.title.length;
        const isOptimal = charCount >= 30 && charCount <= 70;
        messageHtml += '<div class="ai-title-analysis">';
        if (isOptimal) {
          messageHtml += '<span class="ai-title-feedback ai-feedback--positive">✅ Optimal length for SEO</span>';
        } else if (charCount < 30) {
          messageHtml += '<span class="ai-title-feedback ai-feedback--warning">⚠️ Consider making longer (30-70 chars recommended)</span>';
        } else {
          messageHtml += '<span class="ai-title-feedback ai-feedback--warning">⚠️ Consider making shorter (30-70 chars recommended)</span>';
        }
        messageHtml += '</div>';

        messageHtml += '</div></div>';
      }

      // Title strategy explanation
      if (content.plain_text) {
        messageHtml += '<div class="ai-title-strategy">';
        messageHtml += '<h6>💡 Title Strategy & Reasoning</h6>';
        messageHtml += `<p class="ai-strategy-text">${this.escapeHtml(content.plain_text)}</p>`;
        messageHtml += '</div>';
      }

      // Title generation specific actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__title-actions">';
        messageHtml += '<h6>Title Actions:</h6>';
        messageHtml += '<div class="ai-title-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-title-action-btn"
                        data-action="${action}"
                        data-title-content='${JSON.stringify(content).replace(/'/g, "&#39;")}'
                        title="${this.getActionLabel(action)}">
                        ${this.getActionLabel(action)}
                      </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { content, actions, type: 'title_generation' });
    }

    displayKeywordExtractionResponse(content, actions) {
      let messageHtml = '<div class="ai-response ai-response--keyword-extraction">';

      // Add keyword extraction header with stats
      messageHtml += '<div class="ai-keyword-header">';
      messageHtml += '<h5 class="ai-keyword-title">🔑 Keyword Extraction Results</h5>';
      messageHtml += '<div class="ai-keyword-stats">';

      // Get total keywords count
      const totalKeywords = Array.isArray(content.keywords) ? content.keywords.length : 0;
      messageHtml += `<span class="ai-stat-item">📊 ${totalKeywords} keywords found</span>`;

      if (content.readability_score) {
        messageHtml += `<span class="ai-stat-item">📈 ${this.escapeHtml(content.readability_score)}</span>`;
      }
      messageHtml += '</div>';
      messageHtml += '</div>';

      // Parse and display structured keyword analysis from HTML content
      if (content.html_content) {
        const keywordAnalysis = this.parseKeywordHTML(content.html_content);

        if (keywordAnalysis.primary_keywords) {
          messageHtml += '<div class="ai-keyword-section ai-primary-keywords">';
          messageHtml += '<h6>🎯 Primary Keywords</h6>';
          messageHtml += '<div class="ai-keywords-display">';
          keywordAnalysis.primary_keywords.forEach(keyword => {
            messageHtml += `<span class="ai-keyword ai-keyword--primary">${this.escapeHtml(keyword.trim())}</span>`;
          });
          messageHtml += '</div></div>';
        }

        if (keywordAnalysis.longtail_keywords) {
          messageHtml += '<div class="ai-keyword-section ai-longtail-keywords">';
          messageHtml += '<h6>📝 Long-tail Keywords</h6>';
          messageHtml += '<div class="ai-keywords-display">';
          keywordAnalysis.longtail_keywords.forEach(keyword => {
            messageHtml += `<span class="ai-keyword ai-keyword--longtail">${this.escapeHtml(keyword.trim())}</span>`;
          });
          messageHtml += '</div></div>';
        }

        if (keywordAnalysis.related_terms) {
          messageHtml += '<div class="ai-keyword-section ai-related-terms">';
          messageHtml += '<h6>🔗 Related Terms</h6>';
          messageHtml += '<div class="ai-keywords-display">';
          keywordAnalysis.related_terms.forEach(keyword => {
            messageHtml += `<span class="ai-keyword ai-keyword--related">${this.escapeHtml(keyword.trim())}</span>`;
          });
          messageHtml += '</div></div>';
        }

        if (keywordAnalysis.recommended_focus) {
          messageHtml += '<div class="ai-keyword-section ai-focus-keywords">';
          messageHtml += '<h6>⭐ Recommended Focus</h6>';
          messageHtml += '<div class="ai-keywords-display">';
          keywordAnalysis.recommended_focus.forEach(keyword => {
            messageHtml += `<span class="ai-keyword ai-keyword--focus">${this.escapeHtml(keyword.trim())}</span>`;
          });
          messageHtml += '</div></div>';
        }

        if (keywordAnalysis.analysis) {
          messageHtml += '<div class="ai-keyword-analysis">';
          messageHtml += '<h6>📋 Analysis & Recommendations</h6>';
          messageHtml += `<p class="ai-analysis-text">${this.escapeHtml(keywordAnalysis.analysis)}</p>`;
          messageHtml += '</div>';
        }
      }

      // All keywords field (for insertion/copying)
      if (Array.isArray(content.keywords) && content.keywords.length > 0) {
        messageHtml += '<div class="ai-field-item ai-field-item--all-keywords" data-field="keywords">';
        messageHtml += '<div class="ai-field-header">';
        messageHtml += '<div class="ai-field-title-section">';
        messageHtml += '<span class="ai-field-icon">🔑</span>';
        messageHtml += '<h5 class="ai-field-title">All Extracted Keywords</h5>';
        messageHtml += '</div>';

        // Action buttons for keywords
        messageHtml += '<div class="ai-field-actions-header">';
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                      data-action="insert_field"
                      data-field="keywords"
                      data-value='${JSON.stringify(content.keywords).replace(/'/g, "&#39;")}'
                      title="Insert All Keywords">
                      ➕
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                      data-action="copy_field"
                      data-field="keywords"
                      data-value='${JSON.stringify(content.keywords).replace(/'/g, "&#39;")}'
                      title="Copy All Keywords">
                      📋
                    </button>`;
        messageHtml += '</div></div>';

        // Display keywords with count
        messageHtml += '<div class="ai-field-content ai-field-content--keywords">';
        messageHtml += `<div class="ai-keyword-count">${content.keywords.length} keywords total</div>`;
        messageHtml += '<div class="ai-all-keywords-display">';
        content.keywords.slice(0, 10).forEach(keyword => {
          messageHtml += `<span class="ai-keyword">${this.escapeHtml(keyword)}</span>`;
        });
        if (content.keywords.length > 10) {
          messageHtml += `<span class="ai-keyword-more">+${content.keywords.length - 10} more...</span>`;
        }
        messageHtml += '</div></div>';
        messageHtml += '</div>';
      }

      // Keyword report field (HTML content)
      if (content.html_content) {
        messageHtml += '<div class="ai-field-item ai-field-item--keyword-report" data-field="html_content">';
        messageHtml += '<div class="ai-field-header">';
        messageHtml += '<div class="ai-field-title-section">';
        messageHtml += '<span class="ai-field-icon">📊</span>';
        messageHtml += '<h5 class="ai-field-title">Keyword Analysis Report</h5>';
        messageHtml += '</div>';

        // Action buttons for report
        messageHtml += '<div class="ai-field-actions-header">';
        messageHtml += `<button class="ai-preview-btn ai-field-action-btn"
                      data-action="preview_html"
                      data-content='${JSON.stringify(content.html_content).replace(/'/g, "&#39;")}'
                      title="Preview Report">
                      👁️
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                      data-action="insert_field"
                      data-field="html_content"
                      data-value='${JSON.stringify(content.html_content).replace(/'/g, "&#39;")}'
                      title="Insert Report">
                      ➕
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                      data-action="copy_field"
                      data-field="html_content"
                      data-value='${JSON.stringify(content.html_content).replace(/'/g, "&#39;")}'
                      title="Copy Report">
                      📋
                    </button>`;
        messageHtml += '</div></div>';

        messageHtml += '<div class="ai-field-content ai-field-content--report-preview">';
        messageHtml += '<div class="ai-report-summary">Detailed keyword analysis report ready for insertion</div>';
        messageHtml += '</div></div>';
      }

      // Keyword extraction specific actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__keyword-actions">';
        messageHtml += '<h6>Keyword Actions:</h6>';
        messageHtml += '<div class="ai-keyword-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-keyword-action-btn"
                        data-action="${action}"
                        data-keyword-content='${JSON.stringify(content).replace(/'/g, "&#39;")}'
                        title="${this.getActionLabel(action)}">
                        ${this.getActionLabel(action)}
                      </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { content, actions, type: 'keyword_extraction' });
    }

    // Helper method to parse structured keyword HTML
    parseKeywordHTML(htmlContent) {
      const analysis = {
        primary_keywords: [],
        longtail_keywords: [],
        related_terms: [],
        recommended_focus: [],
        analysis: ''
      };

      try {
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = htmlContent;

        // Extract each section based on headings
        const sections = tempDiv.querySelectorAll('h3');
        sections.forEach(section => {
          const sectionTitle = section.textContent.toLowerCase();
          const nextP = section.nextElementSibling;

          if (nextP && nextP.tagName === 'P') {
            const content = nextP.textContent.trim();

            if (sectionTitle.includes('primary')) {
              analysis.primary_keywords = content.split(',').map(k => k.trim()).filter(k => k);
            } else if (sectionTitle.includes('long-tail')) {
              analysis.longtail_keywords = content.split(',').map(k => k.trim()).filter(k => k);
            } else if (sectionTitle.includes('related')) {
              analysis.related_terms = content.split(',').map(k => k.trim()).filter(k => k);
            } else if (sectionTitle.includes('focus')) {
              analysis.recommended_focus = content.split(',').map(k => k.trim()).filter(k => k);
            } else if (sectionTitle.includes('analysis')) {
              analysis.analysis = content;
            }
          }
        });
      } catch (error) {
        console.warn('Error parsing keyword HTML:', error);
      }

      return analysis;
    }

    displayContentAnalysisResponse(analysis, actions) {
      let messageHtml = '<div class="ai-response ai-response--content-analysis">';

      // Add analysis header with overall score
      messageHtml += '<div class="ai-analysis-header">';
      messageHtml += '<h5 class="ai-analysis-title">📊 Content Analysis Report</h5>';
      if (analysis.overall_score !== null && analysis.overall_score !== undefined) {
        const scoreClass = analysis.overall_score >= 80 ? 'excellent' :
          analysis.overall_score >= 70 ? 'good' :
            analysis.overall_score >= 60 ? 'medium' : 'poor';
        messageHtml += `<div class="ai-analysis-overall-score ai-score--${scoreClass}">
      <span class="ai-score-label">Overall Score:</span>
      <span class="ai-score-value">${analysis.overall_score}/100</span>
    </div>`;
      }
      messageHtml += '</div>';

      // Analysis overview text
      if (analysis.analysis_text) {
        messageHtml += `<div class="ai-analysis-overview">
      <h6>📝 Analysis Overview</h6>
      <p>${this.escapeHtml(analysis.analysis_text)}</p>
    </div>`;
      }

      // Content Statistics
      if (analysis.word_count || analysis.estimated_read_time) {
        messageHtml += '<div class="ai-content-stats">';
        messageHtml += '<h6>📈 Content Statistics</h6>';
        messageHtml += '<div class="ai-stats-grid">';

        if (analysis.word_count) {
          messageHtml += `<div class="ai-stat-item">
        <span class="ai-stat-icon">📝</span>
        <span class="ai-stat-label">Word Count:</span>
        <span class="ai-stat-value">${analysis.word_count} words</span>
      </div>`;
        }

        if (analysis.estimated_read_time) {
          messageHtml += `<div class="ai-stat-item">
        <span class="ai-stat-icon">⏱️</span>
        <span class="ai-stat-label">Read Time:</span>
        <span class="ai-stat-value">${this.escapeHtml(analysis.estimated_read_time)}</span>
      </div>`;
        }

        messageHtml += '</div></div>';
      }

      // Readability Analysis
      if (analysis.readability) {
        messageHtml += '<div class="ai-analysis-section ai-readability-section">';
        messageHtml += '<h6>🎯 Readability Analysis</h6>';
        messageHtml += '<div class="ai-readability-metrics">';

        if (analysis.readability.grade_level) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Reading Level:</span>
        <span class="ai-metric-value">${this.escapeHtml(analysis.readability.grade_level)}</span>
      </div>`;
        }

        if (analysis.readability.reading_ease) {
          const easeClass = analysis.readability.reading_ease >= 70 ? 'good' :
            analysis.readability.reading_ease >= 50 ? 'medium' : 'poor';
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Reading Ease:</span>
        <span class="ai-metric-value ai-metric--${easeClass}">${analysis.readability.reading_ease}/100</span>
      </div>`;
        }

        if (analysis.readability.avg_sentence_length) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Avg. Sentence Length:</span>
        <span class="ai-metric-value">${analysis.readability.avg_sentence_length} words</span>
      </div>`;
        }

        messageHtml += '</div></div>';
      }

      // Engagement Analysis
      if (analysis.engagement) {
        messageHtml += '<div class="ai-analysis-section ai-engagement-section">';
        messageHtml += '<h6>💡 Engagement Analysis</h6>';

        if (analysis.engagement.score) {
          const engagementClass = analysis.engagement.score >= 80 ? 'good' :
            analysis.engagement.score >= 60 ? 'medium' : 'poor';
          messageHtml += `<div class="ai-engagement-score">
        <span class="ai-metric-label">Engagement Score:</span>
        <span class="ai-metric-value ai-metric--${engagementClass}">${analysis.engagement.score}/100</span>
      </div>`;
        }

        if (analysis.engagement.emotional_appeal) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Emotional Appeal:</span>
        <span class="ai-metric-value">${this.escapeHtml(analysis.engagement.emotional_appeal)}</span>
      </div>`;
        }

        if (analysis.engagement.strengths && Array.isArray(analysis.engagement.strengths)) {
          messageHtml += '<div class="ai-engagement-strengths">';
          messageHtml += '<span class="ai-metric-label">Strengths:</span>';
          messageHtml += '<div class="ai-strengths-tags">';
          analysis.engagement.strengths.forEach(strength => {
            messageHtml += `<span class="ai-strength-tag">${this.escapeHtml(strength)}</span>`;
          });
          messageHtml += '</div></div>';
        }

        messageHtml += '</div>';
      }

      // Structure Analysis
      if (analysis.structure) {
        messageHtml += '<div class="ai-analysis-section ai-structure-section">';
        messageHtml += '<h6>🏗️ Structure Analysis</h6>';
        messageHtml += '<div class="ai-structure-metrics">';

        if (analysis.structure.score) {
          const structureClass = analysis.structure.score >= 80 ? 'good' :
            analysis.structure.score >= 60 ? 'medium' : 'poor';
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Structure Score:</span>
        <span class="ai-metric-value ai-metric--${structureClass}">${analysis.structure.score}/100</span>
      </div>`;
        }

        if (analysis.structure.paragraph_count) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Paragraphs:</span>
        <span class="ai-metric-value">${analysis.structure.paragraph_count}</span>
      </div>`;
        }

        if (analysis.structure.flow_rating) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Flow Rating:</span>
        <span class="ai-metric-value">${this.escapeHtml(analysis.structure.flow_rating)}</span>
      </div>`;
        }

        // Structure indicators
        const structureIndicators = [];
        if (analysis.structure.has_clear_intro) structureIndicators.push('✅ Clear Introduction');
        if (analysis.structure.has_conclusion) structureIndicators.push('✅ Has Conclusion');

        if (structureIndicators.length > 0) {
          messageHtml += '<div class="ai-structure-indicators">';
          structureIndicators.forEach(indicator => {
            messageHtml += `<div class="ai-structure-indicator">${indicator}</div>`;
          });
          messageHtml += '</div>';
        }

        messageHtml += '</div></div>';
      }

      // SEO Potential
      if (analysis.seo_potential) {
        messageHtml += '<div class="ai-analysis-section ai-seo-section">';
        messageHtml += '<h6>🔍 SEO Potential</h6>';
        messageHtml += '<div class="ai-seo-metrics">';

        if (analysis.seo_potential.score) {
          const seoClass = analysis.seo_potential.score >= 80 ? 'good' :
            analysis.seo_potential.score >= 60 ? 'medium' : 'poor';
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">SEO Score:</span>
        <span class="ai-metric-value ai-metric--${seoClass}">${analysis.seo_potential.score}/100</span>
      </div>`;
        }

        if (analysis.seo_potential.keyword_density) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Keyword Density:</span>
        <span class="ai-metric-value">${this.escapeHtml(analysis.seo_potential.keyword_density)}</span>
      </div>`;
        }

        if (analysis.seo_potential.heading_structure) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Heading Structure:</span>
        <span class="ai-metric-value">${this.escapeHtml(analysis.seo_potential.heading_structure)}</span>
      </div>`;
        }

        // SEO indicators
        const seoIndicators = [];
        if (analysis.seo_potential.meta_friendly) seoIndicators.push('✅ Meta Friendly');

        if (seoIndicators.length > 0) {
          messageHtml += '<div class="ai-seo-indicators">';
          seoIndicators.forEach(indicator => {
            messageHtml += `<div class="ai-seo-indicator">${indicator}</div>`;
          });
          messageHtml += '</div>';
        }

        messageHtml += '</div></div>';
      }

      // Tone Analysis
      if (analysis.tone_analysis) {
        messageHtml += '<div class="ai-analysis-section ai-tone-section">';
        messageHtml += '<h6>🎭 Tone Analysis</h6>';
        messageHtml += '<div class="ai-tone-metrics">';

        if (analysis.tone_analysis.primary_tone) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Primary Tone:</span>
        <span class="ai-metric-value ai-tone-badge">${this.escapeHtml(analysis.tone_analysis.primary_tone)}</span>
      </div>`;
        }

        if (analysis.tone_analysis.consistency_score) {
          const consistencyClass = analysis.tone_analysis.consistency_score >= 80 ? 'good' :
            analysis.tone_analysis.consistency_score >= 60 ? 'medium' : 'poor';
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Tone Consistency:</span>
        <span class="ai-metric-value ai-metric--${consistencyClass}">${analysis.tone_analysis.consistency_score}/100</span>
      </div>`;
        }

        if (analysis.tone_analysis.target_audience) {
          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">Target Audience:</span>
        <span class="ai-metric-value">${this.escapeHtml(analysis.tone_analysis.target_audience)}</span>
      </div>`;
        }

        messageHtml += '</div></div>';
      }

      // Improvement Suggestions
      if (analysis.improvement_suggestions && analysis.improvement_suggestions.length > 0) {
        messageHtml += '<div class="ai-analysis-suggestions">';
        messageHtml += '<h6>💡 Improvement Suggestions</h6>';
        messageHtml += '<ul class="ai-suggestions-list">';
        analysis.improvement_suggestions.forEach(suggestion => {
          messageHtml += `<li>${this.escapeHtml(suggestion)}</li>`;
        });
        messageHtml += '</ul></div>';
      }

      // Content Analysis Actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__analysis-actions">';
        messageHtml += '<h6>Quick Actions:</h6>';
        messageHtml += '<div class="ai-analysis-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-content-analysis-action-btn"
                        data-action="${action}"
                        data-analysis-content='${JSON.stringify(analysis).replace(/'/g, "&#39;")}'
                        title="${this.getActionLabel(action)}">
                        ${this.getActionLabel(action)}
                      </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { analysis, actions, type: 'content_analysis' });
    }

    displaySummaryResponse(content, actions) {
      let messageHtml = '<div class="ai-response ai-response--summary">';

      // Add summary header with stats
      messageHtml += '<div class="ai-summary-header">';
      messageHtml += '<h5 class="ai-summary-title">📄 Content Summary Generated</h5>';
      messageHtml += '<div class="ai-summary-stats">';
      if (content.word_count) {
        messageHtml += `<span class="ai-stat-item">📊 ${content.word_count} words</span>`;
      }
      if (content.readability_score) {
        messageHtml += `<span class="ai-stat-item">🎯 ${this.escapeHtml(content.readability_score)} reading level</span>`;
      }
      messageHtml += '</div>';
      messageHtml += '</div>';

      // Define field mappings specific to content summary
      const summaryFieldMappings = {
        'html_content': { label: 'Summary Content', icon: '📝', priority: 1 },
        'plain_text': { label: 'Plain Text Version', icon: '📄', priority: 2 },
        'keywords': { label: 'Extracted Keywords', icon: '🔑', priority: 3 }
      };

      // Sort fields by priority and only show fields that have content
      const sortedFields = Object.entries(summaryFieldMappings)
        .filter(([fieldKey]) => content[fieldKey] && content[fieldKey] !== '')
        .sort(([, a], [, b]) => a.priority - b.priority);

      // Process each summary field
      sortedFields.forEach(([fieldKey, fieldInfo]) => {
        const fieldValue = content[fieldKey];

        messageHtml += `<div class="ai-field-item ai-field-item--summary" data-field="${fieldKey}">`;
        messageHtml += `<div class="ai-field-header">`;
        messageHtml += `<div class="ai-field-title-section">`;
        messageHtml += `<span class="ai-field-icon">${fieldInfo.icon}</span>`;
        messageHtml += `<h5 class="ai-field-title">${fieldInfo.label}</h5>`;
        messageHtml += `</div>`;

        // Action buttons for each field
        messageHtml += `<div class="ai-field-actions-header">`;

        // Special preview button for HTML content
        if (fieldKey === 'html_content') {
          messageHtml += `<button class="ai-preview-btn ai-field-action-btn"
                        data-action="preview_html"
                        data-content='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                        title="Preview Summary">
                        👁️
                      </button>`;
        }

        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                      data-action="insert_field"
                      data-field="${fieldKey}"
                      data-value='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                      title="Insert ${fieldInfo.label}">
                      ➕
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                      data-action="copy_field"
                      data-field="${fieldKey}"
                      data-value='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                      title="Copy ${fieldInfo.label}">
                      📋
                    </button>`;
        messageHtml += `</div></div>`;

        // Display content based on field type
        if (fieldKey === 'keywords' && Array.isArray(fieldValue)) {
          messageHtml += `<div class="ai-field-content ai-field-content--keywords">`;
          fieldValue.forEach(keyword => {
            messageHtml += `<span class="ai-keyword">${this.escapeHtml(keyword)}</span>`;
          });
          messageHtml += `</div>`;
        } else if (fieldKey === 'html_content') {
          // Show truncated HTML content preview
          const truncatedContent = fieldValue.length > 300
            ? fieldValue.substring(0, 300) + '...'
            : fieldValue;
          messageHtml += `<div class="ai-field-content ai-field-content--html-preview">`;
          messageHtml += `<div class="html-preview-text">${this.escapeHtml(truncatedContent)}</div>`;
          if (fieldValue.length > 300) {
            messageHtml += `<div class="ai-content-indicator">Content truncated for preview...</div>`;
          }
          messageHtml += `</div>`;
        } else if (fieldKey === 'plain_text') {
          // Show truncated plain text preview
          const truncatedText = fieldValue.length > 250
            ? fieldValue.substring(0, 250) + '...'
            : fieldValue;
          messageHtml += `<div class="ai-field-content ai-field-content--plain-text">`;
          messageHtml += `<div class="plain-text-preview">${this.escapeHtml(truncatedText)}</div>`;
          if (fieldValue.length > 250) {
            messageHtml += `<div class="ai-content-indicator">Text truncated for preview...</div>`;
          }
          messageHtml += `</div>`;
        } else {
          messageHtml += `<div class="ai-field-content">`;
          messageHtml += `<span class="field-value">${this.escapeHtml(fieldValue)}</span>`;
          messageHtml += `</div>`;
        }

        messageHtml += `</div>`;
      });

      // Summary-specific global actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__summary-actions">';
        messageHtml += '<h6>Summary Actions:</h6>';
        messageHtml += '<div class="ai-summary-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-summary-action-btn"
                        data-action="${action}"
                        data-summary-content='${JSON.stringify(content).replace(/'/g, "&#39;")}'
                        title="${this.getActionLabel(action)}">
                        ${this.getActionLabel(action)}
                      </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { content, actions, type: 'summary' });
    }

    displayMetaResponse(content, actions) {
      let messageHtml = '<div class="ai-response ai-response--meta">';

      // Add meta generation header
      messageHtml += '<div class="ai-meta-header">';
      messageHtml += '<h5 class="ai-meta-title">🏷️ SEO Meta Data Generated</h5>';
      messageHtml += '<p class="ai-meta-subtitle">Search engine optimized metadata ready for your content</p>';
      messageHtml += '</div>';

      // Define field mappings specific to meta generation
      const metaFieldMappings = {
        'meta_title': { label: 'Meta Title', icon: '🏷️', priority: 1 },
        'meta_description': { label: 'Meta Description', icon: '📝', priority: 2 },
        'url_alias': { label: 'URL Alias', icon: '🔗', priority: 3 },
        'keywords': { label: 'Keywords/Tags', icon: '🔑', priority: 4 }
      };

      // Sort fields by priority and only show fields that have content
      const sortedFields = Object.entries(metaFieldMappings)
        .filter(([fieldKey]) => content[fieldKey] && content[fieldKey] !== '')
        .sort(([, a], [, b]) => a.priority - b.priority);

      // Process each meta field
      sortedFields.forEach(([fieldKey, fieldInfo]) => {
        const fieldValue = content[fieldKey];

        messageHtml += `<div class="ai-field-item ai-field-item--meta" data-field="${fieldKey}">`;
        messageHtml += `<div class="ai-field-header">`;
        messageHtml += `<div class="ai-field-title-section">`;
        messageHtml += `<span class="ai-field-icon">${fieldInfo.icon}</span>`;
        messageHtml += `<h5 class="ai-field-title">${fieldInfo.label}</h5>`;
        messageHtml += `</div>`;

        // Action buttons for each field
        messageHtml += `<div class="ai-field-actions-header">`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                      data-action="insert_field"
                      data-field="${fieldKey}"
                      data-value='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                      title="Insert ${fieldInfo.label}">
                      ➕
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                      data-action="copy_field"
                      data-field="${fieldKey}"
                      data-value='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                      title="Copy ${fieldInfo.label}">
                      📋
                    </button>`;
        messageHtml += `</div></div>`;

        // Display content based on field type
        if (fieldKey === 'keywords' && Array.isArray(fieldValue)) {
          messageHtml += `<div class="ai-field-content ai-field-content--keywords">`;
          fieldValue.forEach(keyword => {
            messageHtml += `<span class="ai-keyword">${this.escapeHtml(keyword)}</span>`;
          });
          messageHtml += `</div>`;
        } else if (fieldKey === 'meta_description') {
          // Show character count for meta description
          const charCount = fieldValue.length;
          const isOptimal = charCount >= 120 && charCount <= 160;
          const charClass = isOptimal ? 'ai-char-count--optimal' : 'ai-char-count--warning';

          messageHtml += `<div class="ai-field-content ai-field-content--meta-description">`;
          messageHtml += `<span class="field-value">${this.escapeHtml(fieldValue)}</span>`;
          messageHtml += `<div class="ai-char-count ${charClass}">`;
          messageHtml += `<span>${charCount} characters</span>`;
          messageHtml += `<span class="ai-char-recommendation">${isOptimal ? '✅ Optimal length' : '⚠️ Consider 120-160 chars'}</span>`;
          messageHtml += `</div></div>`;
        } else if (fieldKey === 'meta_title') {
          // Show character count for meta title
          const charCount = fieldValue.length;
          const isOptimal = charCount >= 30 && charCount <= 60;
          const charClass = isOptimal ? 'ai-char-count--optimal' : 'ai-char-count--warning';

          messageHtml += `<div class="ai-field-content ai-field-content--meta-title">`;
          messageHtml += `<span class="field-value">${this.escapeHtml(fieldValue)}</span>`;
          messageHtml += `<div class="ai-char-count ${charClass}">`;
          messageHtml += `<span>${charCount} characters</span>`;
          messageHtml += `<span class="ai-char-recommendation">${isOptimal ? '✅ Optimal length' : '⚠️ Consider 30-60 chars'}</span>`;
          messageHtml += `</div></div>`;
        } else {
          messageHtml += `<div class="ai-field-content">`;
          messageHtml += `<span class="field-value">${this.escapeHtml(fieldValue)}</span>`;
          messageHtml += `</div>`;
        }

        messageHtml += `</div>`;
      });

      // Meta-specific global actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__meta-actions">';
        messageHtml += '<h6>Meta Actions:</h6>';
        messageHtml += '<div class="ai-meta-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-meta-action-btn"
                        data-action="${action}"
                        data-meta-content='${JSON.stringify(content).replace(/'/g, "&#39;")}'
                        title="${this.getActionLabel(action)}">
                        ${this.getActionLabel(action)}
                      </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { content, actions, type: 'meta' });
    }

    displayTableResponse(content, actions) {
      let messageHtml = '<div class="ai-response ai-response--table">';

      // Don't decode here - keep original HTML
      const tableHtml = content.html_content || '';

      // Table header
      messageHtml += '<div class="ai-table-header">';
      messageHtml += '<h5 class="ai-table-title">📊 Generated Table</h5>';
      if (content.plain_text) {
        messageHtml += `<p class="ai-table-description">${this.escapeHtml(content.plain_text)}</p>`;
      }
      messageHtml += '</div>';

      // Table content field
      if (content.html_content) {
        messageHtml += '<div class="ai-field-item ai-field-item--table" data-field="html_content">';
        messageHtml += '<div class="ai-field-header">';
        messageHtml += '<div class="ai-field-title-section">';
        messageHtml += '<span class="ai-field-icon">📋</span>';
        messageHtml += '<h5 class="ai-field-title">Table HTML</h5>';
        messageHtml += '</div>';
        messageHtml += '<div class="ai-field-actions-header">';

        // Store raw HTML content (don't JSON stringify it)
        const encodedHtml = tableHtml.replace(/'/g, "&#39;");

        messageHtml += `<button class="ai-preview-btn ai-field-action-btn"
                    data-action="preview_table"
                    data-table-html='${encodedHtml}'
                    title="Preview Table">
                    👁️
                  </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                    data-action="insert_field"
                    data-field="html_content"
                    data-value='${encodedHtml}'
                    title="Insert Table">
                    ➕
                  </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                    data-action="copy_field"
                    data-field="html_content"
                    data-value='${encodedHtml}'
                    title="Copy Table HTML">
                    📋
                  </button>`;
        messageHtml += '</div></div>';

        // Show table summary
        messageHtml += '<div class="ai-field-content ai-field-content--table-preview">';
        messageHtml += '<div class="ai-table-summary">HTML Table Ready for Insertion</div>';
        messageHtml += '</div></div>';
      }

      // Table actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__actions ai-table-actions">';
        messageHtml += '<h6>Table Actions:</h6>';
        messageHtml += '<div class="ai-table-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-table-action-btn"
                      data-action="${action}"
                      data-table-content='${JSON.stringify(content).replace(/'/g, "&#39;")}'
                      data-table-html='${tableHtml.replace(/'/g, "&#39;")}'
                      title="${this.getActionLabel(action)}">
                      ${this.getActionLabel(action)}
                    </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { content, actions, type: 'table' });
    }

    decodeHtmlEntities(html) {
      // Properly decode HTML entities and clean up formatting
      const textarea = document.createElement('textarea');
      textarea.innerHTML = html;
      let decoded = textarea.value;

      // Clean up extra newlines and quotes that might be causing display issues
      decoded = decoded.replace(/\\n/g, '\n')  // Convert literal \n to actual newlines
        .replace(/\\"/g, '"')   // Convert escaped quotes
        .replace(/\n\s*\n\s*\n/g, '\n\n') // Reduce multiple newlines
        .trim();

      return decoded;
    }

    extractTableInfo(tableHtml) {
      try {
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = tableHtml;

        const table = tempDiv.querySelector('table');
        if (!table) return { rows: 0, columns: 0, headers: [] };

        const headers = Array.from(table.querySelectorAll('th')).map(th => th.textContent.trim());
        const rows = table.querySelectorAll('tbody tr').length;
        const columns = headers.length || table.querySelectorAll('tr:first-child td, tr:first-child th').length;

        return {
          rows,
          columns: columns,
          headers: headers.slice(0, 3) // Show first 3 headers
        };
      } catch (error) {
        console.warn('Error extracting table info:', error);
        return { rows: 0, columns: 0, headers: [] };
      }
    }



    displayGraphResponse(graph, actions) {
      const chartId = `ai-chart-${Date.now()}`;

      let messageHtml = '<div class="ai-response ai-response--graph">';

      if (graph.title) {
        messageHtml += `<h6>${this.escapeHtml(graph.title)}</h6>`;
      }

      if (graph.description) {
        messageHtml += `<p class="ai-graph-description">${this.escapeHtml(graph.description)}</p>`;
      }

      // Chart container
      messageHtml += `<div class="ai-chart-container" style="position: relative; height: 400px; width: 100%; margin: 20px 0;">`;
      messageHtml += `<canvas id="${chartId}" width="400" height="400"></canvas>`;
      messageHtml += `</div>`;

      // Data insights
      if (graph.data_insights && graph.data_insights.length > 0) {
        messageHtml += '<div class="ai-graph-insights">';
        messageHtml += '<h6>📊 Key Insights:</h6>';
        messageHtml += '<ul>';
        graph.data_insights.forEach(insight => {
          messageHtml += `<li>${this.escapeHtml(insight)}</li>`;
        });
        messageHtml += '</ul></div>';
      }

      // Enhanced action buttons for graphs
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__actions ai-graph-actions">';
        messageHtml += '<h6>Actions:</h6>';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-graph-action-btn"
                      data-action="${action}"
                      data-chart-id="${chartId}"
                      data-graph-data='${JSON.stringify(graph).replace(/'/g, "&#39;")}'
                      title="${this.getActionLabel(action)}">
                      ${this.getActionLabel(action)}
                    </button>`;
        });
        messageHtml += '</div>';
      }

      messageHtml += '</div>';

      this.addMessage('assistant', messageHtml, { graph, actions, chartId });

      // Render the chart after DOM insertion
      setTimeout(() => {
        this.renderChart(chartId, graph);
      }, 100);
    }

    renderChart(chartId, graphData) {
      const ctx = document.getElementById(chartId);
      if (!ctx) {
        console.error('Chart canvas not found:', chartId);
        return;
      }

      try {
        // Load Chart.js if not already loaded
        if (typeof Chart === 'undefined') {
          this.loadChartJS(() => {
            this.createChart(ctx, graphData);
          });
        } else {
          this.createChart(ctx, graphData);
        }
      } catch (error) {
        console.error('Error rendering chart:', error);
        ctx.parentElement.innerHTML = '<div class="ai-chart-error">Failed to render chart</div>';
      }
    }


    createChart(ctx, graphData) {
      const chart = new Chart(ctx, {
        type: graphData.chart_type || 'bar',
        data: graphData.chart_data,
        options: {
          ...graphData.chart_options,
          animation: {
            onComplete: function () {
              // Store chart instance for later use
              ctx.chartInstance = this;
            }
          }
        }
      });

      // Store chart reference
      ctx.chartInstance = chart;
    }

    loadChartJS(callback) {
      if (document.querySelector('script[src*="chart.js"]')) {
        callback();
        return;
      }

      const script = document.createElement('script');
      script.src = 'https://cdnjs.cloudflare.com/ajax/libs/Chart.js/3.9.1/chart.min.js';
      script.onload = callback;
      script.onerror = () => {
        console.error('Failed to load Chart.js');
      };
      document.head.appendChild(script);
    }

    displayAnalysisResponse(analysis, actions) {
      let messageHtml = '<div class="ai-response ai-response--analysis">';

      // SEO Score
      if (analysis.score !== null && analysis.score !== undefined) {
        const scoreClass = analysis.score >= 80 ? 'good' : analysis.score >= 60 ? 'medium' : 'poor';
        messageHtml += `<div class="ai-analysis-score ai-analysis-score--${scoreClass}">
      <span class="ai-score-label">SEO Score:</span>
      <span class="ai-score-value">${analysis.score}/100</span>
    </div>`;
      }

      // Analysis text
      if (analysis.analysis_text) {
        messageHtml += `<div class="ai-analysis-text">
      <h6>📊 Analysis Overview</h6>
      <p>${this.escapeHtml(analysis.analysis_text)}</p>
    </div>`;
      }

      // Metrics
      if (analysis.metrics) {
        messageHtml += '<div class="ai-analysis-metrics">';
        messageHtml += '<h6>📈 Metrics</h6>';
        messageHtml += '<div class="ai-metrics-grid">';

        Object.entries(analysis.metrics).forEach(([key, value]) => {
          const displayKey = key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
          let displayValue = value;

          if (typeof value === 'object') {
            displayValue = Object.entries(value).map(([k, v]) => `${k}: ${v}`).join(', ');
          }

          messageHtml += `<div class="ai-metric-item">
        <span class="ai-metric-label">${displayKey}:</span>
        <span class="ai-metric-value">${this.escapeHtml(displayValue)}</span>
      </div>`;
        });

        messageHtml += '</div></div>';
      }

      // Suggestions
      if (analysis.suggestions && analysis.suggestions.length > 0) {
        messageHtml += '<div class="ai-analysis-suggestions">';
        messageHtml += '<h6>💡 Suggestions for Improvement</h6>';
        messageHtml += '<ul class="ai-suggestions-list">';
        analysis.suggestions.forEach(suggestion => {
          messageHtml += `<li>${this.escapeHtml(suggestion)}</li>`;
        });
        messageHtml += '</ul></div>';
      }

      // Actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-analysis-actions">';
        messageHtml += '<h6>Quick Actions:</h6>';
        messageHtml += '<div class="ai-global-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-image-action-btn ai-action-btn--global" data-action="${action}">
                    ${this.getActionLabel(action)}
                  </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { analysis, actions });
    }

    handleApiError(response) {
      const errorCode = response.error_code;

      switch (errorCode) {
        case 'NOT_AUTHENTICATED':
        case 'AUTH_ERROR':
          this.authState.authenticated = false;
          this.showAuthenticationInterface();
          this.addMessage('error', 'Session expired. Please login again.');
          break;

        case 'USAGE_LIMIT_EXCEEDED':
        case 'SUBSCRIPTION_REQUIRED':
          this.showSubscriptionInterface();
          this.addMessage('error', response.error);
          break;

        default:
          this.addMessage('error', response.error || 'Something went wrong');
      }
    }

    displayOptimizedContentResponse(content, actions) {
      let messageHtml = '<div class="ai-response ai-response--content ai-response--optimization">';

      // Add optimization header
      messageHtml += '<div class="ai-optimization-header">';
      messageHtml += '<h5 class="ai-optimization-title">🔧 Content Optimization Results</h5>';
      messageHtml += '<p class="ai-optimization-subtitle">Your content has been optimized for SEO and readability</p>';
      messageHtml += '</div>';

      const optimizedContent = content.optimized_content;
      // Display optimized title if available
      if (optimizedContent.title) {
        messageHtml += '<div class="ai-field-item ai-field-item--optimized" data-field="title">';
        messageHtml += '<div class="ai-field-header">';
        messageHtml += '<div class="ai-field-title-section">';
        messageHtml += '<span class="ai-field-icon">📰</span>';
        messageHtml += '<h5 class="ai-field-title">Optimized Title</h5>';
        messageHtml += '</div>';
        messageHtml += '<div class="ai-field-actions-header">';
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                      data-action="insert_field"
                      data-field="title"
                      data-value='${JSON.stringify(optimizedContent.title).replace(/'/g, "&#39;")}'
                      title="Insert Optimized Title">
                      ➕
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                      data-action="copy_field"
                      data-field="title"
                      data-value='${JSON.stringify(optimizedContent.title).replace(/'/g, "&#39;")}'
                      title="Copy Optimized Title">
                      📋
                    </button>`;
        messageHtml += '</div></div>';
        messageHtml += '<div class="ai-field-content">';
        messageHtml += `<span class="field-value">${this.escapeHtml(optimizedContent.title)}</span>`;
        messageHtml += '</div></div>';
      }

      // Display optimized content
      if (optimizedContent.content) {
        messageHtml += '<div class="ai-field-item ai-field-item--optimized" data-field="html_content">';
        messageHtml += '<div class="ai-field-header">';
        messageHtml += '<div class="ai-field-title-section">';
        messageHtml += '<span class="ai-field-icon">📄</span>';
        messageHtml += '<h5 class="ai-field-title">Optimized Content</h5>';
        messageHtml += '</div>';
        messageHtml += '<div class="ai-field-actions-header">';
        messageHtml += `<button class="ai-preview-btn ai-field-action-btn"
                      data-action="preview_html"
                      data-content='${JSON.stringify(optimizedContent.content).replace(/'/g, "&#39;")}'>
                      👁️
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                      data-action="insert_field"
                      data-field="html_content"
                      data-value='${JSON.stringify(optimizedContent.content).replace(/'/g, "&#39;")}'
                      title="Insert Optimized Content">
                      ➕
                    </button>`;
        messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                      data-action="copy_field"
                      data-field="html_content"
                      data-value='${JSON.stringify(optimizedContent.content).replace(/'/g, "&#39;")}'
                      title="Copy Optimized Content">
                      📋
                    </button>`;
        messageHtml += '</div></div>';

        // Show truncated content preview
        const truncatedContent = optimizedContent.content.length > 200
          ? optimizedContent.content.substring(0, 200) + '...'
          : optimizedContent.content;
        messageHtml += '<div class="ai-field-content ai-field-content--html-preview">';
        messageHtml += `<div class="html-preview-text">${this.escapeHtml(truncatedContent)}</div>`;
        messageHtml += '</div></div>';
      }

      // Display improvement notes if available
      if (content.improvement_notes && content.improvement_notes.length > 0) {
        messageHtml += '<div class="ai-improvement-notes">';
        messageHtml += '<h6>💡 Optimization Improvements Applied</h6>';
        messageHtml += '<ul class="ai-improvements-list">';
        content.improvement_notes.forEach(note => {
          messageHtml += `<li>${this.escapeHtml(note)}</li>`;
        });
        messageHtml += '</ul></div>';
      }

      // Display field suggestions if available
      if (content.field_suggestions && Object.keys(content.field_suggestions).length > 0) {
        messageHtml += '<div class="ai-field-suggestions">';
        messageHtml += '<h6>📝 Field Suggestions</h6>';
        messageHtml += '<div class="ai-suggestions-grid">';
        Object.entries(content.field_suggestions).forEach(([field, suggestion]) => {
          const displayField = field.charAt(0).toUpperCase() + field.slice(1);
          messageHtml += `<div class="ai-suggestion-item">
        <span class="ai-suggestion-field">${displayField}:</span>
        <span class="ai-suggestion-text">${this.escapeHtml(suggestion)}</span>
      </div>`;
        });
        messageHtml += '</div></div>';
      }

      // Global actions for optimized content
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__global-actions">';
        messageHtml += '<h6>Quick Actions:</h6>';
        messageHtml += '<div class="ai-global-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-image-action-btn ai-action-btn--global" data-action="${action}">
                        ${this.getActionLabel(action)}
                      </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';

      // Store the optimized content in a format compatible with existing methods
      const contentForStorage = {
        title: optimizedContent.title || null,
        html_content: optimizedContent.content || null,
        improvement_notes: content.improvement_notes || [],
        field_suggestions: content.field_suggestions || {}
      };

      this.addMessage('assistant', messageHtml, { content: contentForStorage, actions });
    }

    displayContentResponse(content, actions) {
      let messageHtml = '<div class="ai-response ai-response--content">';

      const fieldMappings = {
        'title': { label: 'Title', icon: '📰', priority: 1 },
        'kicker': { label: 'Kicker/Subtitle', icon: '💫', priority: 2 },
        'html_content': { label: 'Content', icon: '📄', priority: 3 },
        'meta_title': { label: 'Meta Title', icon: '🏷️', priority: 4 },
        'meta_description': { label: 'Meta Description', icon: '📝', priority: 5 },
        'url_alias': { label: 'URL Alias', icon: '🔗', priority: 6 },
        'keywords': { label: 'Keywords', icon: '🔑', priority: 7 }
      };

      // Sort fields by priority
      const sortedFields = Object.entries(fieldMappings)
        .sort(([, a], [, b]) => a.priority - b.priority);

      // Process each field
      sortedFields.forEach(([fieldKey, fieldInfo]) => {
        const fieldValue = content[fieldKey];

        if (fieldValue && fieldValue !== '') {
          messageHtml += `<div class="ai-field-item" data-field="${fieldKey}">`;
          messageHtml += `<div class="ai-field-header">`;
          messageHtml += `<div class="ai-field-title-section">`;
          messageHtml += `<span class="ai-field-icon">${fieldInfo.icon}</span>`;
          messageHtml += `<h5 class="ai-field-title">${fieldInfo.label}</h5>`;
          messageHtml += `</div>`;

          // Action buttons
          messageHtml += `<div class="ai-field-actions-header">`;
          if (fieldKey === 'html_content') {
            messageHtml += `<button class="ai-preview-btn ai-field-action-btn" data-action="preview_html" data-content='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'}>
                      👁️
                    </button>`;
          }
          messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--insert"
                        data-action="insert_field"
                        data-field="${fieldKey}"
                        data-value='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                        title="Insert ${fieldInfo.label}">
                        ➕
                      </button>`;
          messageHtml += `<button class="ai-field-action-btn ai-field-action-btn--copy"
                        data-action="copy_field"
                        data-field="${fieldKey}"
                        data-value='${JSON.stringify(fieldValue).replace(/'/g, "&#39;")}'
                        title="Copy ${fieldInfo.label}">
                        📋
                      </button>`;
          messageHtml += `</div></div>`;

          // Display content
          if (fieldKey === 'keywords' && Array.isArray(fieldValue)) {
            messageHtml += `<div class="ai-field-content ai-field-content--keywords">`;
            fieldValue.forEach(keyword => {
              messageHtml += `<span class="ai-keyword">${this.escapeHtml(keyword)}</span>`;
            });
            messageHtml += `</div>`;
          } else if (fieldKey === 'html_content') {
            const truncatedContent = fieldValue.length > 200
              ? fieldValue.substring(0, 200) + '...'
              : fieldValue;
            messageHtml += `<div class="ai-field-content ai-field-content--html-preview">`;
            messageHtml += `<div class="html-preview-text">${this.escapeHtml(truncatedContent)}</div>`;
            messageHtml += `</div>`;
          } else {
            messageHtml += `<div class="ai-field-content">`;
            messageHtml += `<span class="field-value">${this.escapeHtml(fieldValue)}</span>`;
            messageHtml += `</div>`;
          }

          messageHtml += `</div>`;
        }
      });

      // Global actions
      if (actions.length > 0) {
        messageHtml += '<div class="ai-response__global-actions">';
        messageHtml += '<h6>Quick Actions:</h6>';
        messageHtml += '<div class="ai-global-actions-grid">';
        actions.forEach(action => {
          messageHtml += `<button class="ai-action-btn ai-image-action-btn ai-action-btn--global" data-action="${action}">
                        ${this.getActionLabel(action)}
                      </button>`;
        });
        messageHtml += '</div></div>';
      }

      messageHtml += '</div>';
      this.addMessage('assistant', messageHtml, { content, actions });
    }

    displayImageResponse(image, actions) {
      let messageHtml = `
    <div class="ai-response ai-response--image">
      <img src="${image.image_url}" alt="${this.escapeHtml(image.alt_text || '')}" class="ai-response__image">
      ${image.caption ? `<p class="ai-response__caption">${this.escapeHtml(image.caption)}</p>` : ''}
      <div class="ai-response__actions">
  `;

      // Enhanced action buttons for images
      const imageActions = [
        { action: 'insert_image', label: '📷 Insert Image', data: image.image_url },
        { action: 'download_image', label: '⬇️ Download Image', data: image.image_url },
        { action: 'copy_image_url', label: '🔗 Copy URL', data: image.image_url },
        { action: 'preview_image', label: '👁️ Preview', data: image.image_url }
      ];

      imageActions.forEach(actionInfo => {
        messageHtml += `<button class="ai-action-btn ai-image-action-btn"
                        data-action="${actionInfo.action}"
                        data-image-url="${actionInfo.data}"
                        title="${actionInfo.label}">
                      ${actionInfo.label}
                    </button>`;
      });

      messageHtml += '</div></div>';
      this.addMessage('assistant', messageHtml, { image, actions });
    }

    getActionLabel(action) {
      const labels = {
        'insert_content': '📝 Insert',
        'copy_content': '📋 Copy All',
        'regenerate': '🔄 Regenerate',
        'optimize_seo': '📈 Optimize SEO',
        'reanalyze': '🔍 Re-analyze',
        'compare_original': '⚖️ Compare Original',
        'download_data': '⬇️ Download Data',
        'export_csv': '📄 Export CSV',
        'download_chart': '🖼️ Download Image',
        'insert_meta': '🏷️ Insert All Meta',
        'copy_meta': '📋 Copy Meta Data',
        'optimize_more': '🔧 Optimize More',
        'generate_variations': '🎯 Generate Variations',
        'create_longer_summary': '📄 Create Longer Summary',
        'extract_highlights': '✨ Extract Highlights',
        'improve_readability': '📖 Improve Readability',
        'insert_keywords': '🔑 Insert Keywords',
        'copy_keywords': '📋 Copy Keywords',
        'analyze_competition': '🥊 Analyze Competition',
        'expand_keywords': '🔍 Expand Keywords',
        'insert_title': '📰 Insert Title',
        'copy_title': '📋 Copy Title',
        'seo_optimize_title': '🔍 SEO Optimize Title',
        'optimize_length': '📏 Optimize Length'
      };
      return labels[action] || action.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
    }

    handleAction(action, $button) {
      try {
        switch (action) {
          case 'insert_field':
            const insertFieldKey = $button.data('field');
            const insertFieldValue = this.parseButtonData($button.data('value'));
            if (insertFieldKey === 'html_content' && this.isTableHtml(insertFieldValue)) {
              this.insertTableIntoBodyField(insertFieldValue);
            } else {
              this.insertSingleField(insertFieldKey, insertFieldValue);
            }
            break;

          case 'copy_field':
            const copyFieldKey = $button.data('field');
            const copyFieldValue = this.parseButtonData($button.data('value'));
            this.copySingleField(copyFieldKey, copyFieldValue);
            break;

          case 'insert_content':
            if ($button.data('chart-id')) {
              this.insertGraphAsImage($button);
            } else if ($button.data('table-html') || $button.data('table-content')) {
              this.insertTableContent($button);
            } else {
              this.insertAllContent();
            }
            break;

          case 'copy_content':
            this.copyAllContent();
            break;

          case 'regenerate':
            this.regenerateContent();
            break;

          case 'optimize_seo':
            this.optimizeSeo();
            break;

          case 'preview_html':
            const htmlContent = this.parseButtonData($button.data('content'));
            this.showHtmlPreview(htmlContent);
            break;

          case 'insert_image':
            this.insertImageIntoField($button.data('image-url'));
            break;

          case 'download_image':
            this.downloadImage($button.data('image-url'));
            break;

          case 'generate_alt_text':
            this.generateAltText($button.data('image-url'));
            break;

          case 'copy_image_url':
            this.copyToClipboard($button.data('image-url'), 'Image URL copied to clipboard!');
            break;

          case 'preview_image':
            this.showImagePreview($button.data('image-url'));
            break;

          case 'apply_suggestions':
            this.applySuggestions();
            break;

          case 'export_report':
            this.exportAnalysisReport();
            break;

          case 'reanalyze':
            this.reanalyzeContent();
            break;

          case 'optimize_content':
            this.optimizeContent();
            break;

          case 'compare_original':
            this.compareOriginalContent();
            break;

          case 'download_data':
            if ($button.data('chart-id')) {
              this.downloadGraphData($button);
            } else if ($button.data('table-content')) {
              this.downloadTableData($button);
            } else {
              this.showNotification('No data available to download', 'warning');
            }
            break;

          case 'export_csv':
            if ($button.data('chart-id')) {
              this.exportGraphAsCSV($button);
            } else if ($button.data('table-content')) {
              this.exportTableAsCSV($button);
            } else {
              this.showNotification('No data available to export', 'warning');
            }
            break;

          case 'download_chart':
            this.downloadChartAsImage($button);
            break;

          case 'preview_table':
            const tableContent = this.parseButtonData($button.data('content'));
            this.showTablePreview(tableContent);
            break;

          case 'insert_meta':
            const metaContent = this.parseButtonData($button.data('meta-content'));
            this.insertAllMetaFields(metaContent);
            break;

          case 'copy_meta':
            const copyMetaContent = this.parseButtonData($button.data('meta-content'));
            this.copyAllMetaContent(copyMetaContent);
            break;

          case 'optimize_more':
            this.optimizeMetaMore();
            break;

          case 'generate_variations':
            this.generateMetaVariations();
            break;

          case 'create_longer_summary':
            this.createLongerSummary();
            break;

          case 'extract_highlights':
            this.extractHighlights();
            break;

          case 'improve_readability':
            this.improveReadability();
            break;

          case 'insert_keywords':
            const keywordContent = this.parseButtonData($button.data('keyword-content'));
            this.insertKeywordsOnly(keywordContent);
            break;

          case 'copy_keywords':
            const copyKeywordContent = this.parseButtonData($button.data('keyword-content'));
            this.copyKeywordsOnly(copyKeywordContent);
            break;

          case 'analyze_competition':
            this.analyzeCompetition();
            break;

          case 'expand_keywords':
            this.expandKeywords();
            break;

          case 'insert_title':
            const titleContent = this.parseButtonData($button.data('title-content'));
            this.insertTitleOnly(titleContent);
            break;

          case 'copy_title':
            const copyTitleContent = this.parseButtonData($button.data('title-content'));
            this.copyTitleOnly(copyTitleContent);
            break;

          case 'seo_optimize_title':
            this.seoOptimizeTitle();
            break;

          case 'optimize_length':
            this.optimizeDescriptionLength();
            break;

          default:
            this.showNotification('Action not implemented: ' + action, 'info');
        }
      } catch (error) {
        console.error('Error handling action:', error);
        this.showNotification('Failed to perform action: ' + error.message, 'error');
      }
    }

    async generateAltText(imageUrl) {
      try {
        this.setProcessingState(true);
        this.showNotification('Generating alt text for image...', 'info');

        const payload = {
          task_type: 'generate_alt_text',
          image_url: imageUrl,
          content_type: this.contentType
        };

        const response = await this.makeApiRequest(payload);

        if (response.success && response.data.alt_text) {
          const altText = response.data.alt_text;
          this.copyToClipboard(altText, 'Alt text copied to clipboard!');
          this.addMessage('assistant', `Generated Alt Text: "${altText}"`);
        } else {
          throw new Error(response.error || 'Failed to generate alt text');
        }
      } catch (error) {
        console.error('Alt text generation failed:', error);
        this.showNotification('Failed to generate alt text: ' + error.message, 'error');
      } finally {
        this.setProcessingState(false);
      }
    }

    downloadImage(imageUrl) {
      try {
        const a = document.createElement('a');
        a.href = imageUrl;
        a.download = `ai-generated-image-${Date.now()}.png`;
        a.target = '_blank';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        this.showNotification('Image download started!', 'success');
      } catch (error) {
        console.error('Download failed:', error);
        this.showNotification('Download failed. Opening image in new tab...', 'warning');
        window.open(imageUrl, '_blank');
      }
    }

    async insertImageIntoField(imageUrl) {
      try {
        // Set processing state like chart insertion
        this.setProcessingState(true);
        this.showNotification('Processing image...', 'info');

        // Upload the image from URL to server (similar to uploadChartImage)
        const localImageUrl = await this.uploadImageFromUrl(imageUrl);

        if (localImageUrl) {
          // Insert into body field using existing method (same as chart)
          this.insertImageIntoBodyField(localImageUrl, 'image');
        } else {
          // Fallback: try to insert URL directly into image fields
          const inserted = this.tryInsertIntoImageFields(imageUrl);
          if (!inserted) {
            // Final fallback: copy to clipboard
            this.copyToClipboard(imageUrl, 'Upload failed. Image URL copied to clipboard!');
          }
        }

      } catch (error) {
        console.error('Error inserting image:', error);
        this.showNotification('Failed to insert image: ' + error.message, 'error');

        // Fallback: try direct insertion or clipboard
        const inserted = this.tryInsertIntoImageFields(imageUrl);
        if (!inserted) {
          this.copyToClipboard(imageUrl, 'Error occurred. Image URL copied to clipboard!');
        }
      } finally {
        this.setProcessingState(false);
      }
    }

    /**
 * Upload image from URL to server (similar to uploadChartImage)
 */
    async uploadImageFromUrl(imageUrl) {
      try {
        // Get CSRF token
        const csrfToken = drupalSettings.user?.csrf_token || '';

        const uploadResponse = await fetch('/ai-assistant/upload-image-from-url', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': csrfToken
          },
          body: JSON.stringify({
            imageUrl: imageUrl,
            type: 'ai-image'
          })
        });

        if (uploadResponse.ok) {
          const result = await uploadResponse.json();
          if (result.success && result.data && result.data.url) {
            return result.data.absolute_url || result.data.url;
          } else {
            this.showNotification(result.error || 'Upload failed', 'error');
            return null;
          }
        } else {
          const errorResult = await uploadResponse.json().catch(() => ({}));
          this.showNotification(errorResult.error || 'Upload failed', 'error');
          return null;
        }

      } catch (error) {
        this.showNotification('Network error during upload', 'error');
        return null;
      }
    }

    showHtmlPreview(htmlContent) {
      $('#ai-preview-modal').remove();

      const modalHtml = `
    <div class="ai-modal-overlay" id="ai-preview-modal">
      <div class="ai-modal ai-modal--large">
        <div class="ai-modal-header">
          <h3>HTML Content Preview</h3>
          <button class="ai-modal-close" id="close-preview-modal">×</button>
        </div>
        <div class="ai-modal-body">
          <iframe id="preview-iframe" style="width: 100%; height: 400px; border: 1px solid #ddd;"></iframe>
        </div>
        <div class="ai-modal-footer">
          <button id="close-modal-btn" class="ai-btn ai-btn--primary">Close</button>
        </div>
      </div>
    </div>
  `;

      $('body').append(modalHtml);

      // Set iframe content safely
      const iframe = document.getElementById('preview-iframe');
      iframe.srcdoc = htmlContent;

      // Bind close events
      $('#close-preview-modal, #close-modal-btn').on('click', function () {
        $('#ai-preview-modal').remove();
      });

      // Bind copy event
      $('#copy-html-btn').on('click', function () {
        navigator.clipboard.writeText(htmlContent).then(() => {
          $(this).text('Copied!').addClass('copied');
          setTimeout(() => {
            $(this).text('Copy HTML').removeClass('copied');
          }, 2000);
        });
      });

      // Close on overlay click
      $('#ai-preview-modal').on('click', function (e) {
        if (e.target === this) {
          $(this).remove();
        }
      });
    }

    showImagePreview(imageUrl) {
      const modalHtml = `
    <div class="ai-modal-overlay" id="ai-preview-modal">
      <div class="ai-modal">
        <div class="ai-modal-header">
          <h3>Image Preview</h3>
          <button class="ai-modal-close" onclick="$('#ai-preview-modal').remove()">×</button>
        </div>
        <div class="ai-modal-body">
          <img src="${imageUrl}" style="max-width: 100%; height: auto;" alt="Preview">
        </div>
        <div class="ai-modal-footer">
          <button onclick="window.open('${imageUrl}', '_blank')">Open Full Size</button>
          <button onclick="jQuery('#ai-preview-modal').remove()">Close</button>
        </div>
      </div>
    </div>
  `;
      $('body').append(modalHtml);
    }

    applySuggestions() {
      try {
        // First try to get SEO analysis suggestions
        const lastAnalysis = this.getLastAnalysisResponse();



        // Then try to get content optimization improvement notes
        const lastOptimization = this.getLastOptimizationResponse();


        let suggestions = [];
        let title = 'Suggestions';
        let optimizedContent = null;

        // Check for SEO analysis suggestions
        if (lastAnalysis && lastAnalysis.suggestions && lastAnalysis.suggestions.length > 0) {
          suggestions = lastAnalysis.suggestions;
          title = 'SEO Improvement Suggestions';
        }
        // Check for content optimization improvement notes
        else if (lastOptimization && lastOptimization.improvement_notes && lastOptimization.improvement_notes.length > 0) {
          suggestions = lastOptimization.improvement_notes;
          title = 'Content Optimization Improvements';

          // Also get optimized content if available
          if (lastOptimization) {
            optimizedContent = lastOptimization;
          }
        }

        if (lastOptimization && lastOptimization.improvement_notes && lastOptimization.improvement_notes.length > 0) {
          if (lastOptimization) {
            optimizedContent = lastOptimization;
          }
        }

        if (suggestions.length === 0 && !optimizedContent) {
          this.showNotification('No suggestions or optimized content found', 'warning');
          return;
        }

        // Show enhanced modal with both suggestions and optimized content
        this.showEnhancedSuggestionsModal(suggestions, title, optimizedContent);

      } catch (error) {
        console.error('Error applying suggestions:', error);
        this.showNotification('Failed to apply suggestions: ' + error.message, 'error');
      }
    }

    getLastOptimizationResponse() {
      if (this.chatHistory && this.chatHistory.length > 0) {
        for (let i = this.chatHistory.length - 1; i >= 0; i--) {
          const message = this.chatHistory[i];
          if (message.type === 'assistant' && message.data && message.data.content) {
            const content = message.data.content;
            // Check if this is an optimization response with improvement notes or optimized content
            if ((content.improvement_notes && Array.isArray(content.improvement_notes)) ||
              (content.optimized_content && typeof content.optimized_content === 'object')) {
              return content;
            }
          }
        }
      }
      return null;
    }

    showEnhancedSuggestionsModal(suggestions, title = 'Suggestions', optimizedContent = null) {
      // Remove existing modal if any
      $('#ai-suggestions-modal').remove();

      const modalHtml = `
    <div class="ai-modal-overlay" id="ai-suggestions-modal">
      <div class="ai-modal ai-modal--suggestions">
        <div class="ai-modal-header">
          <h3>${this.escapeHtml(title)}</h3>
          <button class="ai-modal-close" id="close-suggestions-modal">×</button>
        </div>
        <div class="ai-modal-body">
          ${suggestions.length > 0 ? this.renderSuggestionsSection(suggestions, title) : ''}
          ${optimizedContent ? this.renderOptimizedContentSection(optimizedContent) : ''}
        </div>
        <div class="ai-modal-footer">
          ${suggestions.length > 0 ? '<button id="copy-suggestions-btn" class="ai-btn ai-btn--secondary">Copy Suggestions</button>' : ''}
          ${optimizedContent ? '<button id="insert-optimized-btn" class="ai-btn ai-btn--primary">Insert Optimized Content</button>' : ''}
          ${optimizedContent ? '<button id="copy-optimized-btn" class="ai-btn ai-btn--outline">Copy Optimized Content</button>' : ''}
          <button id="close-suggestions-btn" class="ai-btn ai-btn--outline">Close</button>
        </div>
      </div>
    </div>
  `;

      $('body').append(modalHtml);

      // Bind close events
      $('#close-suggestions-modal, #close-suggestions-btn').on('click', () => {
        $('#ai-suggestions-modal').remove();
      });

      // Bind copy suggestions event
      $('#copy-suggestions-btn').on('click', () => {
        const suggestionsText = suggestions.map((s, i) => `${i + 1}. ${s}`).join('\n');
        this.copyToClipboard(`${title}:\n\n${suggestionsText}`, 'Suggestions copied to clipboard!');
      });

      // Bind insert optimized content event
      $('#insert-optimized-btn').on('click', () => {
        this.insertOptimizedContent(optimizedContent);
        $('#ai-suggestions-modal').remove();
      });

      // Bind copy optimized content event
      $('#copy-optimized-btn').on('click', () => {
        this.copyOptimizedContent(optimizedContent);
      });
    }

    copyOptimizedContent(optimizedContent) {
      try {
        let textToCopy = 'Optimized Content:\n\n';

        if (optimizedContent.title) {
          textToCopy += `TITLE:\n${optimizedContent.title}\n\n`;
        }

        if (optimizedContent.content) {
          // Clean HTML for clipboard
          const cleanContent = optimizedContent.content.replace(/<[^>]*>/g, '').replace(/\s+/g, ' ').trim();
          textToCopy += `CONTENT:\n${cleanContent}`;
        }

        this.copyToClipboard(textToCopy, 'Optimized content copied to clipboard!');

      } catch (error) {
        console.error('Error copying optimized content:', error);
        this.showNotification('Failed to copy optimized content: ' + error.message, 'error');
      }
    }

    insertOptimizedContent(optimizedContent) {
      try {
        let insertedCount = 0;

        // Insert optimized title
        if (optimizedContent.title) {
          this.insertSingleField('title', optimizedContent.title);
          insertedCount++;
        }

        // Insert optimized content
        if (optimizedContent.content) {
          this.insertSingleField('html_content', optimizedContent.content);
          insertedCount++;
        }

        if (optimizedContent.html_content) {
          this.insertSingleField('html_content', optimizedContent.html_content);
          insertedCount++;
        }

        if (insertedCount > 0) {
          this.showNotification(`Successfully inserted ${insertedCount} optimized field(s)!`, 'success');
        } else {
          this.showNotification('No optimized content to insert', 'warning');
        }

      } catch (error) {
        console.error('Error inserting optimized content:', error);
        this.showNotification('Failed to insert optimized content: ' + error.message, 'error');
      }
    }


    showSuggestionsModal(suggestions, title = 'Suggestions') {
      const modalHtml = `
    <div class="ai-modal-overlay" id="ai-suggestions-modal">
      <div class="ai-modal">
        <div class="ai-modal-header">
          <h3>${this.escapeHtml(title)}</h3>
          <button class="ai-modal-close" id="close-suggestions-modal">×</button>
        </div>
        <div class="ai-modal-body">
          <ul class="ai-suggestions-list">
            ${suggestions.map((suggestion, index) =>
        `<li><strong>${index + 1}.</strong> ${this.escapeHtml(suggestion)}</li>`
      ).join('')}
          </ul>
        </div>
        <div class="ai-modal-footer">
          <button id="copy-suggestions-btn" class="ai-btn ai-btn--secondary">Copy All</button>
          <button id="close-suggestions-btn" class="ai-btn ai-btn--primary">Close</button>
        </div>
      </div>
    </div>
  `;

      $('body').append(modalHtml);

      $('#close-suggestions-modal, #close-suggestions-btn').on('click', () => {
        $('#ai-suggestions-modal').remove();
      });

      $('#copy-suggestions-btn').on('click', () => {
        const suggestionsText = suggestions.map((s, i) => `${i + 1}. ${s}`).join('\n');
        this.copyToClipboard(`${title}:\n\n${suggestionsText}`, 'All suggestions copied!');
      });
    }

    renderOptimizedContentSection(optimizedContent) {
      let html = `
    <div class="ai-optimized-content-section">
      <div class="ai-section-title">
        <span>✨</span>
        <span>Optimized Content</span>
      </div>
  `;

      if (optimizedContent.title) {
        html += `
      <div class="ai-optimized-field">
        <div class="ai-optimized-field-label">Optimized Title:</div>
        <div class="ai-optimized-field-content">
          <div class="ai-optimized-title">${this.escapeHtml(optimizedContent.title)}</div>
        </div>
      </div>
    `;
      }

      if (optimizedContent.html_content) {
        // Clean HTML for preview
        const cleanContent = optimizedContent.html_content.replace(/<[^>]*>/g, '').replace(/\s+/g, ' ').trim();
        const previewContent = cleanContent.length > 300 ? cleanContent.substring(0, 300) + '...' : cleanContent;

        html += `
      <div class="ai-optimized-field">
        <div class="ai-optimized-field-label">Optimized Content:</div>
        <div class="ai-optimized-field-content">
          <div class="ai-optimized-content">${this.escapeHtml(previewContent)}</div>
          ${cleanContent.length > 300 ? '<div style="margin-top: 10px; font-style: italic; color: #6c757d; font-size: 12px;">Content truncated for preview...</div>' : ''}
        </div>
      </div>
    `;
      }

      html += '</div>';
      return html;
    }

    exportAnalysisReport() {
      try {
        const lastAnalysis = this.getLastAnalysisResponse();
        if (!lastAnalysis) {
          this.showNotification('No analysis report to export', 'warning');
          return;
        }

        let report = `SEO Analysis Report\n`;
        report += `=====================\n`;
        report += `Generated: ${new Date().toLocaleString()}\n\n`;

        if (lastAnalysis.score !== null && lastAnalysis.score !== undefined) {
          report += `Overall SEO Score: ${lastAnalysis.score}/100\n\n`;
        }

        if (lastAnalysis.analysis_text) {
          report += `Analysis Overview:\n`;
          report += `${lastAnalysis.analysis_text}\n\n`;
        }

        if (lastAnalysis.metrics && Object.keys(lastAnalysis.metrics).length > 0) {
          report += `Detailed Metrics:\n`;
          report += `================\n`;
          Object.entries(lastAnalysis.metrics).forEach(([key, value]) => {
            const displayKey = key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
            let displayValue = value;

            if (typeof value === 'object') {
              displayValue = Object.entries(value)
                .map(([k, v]) => `${k}: ${v}`)
                .join(', ');
            }

            report += `• ${displayKey}: ${displayValue}\n`;
          });
          report += `\n`;
        }

        if (lastAnalysis.suggestions && lastAnalysis.suggestions.length > 0) {
          report += `Recommendations:\n`;
          report += `===============\n`;
          lastAnalysis.suggestions.forEach((suggestion, index) => {
            report += `${index + 1}. ${suggestion}\n`;
          });
        }

        // Create downloadable file
        this.downloadTextFile(report, `seo-analysis-report-${new Date().toISOString().split('T')[0]}.txt`);
        this.showNotification('SEO report downloaded successfully!', 'success');

      } catch (error) {
        console.error('Error exporting report:', error);
        this.showNotification('Failed to export report: ' + error.message, 'error');
      }
    }

    renderSuggestionsSection(suggestions, title) {
      return `
    <div class="ai-suggestions-section">
      <div class="ai-section-title">
        <span>💡</span>
        <span>${this.escapeHtml(title)}</span>
      </div>
      <ul class="ai-suggestions-list">
        ${suggestions.map((suggestion, index) =>
        `<li><strong>${index + 1}.</strong> ${this.escapeHtml(suggestion)}</li>`
      ).join('')}
      </ul>
    </div>
  `;
    }

    downloadTextFile(content, filename) {
      const blob = new Blob([content], { type: 'text/plain' });
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      window.URL.revokeObjectURL(url);
    }

    reanalyzeContent() {
      if (this.isProcessing) {
        return;
      }

      try {
        // Check if there's existing content to analyze
        if (!this.checkExistingContent()) {
          this.showNotification('Please add some content first before analyzing', 'warning');
          return;
        }

        this.selectedTaskType = TaskType.SEO_ANALYSIS;
        const input = 'Reanalyze this content for SEO performance and provide detailed recommendations';

        // Clear input and set new value
        $('#ai-chatbot-input').val('');
        $('#ai-chatbot-input').val(input);

        // Add user message immediately to prevent confusion
        this.addMessage('user', input);
        this.setProcessingState(true);

        // Send the request directly without using sendMessage to avoid double processing
        setTimeout(() => {
          this.processReanalysisRequest(input);
        }, 100);

      } catch (error) {
        this.showNotification('Failed to reanalyze content: ' + error.message, 'error');
      }
    }

    optimizeContent() {
      if (this.isProcessing) {
        return;
      }

      try {
        // Check if there's existing content to analyze
        if (!this.checkExistingContent()) {
          this.showNotification('Please add some content first before analyzing', 'warning');
          return;
        }

        this.selectedTaskType = TaskType.CONTENT_OPTIMIZATION;
        const input = 'Reanalyze this content and optimize content';

        // Clear input and set new value
        $('#ai-chatbot-input').val('');
        $('#ai-chatbot-input').val(input);

        // Add user message immediately to prevent confusion
        this.addMessage('user', input);
        this.setProcessingState(true);

        // Send the request directly without using sendMessage to avoid double processing
        setTimeout(() => {
          this.processReanalysisRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error in reanalyzeContent:', error);
        this.showNotification('Failed to reanalyze content: ' + error.message, 'error');
      }
    }

    async processReanalysisRequest(input) {
      try {
        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
        this.selectedTaskType = null;
        $('#ai-chatbot-input').val('').attr('placeholder', 'Type your message or \'/\' for commands...');
      }
    }

    getLastAnalysisResponse() {
      if (this.chatHistory && this.chatHistory.length > 0) {
        for (let i = this.chatHistory.length - 1; i >= 0; i--) {
          const message = this.chatHistory[i];
          if (message.type === 'assistant' && message.data && message.data.analysis) {
            return message.data.analysis;
          }
        }
      }
      return null;
    }

    parseButtonData(data) {
      if (typeof data === 'string') {
        try {
          return JSON.parse(data);
        } catch (e) {
          return data;
        }
      }
      return data;
    }

    copySingleField(fieldKey, fieldValue) {
      try {
        let textToCopy = fieldValue;
        if (Array.isArray(fieldValue)) {
          textToCopy = fieldValue.join(', ');
        }
        this.copyToClipboard(textToCopy, `${fieldKey.replace('_', ' ')} copied to clipboard!`);
      } catch (error) {
        this.showNotification('Failed to copy field: ' + error.message, 'error');
      }
    }

    copyAllContent() {
      try {
        const lastResponse = this.getLastResponseContent();
        if (!lastResponse || Object.keys(lastResponse).length === 0) {
          this.showNotification('No content to copy', 'warning');
          return;
        }

        let allContent = '';
        const fieldLabels = {
          'title': 'TITLE',
          'kicker': 'SUBTITLE/KICKER',
          'html_content': 'CONTENT',
          'meta_title': 'META TITLE',
          'meta_description': 'META DESCRIPTION',
          'url_alias': 'URL ALIAS',
          'keywords': 'KEYWORDS'
        };

        Object.entries(fieldLabels).forEach(([fieldKey, label]) => {
          const fieldValue = lastResponse[fieldKey];
          if (fieldValue && fieldValue !== '') {
            allContent += `\n=== ${label} ===\n`;
            if (Array.isArray(fieldValue)) {
              allContent += fieldValue.join(', ');
            } else {
              allContent += fieldValue;
            }
            allContent += '\n';
          }
        });

        if (allContent.trim()) {
          this.copyToClipboard(allContent.trim(), 'All content copied to clipboard!');
        } else {
          this.showNotification('No content to copy', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to copy content: ' + error.message, 'error');
      }
    }

    insertTableContent($button) {
      try {
        let tableHtml = $button.data('table-html');

        if (!tableHtml) {
          const tableData = this.parseButtonData($button.data('table-content'));
          if (tableData && tableData.html_content) {
            tableHtml = tableData.html_content;
          }
        }
        if (tableHtml) {
          this.insertTableIntoBodyField(tableHtml);
        } else {
          this.showNotification('No table content to insert', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to insert table content: ' + error.message, 'error');
      }
    }

    isTableHtml(content) {
      return typeof content === 'string' &&
        (content.includes('<table') || content.includes('&lt;table'));
    }

    downloadTableData($button) {
      try {
        const tableData = this.parseButtonData($button.data('table-content'));
        const tableHtml = $button.data('table-html');

        if (!tableData && !tableHtml) {
          this.showNotification('No table data available', 'error');
          return;
        }

        let exportData = {
          html_table: tableHtml || this.decodeHtmlEntities(tableData.html_content || ''),
          description: tableData.plain_text || '',
          generated_at: new Date().toISOString()
        };

        const jsonData = JSON.stringify(exportData, null, 2);
        const blob = new Blob([jsonData], { type: 'application/json' });
        const url = window.URL.createObjectURL(blob);

        const a = document.createElement('a');
        a.href = url;
        a.download = `table-data-${Date.now()}.json`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);

        this.showNotification('Table data downloaded!', 'success');
      } catch (error) {
        console.error('Error downloading table data:', error);
        this.showNotification('Failed to download data: ' + error.message, 'error');
      }
    }

    exportTableAsCSV($button) {
      try {
        const tableData = this.parseButtonData($button.data('table-content'));

        if (!tableData || !tableData.html_content) {
          this.showNotification('No table data available', 'error');
          return;
        }

        // For now, export the content as text - in the future, this could parse actual table data
        let csvContent = 'Field,Content\n';
        csvContent += `"Table Content","${tableData.html_content.replace(/"/g, '""')}"\n`;
        if (tableData.plain_text) {
          csvContent += `"Description","${tableData.plain_text.replace(/"/g, '""')}"\n`;
        }

        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);

        const a = document.createElement('a');
        a.href = url;
        a.download = `table-export-${Date.now()}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);

        this.showNotification('Table data exported as CSV!', 'success');
      } catch (error) {
        console.error('Error exporting CSV:', error);
        this.showNotification('Failed to export CSV: ' + error.message, 'error');
      }
    }

    insertAllMetaFields(metaContent) {
      try {
        let insertedCount = 0;
        const metaFieldMappings = {
          'meta_title': metaContent.meta_title,
          'meta_description': metaContent.meta_description,
          'url_alias': metaContent.url_alias,
          'keywords': metaContent.keywords
        };

        Object.entries(metaFieldMappings).forEach(([fieldKey, fieldValue]) => {
          if (fieldValue && fieldValue !== '') {
            try {
              this.insertSingleField(fieldKey, fieldValue);
              insertedCount++;
            } catch (error) {
              console.error(`Failed to insert ${fieldKey}:`, error);
            }
          }
        });

        if (insertedCount > 0) {
          this.showNotification(`Successfully inserted ${insertedCount} meta field(s)!`, 'success');
        } else {
          this.showNotification('No meta fields to insert', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to insert meta content: ' + error.message, 'error');
      }
    }

    copyAllMetaContent(metaContent) {
      try {
        let allMetaContent = 'SEO META DATA:\n\n';
        const metaLabels = {
          'meta_title': 'META TITLE',
          'meta_description': 'META DESCRIPTION',
          'url_alias': 'URL ALIAS',
          'keywords': 'KEYWORDS'
        };

        Object.entries(metaLabels).forEach(([fieldKey, label]) => {
          const fieldValue = metaContent[fieldKey];
          if (fieldValue && fieldValue !== '') {
            allMetaContent += `=== ${label} ===\n`;
            if (Array.isArray(fieldValue)) {
              allMetaContent += fieldValue.join(', ');
            } else {
              allMetaContent += fieldValue;
            }
            allMetaContent += '\n\n';
          }
        });

        if (allMetaContent.trim() !== 'SEO META DATA:') {
          this.copyToClipboard(allMetaContent.trim(), 'All meta content copied to clipboard!');
        } else {
          this.showNotification('No meta content to copy', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to copy meta content: ' + error.message, 'error');
      }
    }

    createLongerSummary() {
      if (this.isProcessing) {
        return;
      }

      try {
        // Check if there's existing content to summarize
        if (!this.checkExistingContent()) {
          this.showNotification('Please add some content first before creating a longer summary', 'warning');
          return;
        }

        this.selectedTaskType = TaskType.CONTENT_SUMMARY;
        const input = 'Create a more detailed and comprehensive summary of this content with expanded explanations';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processSummaryRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error creating longer summary:', error);
        this.showNotification('Failed to create longer summary: ' + error.message, 'error');
      }
    }

    insertKeywordsOnly(keywordContent) {
      try {
        if (Array.isArray(keywordContent.keywords) && keywordContent.keywords.length > 0) {
          this.insertSingleField('keywords', keywordContent.keywords);
          this.showNotification(`Successfully inserted ${keywordContent.keywords.length} keywords!`, 'success');
        } else {
          this.showNotification('No keywords to insert', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to insert keywords: ' + error.message, 'error');
      }
    }

    copyKeywordsOnly(keywordContent) {
      try {
        if (Array.isArray(keywordContent.keywords) && keywordContent.keywords.length > 0) {
          const keywordText = keywordContent.keywords.join(', ');
          this.copyToClipboard(keywordText, `${keywordContent.keywords.length} keywords copied to clipboard!`);
        } else {
          this.showNotification('No keywords to copy', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to copy keywords: ' + error.message, 'error');
      }
    }

    analyzeCompetition() {
      if (this.isProcessing) {
        return;
      }

      try {
        this.selectedTaskType = TaskType.KEYWORD_EXTRACTION;
        const input = 'Analyze competitor keywords and suggest competitive keyword strategies for better ranking';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processKeywordRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error analyzing competition:', error);
        this.showNotification('Failed to analyze competition: ' + error.message, 'error');
      }
    }

    insertTitleOnly(titleContent) {
      try {
        if (titleContent.title && titleContent.title.trim().length > 0) {
          this.insertSingleField('title', titleContent.title);
          this.showNotification('Title inserted successfully!', 'success');
        } else {
          this.showNotification('No title to insert', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to insert title: ' + error.message, 'error');
      }
    }

    copyTitleOnly(titleContent) {
      try {
        if (titleContent.title && titleContent.title.trim().length > 0) {
          this.copyToClipboard(titleContent.title, 'Title copied to clipboard!');
        } else {
          this.showNotification('No title to copy', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to copy title: ' + error.message, 'error');
      }
    }

    optimizeDescriptionLength() {
      if (this.isProcessing) {
        return;
      }

      try {
        // Check if there's existing content for context
        const existingContent = this.getCurrentDescriptionContent();

        this.selectedTaskType = TaskType.DESCRIPTION_GENERATION;
        const input = existingContent
          ? `Optimize the length of these descriptions for better SEO and readability: Meta description and full description`
          : 'Generate descriptions with optimal length for SEO (meta description 120-160 chars) and readability';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processDescriptionRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error optimizing description length:', error);
        this.showNotification('Failed to optimize description length: ' + error.message, 'error');
      }
    }

    getCurrentDescriptionContent() {
      // Try to get current meta description from form
      const $metaDescField = $('textarea[name*="meta_description"], input[name*="meta_description"]');
      if ($metaDescField.length > 0) {
        const currentMeta = $metaDescField.val();
        if (currentMeta && currentMeta.trim().length > 0) {
          return { meta_description: currentMeta.trim() };
        }
      }

      // Fallback: get from last generated description in chat history
      const lastDescriptionResponse = this.getLastDescriptionResponse();
      return lastDescriptionResponse || {};
    }

    getLastDescriptionResponse() {
      if (this.chatHistory && this.chatHistory.length > 0) {
        for (let i = this.chatHistory.length - 1; i >= 0; i--) {
          const message = this.chatHistory[i];
          if (message.type === 'assistant' && message.data && message.data.type === 'description_generation' && message.data.content) {
            return message.data.content;
          }
        }
      }
      return null;
    }

    async processDescriptionRequest(input) {
      try {
        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
        this.selectedTaskType = null;
        $('#ai-chatbot-input').val('').attr('placeholder', 'Type your message or \'/\' for commands...');
      }
    }

    seoOptimizeTitle() {
      if (this.isProcessing) {
        return;
      }

      try {
        // Get current title for optimization context
        const currentTitle = this.getCurrentTitleContent();

        this.selectedTaskType = TaskType.TITLE_GENERATION;
        const input = currentTitle
          ? `Optimize this title for better SEO while maintaining readability: "${currentTitle}"`
          : 'Generate an SEO-optimized title for this content with focus on search visibility and click-through rates';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processTitleRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error optimizing title for SEO:', error);
        this.showNotification('Failed to optimize title: ' + error.message, 'error');
      }
    }

    getCurrentTitleContent() {
      // Try to get current title from form
      const $titleField = $('input[name="title[0][value]"]');
      if ($titleField.length > 0) {
        const currentTitle = $titleField.val();
        if (currentTitle && currentTitle.trim().length > 0) {
          return currentTitle.trim();
        }
      }

      // Fallback: get from last generated title in chat history
      const lastTitleResponse = this.getLastTitleResponse();
      return lastTitleResponse ? lastTitleResponse.title : '';
    }

    getLastTitleResponse() {
      if (this.chatHistory && this.chatHistory.length > 0) {
        for (let i = this.chatHistory.length - 1; i >= 0; i--) {
          const message = this.chatHistory[i];
          if (message.type === 'assistant' && message.data && message.data.type === 'title_generation' && message.data.content) {
            return message.data.content;
          }
        }
      }
      return null;
    }

    async processTitleRequest(input) {
      try {
        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
        this.selectedTaskType = null;
        $('#ai-chatbot-input').val('').attr('placeholder', 'Type your message or \'/\' for commands...');
      }
    }

    expandKeywords() {
      if (this.isProcessing) {
        return;
      }

      try {
        this.selectedTaskType = TaskType.KEYWORD_EXTRACTION;
        const input = 'Expand the current keyword list with more related keywords, synonyms, and long-tail variations for comprehensive SEO coverage';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processKeywordRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error expanding keywords:', error);
        this.showNotification('Failed to expand keywords: ' + error.message, 'error');
      }
    }

    async processKeywordRequest(input) {
      try {
        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
        this.selectedTaskType = null;
        $('#ai-chatbot-input').val('').attr('placeholder', 'Type your message or \'/\' for commands...');
      }
    }

    improveReadability() {
      if (this.isProcessing) {
        return;
      }

      try {
        // Check if there's existing content to improve
        if (!this.checkExistingContent()) {
          this.showNotification('Please add some content first before improving readability', 'warning');
          return;
        }

        this.selectedTaskType = TaskType.CONTENT_OPTIMIZATION;
        const input = 'Improve the readability of this content by simplifying sentences, improving flow, and making it easier to understand';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processReadabilityRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error improving readability:', error);
        this.showNotification('Failed to improve readability: ' + error.message, 'error');
      }
    }

    async processReadabilityRequest(input) {
      try {
        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
        this.selectedTaskType = null;
        $('#ai-chatbot-input').val('').attr('placeholder', 'Type your message or \'/\' for commands...');
      }
    }

    extractHighlights() {
      if (this.isProcessing) {
        return;
      }

      try {
        // Check if there's existing content
        if (!this.checkExistingContent()) {
          this.showNotification('Please add some content first before extracting highlights', 'warning');
          return;
        }

        this.selectedTaskType = TaskType.CONTENT_SUMMARY;
        const input = 'Extract key highlights and bullet points from this content for easy scanning';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processSummaryRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error extracting highlights:', error);
        this.showNotification('Failed to extract highlights: ' + error.message, 'error');
      }
    }

    async processSummaryRequest(input) {
      try {
        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
        this.selectedTaskType = null;
        $('#ai-chatbot-input').val('').attr('placeholder', 'Type your message or \'/\' for commands...');
      }
    }

    optimizeMetaMore() {
      if (this.isProcessing) {
        return;
      }

      try {
        this.selectedTaskType = TaskType.META_GENERATION;
        const input = 'Generate more optimized and compelling meta data variations for better SEO performance';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processMetaOptimizationRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error optimizing meta more:', error);
        this.showNotification('Failed to optimize meta: ' + error.message, 'error');
      }
    }

    generateMetaVariations() {
      if (this.isProcessing) {
        return;
      }

      try {
        this.selectedTaskType = TaskType.META_GENERATION;
        const input = 'Generate multiple variations of meta titles, descriptions, and keywords for A/B testing';

        $('#ai-chatbot-input').val(input);
        this.addMessage('user', input);
        this.setProcessingState(true);

        setTimeout(() => {
          this.processMetaOptimizationRequest(input);
        }, 100);

      } catch (error) {
        console.error('Error generating meta variations:', error);
        this.showNotification('Failed to generate variations: ' + error.message, 'error');
      }
    }

    async processMetaOptimizationRequest(input) {
      try {
        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
        this.selectedTaskType = null;
        $('#ai-chatbot-input').val('').attr('placeholder', 'Type your message or \'/\' for commands...');
      }
    }

    showTablePreview(tableContent) {
      $('#ai-table-preview-modal').remove();

      const $button = $(event.target);
      let tableHtml = $button.data('table-html');

      if (!tableHtml && tableContent) {
        tableHtml = this.decodeHtmlEntities(tableContent);
      }

      // Clean the HTML for proper display
      const cleanTableHtml = tableHtml
        .replace(/\\n/g, '')  // Remove literal \n
        .replace(/\\"/g, '"') // Fix escaped quotes
        .replace(/\s+/g, ' ') // Normalize whitespace
        .trim();

      const modalHtml = `
    <div class="ai-modal-overlay" id="ai-table-preview-modal">
      <div class="ai-modal ai-modal--table-preview">
        <div class="ai-modal-header">
          <h3>📊 Table Preview</h3>
          <button class="ai-modal-close" id="close-table-preview-modal">×</button>
        </div>
        <div class="ai-modal-body">
          <div class="ai-table-preview-content">
            ${cleanTableHtml}
          </div>
        </div>
        <div class="ai-modal-footer">
          <button id="copy-table-html-btn" class="ai-btn ai-btn--secondary">Copy HTML</button>
          <button id="export-table-csv-btn" class="ai-btn ai-btn--secondary">Export CSV</button>
          <button id="close-table-modal-btn" class="ai-btn ai-btn--primary">Close</button>
        </div>
      </div>
    </div>
  `;

      $('body').append(modalHtml);

      // Bind events
      $('#close-table-preview-modal, #close-table-modal-btn').on('click', function () {
        $('#ai-table-preview-modal').remove();
      });

      $('#copy-table-html-btn').on('click', () => {
        this.copyToClipboard(cleanTableHtml, 'Table HTML copied to clipboard!');
      });

      $('#export-table-csv-btn').on('click', () => {
        this.exportHtmlTableAsCSV(cleanTableHtml);
      });

      $('#ai-table-preview-modal').on('click', function (e) {
        if (e.target === this) {
          $(this).remove();
        }
      });
    }

    cleanTableHtml(rawHtml) {
      let cleaned = rawHtml;

      // If it's JSON-encoded, parse it first
      if (typeof rawHtml === 'string' && (rawHtml.startsWith('"') || rawHtml.includes('\\"'))) {
        try {
          cleaned = JSON.parse(rawHtml);
        } catch (e) {
          // If JSON parsing fails, manually clean
          cleaned = rawHtml.replace(/^"/, '').replace(/"$/, '').replace(/\\"/g, '"');
        }
      }

      // Clean up HTML entities and formatting
      const tempDiv = document.createElement('div');
      tempDiv.innerHTML = cleaned;
      cleaned = tempDiv.innerHTML;

      // Remove extra whitespace and newlines
      cleaned = cleaned.replace(/\s+/g, ' ').trim();

      return cleaned;
    }

    insertTableIntoBodyField(tableHtml) {
      if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
        try {
          let inserted = false;
          Drupal.CKEditor5Instances.forEach((editor, key) => {
            if (editor.sourceElement.id === 'edit-body-0-value') {
              const editorInstance = Drupal.CKEditor5Instances.get(key);
              if (editorInstance) {
                // Clean the HTML properly
                const cleanHtml = this.cleanTableHtml(tableHtml);

                // Insert HTML content using CKEditor's data API (same as images)
                editorInstance.model.change(writer => {
                  // Convert HTML to model fragment
                  const viewFragment = editorInstance.data.processor.toView(cleanHtml);
                  const modelFragment = editorInstance.data.toModel(viewFragment);

                  // Insert at current position
                  const selection = editorInstance.model.document.selection;
                  const position = selection.getFirstPosition();

                  writer.insert(modelFragment, position);
                });

                inserted = true;
                this.showNotification('Table inserted successfully!', 'success');
              }
            }
          });

          if (inserted) return true;
        } catch (error) {
          console.warn('CKEditor 5 insertion failed, using fallback:', error);
        }
      }

      // Fallback method
      return this.insertHtmlContentFallback(tableHtml);
    }

    insertHtmlContent(editorInstance, htmlContent, writer) {
      // For complex HTML, insert as HTML block if available
      try {
        // Try to insert as source HTML if the plugin is available
        if (editorInstance.commands.get('sourceEditing')) {
          const htmlElement = writer.createElement('htmlEmbed', { value: htmlContent });
          const selection = editorInstance.model.document.selection;
          const position = selection.getFirstPosition();
          writer.insert(htmlElement, position);
        } else {
          // Convert HTML to plain text and insert
          const textContent = this.stripHtmlTags(htmlContent);
          const paragraph = writer.createElement('paragraph');
          const text = writer.createText(textContent);
          writer.append(text, paragraph);

          const selection = editorInstance.model.document.selection;
          const position = selection.getFirstPosition();
          writer.insert(paragraph, position);
        }
      } catch (error) {
        console.error('HTML content insertion failed:', error);
      }
    }

    insertHtmlContentFallback(tableHtml) {
      // Method 2: Fallback to textarea insertion (append, don't replace)
      const $bodyField = $('#edit-body-0-value');
      if ($bodyField.length > 0) {
        const cleanTableHtml = this.decodeHtmlEntities(tableHtml)
          .replace(/\\n/g, '')
          .replace(/\\"/g, '"')
          .trim();

        // Get current content and append (don't replace)
        let currentContent = $bodyField.val() || '';
        const newContent = currentContent + '\n\n' + cleanTableHtml;
        $bodyField.val(newContent);

        // Trigger change event
        $bodyField.trigger('input').trigger('change');

        // Try to sync with CKEditor if possible
        if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
          try {
            Drupal.CKEditor5Instances.forEach((editor, key) => {
              if (editor.sourceElement.id === 'edit-body-0-value') {
                const editorInstance = Drupal.CKEditor5Instances.get(key);
                if (editorInstance && editorInstance.setData) {
                  editorInstance.setData(newContent);
                }
              }
            });
          } catch (e) {
            console.warn('CKEditor sync failed:', e);
          }
        }

        this.showNotification('Table HTML inserted into content field!', 'success');
        this.highlightBodyField();
        return true;
      }

      // Final fallback - copy to clipboard
      const cleanTableHtml = this.decodeHtmlEntities(tableHtml);
      this.copyToClipboard(cleanTableHtml, 'Table HTML copied to clipboard - paste manually into editor');
      return false;
    }

    exportHtmlTableAsCSV(tableHtml) {
      try {
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = tableHtml;

        const table = tempDiv.querySelector('table');
        if (!table) {
          this.showNotification('No table found to export', 'error');
          return;
        }

        let csvContent = '';

        // Get headers
        const headers = table.querySelectorAll('thead th, tr:first-child th, tr:first-child td');
        if (headers.length > 0) {
          csvContent += Array.from(headers).map(th => `"${th.textContent.trim()}"`).join(',') + '\n';
        }

        // Get data rows
        const rows = table.querySelectorAll('tbody tr, tr:not(:first-child)');
        rows.forEach(row => {
          const cells = row.querySelectorAll('td, th');
          csvContent += Array.from(cells).map(td => `"${td.textContent.trim()}"`).join(',') + '\n';
        });

        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);

        const a = document.createElement('a');
        a.href = url;
        a.download = `table-export-${Date.now()}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);

        this.showNotification('Table exported as CSV!', 'success');
      } catch (error) {
        console.error('Error exporting CSV:', error);
        this.showNotification('Failed to export CSV: ' + error.message, 'error');
      }
    }

    async insertGraphAsImage($button) {
      try {
        const chartId = $button.data('chart-id');
        const canvas = document.getElementById(chartId);

        if (!canvas || !canvas.chartInstance) {
          this.showNotification('Chart not found or not ready', 'error');
          return;
        }

        this.setProcessingState(true);
        this.showNotification('Processing chart...', 'info');

        // Convert chart to image
        const imageDataURL = canvas.toDataURL('image/png', 1.0);

        const imageUrl = await this.uploadChartImage(imageDataURL, chartId);

        if (imageUrl) {
          this.insertImageIntoBodyField(imageUrl, 'chart');
        } else {
          this.copyChartImageToClipboard(imageDataURL);
          this.showNotification('Upload failed. Image copied to clipboard - paste manually into editor', 'warning');
        }

      } catch (error) {
        console.error('Error inserting chart as image:', error);
        this.showNotification('Failed to insert chart: ' + error.message, 'error');
      } finally {
        this.setProcessingState(false);
      }
    }

    parseTableFromHtml(tableHtml) {
      try {
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = tableHtml;

        const table = tempDiv.querySelector('table');
        if (!table) return { rows: [], hasHeader: false };

        const rows = [];
        let hasHeader = false;

        // Check for thead/th elements
        const headerRows = table.querySelectorAll('thead tr, tr:has(th)');
        if (headerRows.length > 0) {
          hasHeader = true;
          headerRows.forEach(row => {
            const cells = Array.from(row.querySelectorAll('th, td'))
              .map(cell => cell.textContent.trim());
            if (cells.length > 0) rows.push(cells);
          });
        }

        // Get body rows
        const bodyRows = table.querySelectorAll('tbody tr, tr:not(:has(th))');
        bodyRows.forEach(row => {
          const cells = Array.from(row.querySelectorAll('td, th'))
            .map(cell => cell.textContent.trim());
          if (cells.length > 0) rows.push(cells);
        });

        return { rows, hasHeader };
      } catch (error) {
        console.error('Error parsing table HTML:', error);
        return { rows: [], hasHeader: false };
      }
    }

    insertImageIntoBodyField(imageUrl, type = 'image') {
      if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
        try {
          let inserted = false;
          Drupal.CKEditor5Instances.forEach((editor, key) => {
            if (editor.sourceElement.id === 'edit-body-0-value') {
              const editorInstance = Drupal.CKEditor5Instances.get(key);
              if (editorInstance) {
                // Use proper CKEditor 5 image insertion
                editorInstance.model.change(writer => {
                  // Create image element with proper attributes
                  const imageElement = writer.createElement('imageBlock', {
                    src: imageUrl,
                    alt: type == 'image' ? "inline image" : 'Generated Chart',
                    class: 'image' ? "inline-image" : 'ai-generated-chart'
                  });

                  // Insert at current cursor position
                  const selection = editorInstance.model.document.selection;
                  const position = selection.getFirstPosition();

                  writer.insert(imageElement, position);
                });

                inserted = true;
                this.showNotification('Chart image inserted successfully!', 'success');
              }
            }
          });

          if (inserted) return true;
        } catch (error) {
          console.warn('CKEditor 5 insertion failed, trying alternative method:', error);
        }
      }
    }

    downloadGraphData($button) {
      try {
        const chartId = $button.data('chart-id');
        const canvas = document.getElementById(chartId);

        if (!canvas) {
          this.showNotification('Chart not found', 'error');
          return;
        }

        // Convert chart to image data
        const imageDataURL = canvas.toDataURL('image/png', 1.0);

        // Create download link
        const a = document.createElement('a');
        a.href = imageDataURL;
        a.download = `chart-${Date.now()}.png`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);

        this.showNotification('Chart image downloaded!', 'success');
      } catch (error) {
        console.error('Error downloading chart image:', error);
        this.showNotification('Failed to download chart: ' + error.message, 'error');
      }
    }

    exportGraphAsCSV($button) {
      try {
        const graphData = this.parseButtonData($button.data('graph-data'));

        if (!graphData || !graphData.chart_data) {
          this.showNotification('No graph data available', 'error');
          return;
        }

        const chartData = graphData.chart_data;
        let csvContent = '';

        // Create CSV headers
        if (chartData.labels && chartData.datasets) {
          csvContent += 'Label,' + chartData.datasets.map(d => d.label).join(',') + '\n';

          // Add data rows
          chartData.labels.forEach((label, index) => {
            csvContent += `"${label}"`;
            chartData.datasets.forEach(dataset => {
              csvContent += ',' + (dataset.data[index] || '');
            });
            csvContent += '\n';
          });
        }

        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);

        const a = document.createElement('a');
        a.href = url;
        a.download = `chart-data-${Date.now()}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);

        this.showNotification('Chart data exported as CSV!', 'success');
      } catch (error) {
        console.error('Error exporting CSV:', error);
        this.showNotification('Failed to export CSV: ' + error.message, 'error');
      }
    }

    downloadChartAsImage($button) {
      try {
        const chartId = $button.data('chart-id');
        const canvas = document.getElementById(chartId);

        if (!canvas) {
          this.showNotification('Chart not found', 'error');
          return;
        }

        const imageDataURL = canvas.toDataURL('image/png', 1.0);
        const a = document.createElement('a');
        a.href = imageDataURL;
        a.download = `chart-${Date.now()}.png`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);

        this.showNotification('Chart image downloaded!', 'success');
      } catch (error) {
        console.error('Error downloading chart image:', error);
        this.showNotification('Failed to download chart: ' + error.message, 'error');
      }
    }

    async uploadChartImage(imageDataURL, chartId) {
      try {
        const canvas = document.getElementById(chartId);

        // Get blob directly from canvas instead of data URL
        return new Promise((resolve) => {
          canvas.toBlob(async (blob) => {
            try {
              // Create form data
              const formData = new FormData();
              formData.append('image', blob, `chart-${chartId}-${Date.now()}.png`);
              formData.append('type', 'chart');

              // Get CSRF token
              const csrfToken = drupalSettings.user?.csrf_token || '';
              const uploadResponse = await fetch('/ai-assistant/upload-image', {
                method: 'POST',
                headers: {
                  'X-CSRF-Token': csrfToken
                },
                body: formData
              });

              if (uploadResponse.ok) {
                const result = await uploadResponse.json();

                if (result.success && result.data && result.data.url) {
                  resolve(result.data.absolute_url || result.data.url);
                } else {
                  resolve(null);
                }
              } else {
                resolve(null);
              }

            } catch (error) {
              resolve(null);
            }
          }, 'image/png', 1.0); // Explicitly specify PNG type and quality
        });

      } catch (error) {
        console.error('Error uploading chart image:', error);
        return null;
      }
    }

    copyChartImageToClipboard(imageDataURL) {
      try {
        // For modern browsers
        if (navigator.clipboard && window.ClipboardItem) {
          fetch(imageDataURL)
            .then(response => response.blob())
            .then(blob => {
              const item = new ClipboardItem({ 'image/png': blob });
              navigator.clipboard.write([item]);
              this.showNotification('Chart image copied to clipboard!', 'success');
            })
            .catch(() => {
              this.showNotification('Chart ready - please right-click to copy image', 'info');
            });
        } else {
          this.showNotification('Chart ready - please right-click to copy image', 'info');
        }
      } catch (error) {
        console.error('Error copying to clipboard:', error);
        this.showNotification('Chart ready - please right-click to copy image', 'info');
      }
    }

    insertAllContent() {
      try {
        const lastResponse = this.getLastResponseContent();
        if (!lastResponse || Object.keys(lastResponse).length === 0) {
          this.showNotification('No content to insert', 'warning');
          return;
        }

        let insertedCount = 0;
        const fieldMappings = {
          'title': lastResponse.title,
          'kicker': lastResponse.kicker,
          'html_content': lastResponse.html_content,
          'meta_title': lastResponse.meta_title,
          'meta_description': lastResponse.meta_description,
          'url_alias': lastResponse.url_alias,
          'keywords': lastResponse.keywords
        };

        Object.entries(fieldMappings).forEach(([fieldKey, fieldValue]) => {
          if (fieldValue && fieldValue !== '') {
            try {
              this.insertSingleField(fieldKey, fieldValue);
              insertedCount++;
            } catch (error) {
              console.error(`Failed to insert ${fieldKey}:`, error);
            }
          }
        });

        if (insertedCount > 0) {
          this.showNotification(`Successfully inserted ${insertedCount} fields!`, 'success');
        } else {
          this.showNotification('No fields were inserted', 'warning');
        }
      } catch (error) {
        this.showNotification('Failed to insert content: ' + error.message, 'error');
      }
    }

    compareOriginalContent() {
      try {
        // Get current/original content
        const originalContent = this.getCurrentContent();

        // Get last generated content
        const lastResponse = this.getLastResponseContent();

        if (!originalContent.content && !originalContent.title) {
          this.showNotification('No original content found to compare', 'warning');
          return;
        }

        if (!lastResponse || (!lastResponse.html_content && !lastResponse.title)) {
          this.showNotification('No generated content found to compare', 'warning');
          return;
        }

        this.showComparisonModal(originalContent, lastResponse);

      } catch (error) {
        console.error('Error comparing content:', error);
        this.showNotification('Failed to compare content: ' + error.message, 'error');
      }
    }

    getCurrentContent() {
      let currentContent = {
        title: '',
        content: ''
      };

      // Get title
      const $titleField = $('input[name="title[0][value]"]');
      if ($titleField.length > 0) {
        currentContent.title = $titleField.val() || '';
      }

      // Get body content - try CKEditor 5 first
      if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
        try {
          Drupal.CKEditor5Instances.forEach((editor, key) => {
            if (editor.sourceElement.id === 'edit-body-0-value') {
              const editorInstance = Drupal.CKEditor5Instances.get(key);
              if (editorInstance) {
                currentContent.content = editorInstance.getData() || '';
              }
            }
          });
        } catch (e) {
          console.warn('Failed to get CKEditor 5 content:', e);
        }
      }

      // Fallback to textarea if CKEditor 5 content not available
      if (!currentContent.content) {
        const $bodyField = $('textarea[name="body[0][value]"]');
        if ($bodyField.length > 0) {
          currentContent.content = $bodyField.val() || '';
        }
      }

      return currentContent;
    }

    showComparisonModal(originalContent, generatedContent) {
      // Remove existing modal if any
      $('#ai-comparison-modal').remove();

      const modalHtml = `
    <div class="ai-modal-overlay" id="ai-comparison-modal">
      <div class="ai-modal ai-modal--comparison">
        <div class="ai-modal-header">
          <h3>Content Comparison</h3>
          <button class="ai-modal-close" id="close-comparison-modal">×</button>
        </div>
        <div class="ai-modal-body">
          <div class="ai-comparison-container">
            <div class="ai-comparison-panel ai-comparison-panel--original">
              <div class="ai-comparison-header">
                <h4>📄 Original Content</h4>
                <span class="ai-content-stats">${this.getContentStats(originalContent)}</span>
              </div>
              <div class="ai-comparison-content">
                ${this.renderComparisonContent(originalContent)}
              </div>
            </div>

            <div class="ai-comparison-divider">
              <div class="ai-comparison-vs">VS</div>
            </div>

            <div class="ai-comparison-panel ai-comparison-panel--generated">
              <div class="ai-comparison-header">
                <h4>✨ Generated Content</h4>
                <span class="ai-content-stats">${this.getContentStats(generatedContent)}</span>
              </div>
              <div class="ai-comparison-content">
                ${this.renderComparisonContent(generatedContent)}
              </div>
            </div>
          </div>
        </div>
        <div class="ai-modal-footer">
          <button id="use-original-btn" class="ai-btn ai-btn--secondary">Keep Original</button>
          <button id="use-generated-btn" class="ai-btn ai-btn--primary">Use Generated</button>
          <button id="close-comparison-btn" class="ai-btn ai-btn--outline">Close</button>
        </div>
      </div>
    </div>
  `;

      $('body').append(modalHtml);

      // Bind close events
      $('#close-comparison-modal, #close-comparison-btn').on('click', () => {
        $('#ai-comparison-modal').remove();
      });

      // Bind action events
      $('#use-original-btn').on('click', () => {
        this.restoreOriginalContent(originalContent);
        $('#ai-comparison-modal').remove();
      });

      $('#use-generated-btn').on('click', () => {
        this.insertAllContent();
        $('#ai-comparison-modal').remove();
      });

      // Close on overlay click
      $('#ai-comparison-modal').on('click', function (e) {
        if (e.target === this) {
          $(this).remove();
        }
      });
    }

    renderComparisonContent(contentData) {
      let html = '';

      // Handle different content structures
      const title = contentData.title || contentData.title || '';
      const content = contentData.content || contentData.html_content || '';

      if (title) {
        html += `
      <div class="ai-comparison-field">
        <div class="ai-field-label">Title:</div>
        <div class="ai-field-content ai-field-content--title">${this.escapeHtml(title)}</div>
      </div>
    `;
      }

      if (content) {
        // Clean HTML content for better display
        const cleanContent = content.replace(/<[^>]*>/g, '').replace(/\s+/g, ' ').trim();
        const shouldTruncate = cleanContent.length > 800;
        const displayContent = shouldTruncate ? cleanContent.substring(0, 800) + '...' : cleanContent;

        html += `
      <div class="ai-comparison-field">
        <div class="ai-field-label">Content:</div>
        <div class="ai-field-content ai-field-content--body">
          <div class="ai-content-preview">${this.escapeHtml(displayContent)}</div>
          ${shouldTruncate ? `<button class="ai-expand-btn" onclick="this.previousElementSibling.innerHTML = '${this.escapeHtml(cleanContent).replace(/'/g, "\\'")}'; this.style.display = 'none';">Show Full Content</button>` : ''}
        </div>
      </div>
    `;
      }

      if (!title && !content) {
        html = '<div class="ai-no-content">No content available</div>';
      }

      return html;
    }

    getContentStats(contentData) {
      const title = contentData.title || contentData.title || '';
      const content = contentData.content || contentData.html_content || '';

      // Remove HTML tags for accurate word count
      const textContent = content.replace(/<[^>]*>/g, ' ').replace(/\s+/g, ' ').trim();
      const wordCount = textContent ? textContent.split(' ').length : 0;
      const charCount = textContent.length;

      return `${wordCount} words, ${charCount} characters`;
    }

    restoreOriginalContent(originalContent) {
      try {
        let restored = 0;

        // Restore title
        if (originalContent.title) {
          const $titleField = $('input[name="title[0][value]"]');
          if ($titleField.length > 0) {
            $titleField.val(originalContent.title);
            $titleField.trigger('input').trigger('change');
            restored++;
          }
        }

        // Restore body content
        if (originalContent.content) {
          // Try CKEditor 5 first
          if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {
            let editorRestored = false;
            try {
              Drupal.CKEditor5Instances.forEach((editor, key) => {
                if (editor.sourceElement.id === 'edit-body-0-value') {
                  const editorInstance = Drupal.CKEditor5Instances.get(key);
                  if (editorInstance) {
                    editorInstance.setData(originalContent.content);
                    editorRestored = true;
                    restored++;
                  }
                }
              });
            } catch (e) {
              console.warn('Failed to restore to CKEditor 5:', e);
            }

            // Fallback to textarea if CKEditor 5 restoration failed
            if (!editorRestored) {
              const $bodyField = $('textarea[name="body[0][value]"]');
              if ($bodyField.length > 0) {
                $bodyField.val(originalContent.content);
                $bodyField.trigger('input').trigger('change');
                restored++;
              }
            }
          }
        }

        if (restored > 0) {
          this.showNotification(`Original content restored successfully!`, 'success');
        } else {
          this.showNotification('No content was restored', 'warning');
        }

      } catch (error) {
        console.error('Error restoring original content:', error);
        this.showNotification('Failed to restore original content: ' + error.message, 'error');
      }
    }

    insertSingleField(fieldKey, fieldValue) {


      // Special handling for HTML content (body field)
      if (fieldKey === 'html_content') {
        const cleanValue = this.cleanTableHtml(fieldValue);
        if (cleanValue.includes('<table')) {
          return this.insertTableIntoBodyField(cleanValue);
        } else {
          return this.insertIntoBodyField(fieldValue);
        }

      }

      // Handle other fields
      let selector = this.getFieldSelector(fieldKey);
      const $field = $(selector);

      if ($field.length === 0) {
        // Try alternative selectors
        const alternatives = this.getAlternativeSelectors(fieldKey);
        let found = false;

        for (const altSelector of alternatives) {
          const $altField = $(altSelector);
          if ($altField.length > 0) {
            this.insertValueIntoField($altField, fieldKey, fieldValue);
            found = true;
            break;
          }
        }

        if (!found) {
          this.showNotification(`Field "${fieldKey}" not found`, 'error');
        }
        return;
      }

      this.insertValueIntoField($field, fieldKey, fieldValue);
    }

    /**
     * Insert content into body field using CKEditor 5 API for Drupal 11
     */
    insertIntoBodyField(content) {
      // Method 1: Try CKEditor 5 (Drupal 11)
      if (typeof Drupal !== 'undefined' && Drupal.CKEditor5Instances) {

        try {
          let inserted = false;
          Drupal.CKEditor5Instances.forEach((editor, key) => {
            if (editor.sourceElement.id === 'edit-body-0-value') {
              editor.setData(content);
              inserted = true;
            }
          });

          if (inserted) {
            this.showNotification('Content successfully inserted into body field!', 'success');
            this.highlightBodyField();
            return true;
          }
        } catch (error) {
          console.error('CKEditor 5 insertion failed:', error);
        }
      }

      // Method 2: Try standard textarea fallback
      const bodySelectors = [
        'textarea[name="body[0][value]"]',
        '#edit-body-0-value',
        '.field--name-body textarea'
      ];

      for (const selector of bodySelectors) {
        const $field = $(selector);
        if ($field.length > 0) {
          $field.val(content);
          $field.trigger('input').trigger('change');

          this.showNotification('Content inserted into body field!', 'success');
          this.highlightBodyField();
          return true;
        }
      }

      this.copyToClipboard(content, 'Body field not found. Content copied to clipboard - please paste manually.');
      return false;
    }

    highlightBodyField() {
      const $bodyField = $('#edit-body-0-value, textarea[name="body[0][value]"]');
      if ($bodyField.length > 0) {
        $bodyField.css({
          'border': '3px solid #4CAF50',
          'background-color': '#f0fff0'
        });

        setTimeout(() => {
          $bodyField.css({
            'border': '',
            'background-color': ''
          });
        }, 3000);

        // Scroll to field
        $bodyField[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
      }
    }

    insertValueIntoField($field, fieldKey, fieldValue) {
      let valueToInsert = fieldValue;

      if (Array.isArray(fieldValue)) {
        valueToInsert = fieldValue.join(', ');
      }

      $field.val(valueToInsert);
      $field.trigger('input').trigger('change');

      this.highlightField($field);
      this.showNotification(`${fieldKey.replace('_', ' ')} inserted successfully!`, 'success');
    }

    getFieldSelector(fieldKey) {
      const selectors = {
        'title': 'input[name="title[0][value]"]',
        'meta_title': 'input[name*="meta_title"]',
        'meta_description': 'textarea[name*="meta_description"]',
        'url_alias': 'input[name="path[0][alias]"]',
        'keywords': 'input[name*="field_tags"], input[name*="tags"]'
      };

      return selectors[fieldKey] || `[name*="${fieldKey}"]`;
    }

    getAlternativeSelectors(fieldKey) {
      const alternatives = {
        'title': [
          '#edit-title-0-value',
          '.field--name-title input[type="text"]'
        ],
        'keywords': [
          'input[name*="field_tags"]',
          '.field--name-field-tags input'
        ],
        'url_alias': [
          '#edit-path-0-alias',
          '.field--name-path input'
        ],
        'meta_title': [
          'input[name*="field_meta_title"]',
          '.field--name-field-meta-title input'
        ],
        'meta_description': [
          'textarea[name*="field_meta_description"]',
          '.field--name-field-meta-description textarea'
        ]
      };

      return alternatives[fieldKey] || [];
    }

    regenerateContent() {
      if (this.chatHistory.length > 0) {
        const lastUserMessage = this.chatHistory.slice().reverse().find(msg => msg.type === 'user');
        if (lastUserMessage) {
          $('#ai-chatbot-input').val(lastUserMessage.content);
          this.sendMessage();
        }
      }
    }

    /**
 * Optimize SEO - trigger SEO optimization task
 */
    optimizeSeo() {
      if (this.isProcessing) {
        return;
      }

      try {
        // Check if there's existing content
        if (!this.checkExistingContent()) {
          this.showNotification('Please add some content first before optimizing for SEO', 'warning');
          return;
        }

        this.selectedTaskType = TaskType.CONTENT_OPTIMIZATION;
        const input = 'Optimize this content for SEO, improve readability, and suggest keyword enhancements';

        // Clear input and set new value
        $('#ai-chatbot-input').val('');
        $('#ai-chatbot-input').val(input);

        // Process the optimization request
        this.processOptimizationRequest(input);

      } catch (error) {
        this.showNotification('Failed to optimize content: ' + error.message, 'error');
      }
    }

    async processOptimizationRequest(input) {
      try {
        this.addMessage('user', input);
        this.setProcessingState(true);

        const payload = this.preparePayload(input);
        const response = await this.makeApiRequest(payload);

        if (response.success) {
          this.handleApiResponse(response.data);
          if (response.usage_info) {
            this.subscriptionInfo.usage_info = response.usage_info;
          }
        } else {
          this.handleApiError(response);
        }
      } catch (error) {
        this.addMessage('error', error.message);
      } finally {
        this.setProcessingState(false);
        this.selectedTaskType = null;
        $('#ai-chatbot-input').val('').attr('placeholder', 'Type your message or \'/\' for commands...');
      }
    }

    getLastResponseContent() {
      try {
        if (this.chatHistory && this.chatHistory.length > 0) {
          for (let i = this.chatHistory.length - 1; i >= 0; i--) {
            const message = this.chatHistory[i];
            if (message.type === 'assistant' && message.data && message.data.content) {
              const content = message.data.content;

              // Handle optimized content structure
              if (content.title || content.html_content) {
                return {
                  title: content.title,
                  html_content: content.html_content,
                  // Add other fields as null to maintain compatibility
                  kicker: null,
                  meta_title: null,
                  meta_description: null,
                  url_alias: null,
                  keywords: null
                };
              }

              // Handle regular content structure
              return content;
            }
          }
        }
        return {};
      } catch (error) {
        console.error('Error in getLastResponseContent:', error);
        return {};
      }
    }

    highlightField($field) {
      $field.css({
        'border': '2px solid #4CAF50',
        'background-color': '#f0fff0'
      });

      setTimeout(() => {
        $field.css({
          'border': '',
          'background-color': ''
        });
      }, 2000);
    }

    async handleSync() {
      const $syncBtn = $('#ai-chatbot-sync');
      $syncBtn.prop('disabled', true);
      this.showNotification('Refreshing subscription status...', 'info');

      try {
        const response = await fetch('/ai-assistant/auth/refresh', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': drupalSettings.user.csrf_token || '',
          },
        });

        const result = await response.json();

        if (result.success) {
          this.authState = {
            authenticated: result.data.isAuthenticated,
            user: result.data.user,
            subscription: result.data.subscription,
          };

          if (this.authState.authenticated && this.authState.subscription) {
            await this.getSubscriptionStatus();
          }

          this.checkInitialState();
          this.showNotification('Status refreshed successfully!', 'success');
        } else {
          throw new Error(result.message);
        }
      } catch (error) {
        console.error('Sync failed:', error);
        this.showNotification('Failed to refresh status: ' + error.message, 'error');
      } finally {
        $syncBtn.prop('disabled', false);
      }
    }

    copyToClipboard(text, successMessage = 'Content copied to clipboard!') {
      if (navigator.clipboard && window.isSecureContext) {
        navigator.clipboard.writeText(text).then(() => {
          this.showNotification(successMessage, 'success');
        }).catch(err => {
          this.fallbackCopyToClipboard(text, successMessage);
        });
      } else {
        this.fallbackCopyToClipboard(text, successMessage);
      }
    }

    fallbackCopyToClipboard(text, successMessage) {
      const textArea = document.createElement('textarea');
      textArea.value = text;
      textArea.style.position = 'fixed';
      textArea.style.left = '-999999px';
      document.body.appendChild(textArea);
      textArea.select();

      try {
        document.execCommand('copy');
        this.showNotification(successMessage, 'success');
      } catch (err) {
        this.showNotification('Failed to copy content', 'error');
      }

      document.body.removeChild(textArea);
    }

    showNotification(message, type = 'info') {
      const $notification = $(`<div class="ai-notification ai-notification--${type}">${message}</div>`);
      $('#ai-chatbot-status').html($notification);
      setTimeout(() => $notification.fadeOut(), 3000);
    }

    setProcessingState(processing) {
      this.isProcessing = processing;
      $('#ai-chatbot-send').prop('disabled', processing);
      if (processing) {
        this.addMessage('system', 'Processing your request...');
      }
    }

    addMessage(type, content, data = null) {
      try {
        const timestamp = new Date().toLocaleTimeString();
        const messageHtml = `
          <div class="ai-chatbot__message ai-chatbot__message--${type}">
            <div class="ai-chatbot__message-content">${content}</div>
            <div class="ai-chatbot__message-time">${timestamp}</div>
          </div>
        `;

        $('#ai-chatbot-messages').append(messageHtml);
        this.chatHistory.push({ type, content, data, timestamp });
        this.scrollToBottom();
      } catch (error) {
        console.error('Error in addMessage:', error);
      }
    }

    scrollToBottom() {
      const $messages = $('#ai-chatbot-messages');
      $messages.scrollTop($messages[0].scrollHeight);
    }

    escapeHtml(text) {
      const div = document.createElement('div');
      div.textContent = text;
      return div.innerHTML;
    }
  }

  // Initialize when DOM is ready
  Drupal.behaviors.aiContentAssistant = {
    attach: function (context, settings) {
      if (context === document) {
        new AIContentAssistant();
      }
    }
  };

})(jQuery, Drupal, drupalSettings);
