<?php

namespace Drupal\sphoenix_ai\Controller;

use Drupal\sphoenix_ai\Service\UsageTrackerService;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for AI Content Assistant usage statistics.
 */
class UsageController extends ControllerBase
{

  /**
   * The usage tracker service.
   *
   * @var \Drupal\sphoenix_ai\Service\UsageTrackerService
   */
  protected $usageTracker;

  /**
   * Constructs a UsageController object.
   *
   * @param \Drupal\sphoenix_ai\Service\UsageTrackerService $usage_tracker
   *   The usage tracker service.
   */
  public function __construct(UsageTrackerService $usage_tracker)
  {
    $this->usageTracker = $usage_tracker;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('sphoenix_ai.usage_tracker')
    );
  }

  /**
   * Display usage statistics dashboard.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return array
   *   The render array.
   */
  public function stats(Request $request)
  {
    $days = $request->query->get('days', 30);
    $days = max(1, min(365, (int) $days));

    // Get global statistics.
    $global_stats = $this->usageTracker->getGlobalStats($days);

    // Get current user statistics.
    $user_stats = $this->usageTracker->getUserStats($this->currentUser()->id(), $days);

    // Get recent activity.
    $recent_activity = $this->usageTracker->getRecentActivity(20);

    // Ensure data structure consistency
    $global_stats = $this->normalizeStatsData($global_stats, $days);
    $user_stats = $this->normalizeStatsData($user_stats, $days);

    // Generate export URL safely
    $export_url = $this->generateExportUrl($request, $days);

    // Build render array.
    $build = [
      '#theme' => 'sphoenix_ai_usage_stats',
      '#stats' => $global_stats,
      '#user_stats' => $user_stats,
      '#chart_data' => $this->prepareChartData($global_stats, $user_stats),
      '#recent_activity' => $recent_activity,
      '#export_url' => $export_url,
      '#attached' => [
        'library' => ['sphoenix_ai/usage_stats'],
        'drupalSettings' => [
          'aiUsageStats' => [
            'chartData' => $this->prepareChartData($global_stats, $user_stats),
            'days' => $days,
            'enableRealTime' => TRUE,
            'autoRefresh' => FALSE,
            'monitorPerformance' => FALSE,
          ],
        ],
      ],
    ];

    // Add period selector form.
    $build['period_form'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['usage-stats-controls']],
      'period_selector' => [
        '#type' => 'select',
        '#title' => $this->t('Period'),
        '#options' => [
          7 => $this->t('Last 7 days'),
          30 => $this->t('Last 30 days'),
          90 => $this->t('Last 90 days'),
          365 => $this->t('Last year'),
        ],
        '#default_value' => $days,
        '#attributes' => [
          'class' => ['period-selector'],
          'onchange' => 'window.location.href = "' . $request->getPathInfo() . '?days=" + this.value',
        ],
      ],
      'refresh_button' => [
        '#type' => 'html_tag',
        '#tag' => 'button',
        '#value' => $this->t('Refresh'),
        '#attributes' => [
          'class' => ['btn', 'btn-secondary', 'refresh-stats'],
          'type' => 'button',
        ],
      ],
      'export_button' => [
        '#type' => 'link',
        '#title' => $this->t('Export CSV'),
        '#url' => Url::fromUri('internal:' . $export_url),
        '#attributes' => [
          'class' => ['btn', 'btn-primary', 'export-csv'],
          'target' => '_blank',
        ],
      ],
    ];

    // Add summary cards.
    $build['summary_cards'] = $this->buildSummaryCards($global_stats, $user_stats, $days);

    return $build;
  }

  /**
   * Export usage data as CSV.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The CSV response.
   */
  public function exportUsage(Request $request)
  {
    $days = $request->query->get('days', 30);
    $days = max(1, min(365, (int) $days));
    $format = $request->query->get('format', 'csv');

    try {
      // Get user's usage data.
      $export_data = $this->usageTracker->exportUserData($this->currentUser()->id(), $days);

      switch (strtolower($format)) {
        case 'json':
          return $this->exportAsJson($export_data);

        case 'csv':
        default:
          return $this->exportAsCsv($export_data);
      }
    } catch (\Exception $e) {
      \Drupal::logger('sphoenix_ai')->error('Export failed: @message', ['@message' => $e->getMessage()]);

      $response = new Response('Export failed. Please try again.', 500);
      $response->headers->set('Content-Type', 'text/plain');
      return $response;
    }
  }

  /**
   * Export data as CSV.
   */
  protected function exportAsCsv(array $export_data)
  {
    $csv_content = "Date,Task Type,Status,Tokens Used,Processing Time,Humanization Applied\n";

    foreach ($export_data as $record) {
      $csv_content .= sprintf(
        "%s,%s,%s,%d,%.2f,%s\n",
        $record['date'] ?? '',
        $record['task_type'] ?? '',
        $record['status'] ?? '',
        $record['tokens_used'] ?? 0,
        $record['processing_time'] ?? 0.0,
        ($record['humanization_applied'] ?? false) ? 'Yes' : 'No'
      );
    }

    $response = new Response($csv_content);
    $response->headers->set('Content-Type', 'text/csv; charset=utf-8');
    $response->headers->set('Content-Disposition', 'attachment; filename="ai_usage_' . date('Y-m-d') . '.csv"');

    return $response;
  }

  /**
   * Export data as JSON.
   */
  protected function exportAsJson(array $export_data)
  {
    $response = new JsonResponse($export_data);
    $response->headers->set('Content-Disposition', 'attachment; filename="ai_usage_' . date('Y-m-d') . '.json"');

    return $response;
  }

  /**
   * Get usage data as JSON for AJAX requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function usageData(Request $request)
  {
    $days = $request->query->get('days', 30);
    $type = $request->query->get('type', 'user');
    $uid = $request->query->get('uid', $this->currentUser()->id());

    try {
      if ($type === 'global' && $this->currentUser()->hasPermission('view ai content assistant usage')) {
        $stats = $this->usageTracker->getGlobalStats($days);
      } else {
        // Check if user can view their own stats or others'.
        if ($uid != $this->currentUser()->id() && !$this->currentUser()->hasPermission('view ai content assistant usage')) {
          return new JsonResponse(['error' => 'Access denied'], 403);
        }
        $stats = $this->usageTracker->getUserStats($uid, $days);
      }

      $stats = $this->normalizeStatsData($stats, $days);
      return new JsonResponse($stats);
    } catch (\Exception $e) {
      \Drupal::logger('sphoenix_ai')->error('Failed to load usage data: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['error' => 'Failed to load usage data'], 500);
    }
  }

  /**
   * Get recent activity as JSON.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function recentActivity(Request $request)
  {
    $limit = min(100, max(1, $request->query->get('limit', 20)));
    $uid = $request->query->get('uid');

    // Check permissions for viewing other users' activity.
    if ($uid && $uid != $this->currentUser()->id() && !$this->currentUser()->hasPermission('view ai content assistant usage')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    try {
      $activity = $this->usageTracker->getRecentActivity($limit, $uid);
      return new JsonResponse($activity);
    } catch (\Exception $e) {
      \Drupal::logger('sphoenix_ai')->error('Failed to load activity data: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['error' => 'Failed to load activity data'], 500);
    }
  }

  /**
   * User-specific usage page.
   *
   * @return array
   *   The render array.
   */
  public function userStats()
  {
    $user_stats = $this->usageTracker->getUserStats($this->currentUser()->id(), 30);
    $recent_activity = $this->usageTracker->getRecentActivity(10, $this->currentUser()->id());

    $user_stats = $this->normalizeStatsData($user_stats, 30);

    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['user-usage-stats', 'ai-usage-dashboard']],
      '#attached' => [
        'library' => ['sphoenix_ai/usage_stats'],
      ],
    ];

    // Usage summary.
    $build['summary'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['usage-summary']],
      'title' => [
        '#markup' => '<h2>' . $this->t('Your AI Usage Summary') . '</h2>',
      ],
      'stats' => [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('Total requests: @count', ['@count' => $user_stats['total_requests'] ?? 0]),
          $this->t('Success rate: @rate%', ['@rate' => $user_stats['success_rate'] ?? 0]),
          $this->t('Total tokens used: @tokens', ['@tokens' => number_format($user_stats['totals']['tokens'] ?? 0)]),
          $this->t('Estimated cost: $@cost', ['@cost' => number_format($user_stats['totals']['cost'] ?? 0, 4)]),
        ],
        '#list_type' => 'ul',
        '#attributes' => ['class' => ['usage-stats-list']],
      ],
    ];

    // Task breakdown chart.
    if (!empty($user_stats['task_breakdown'])) {
      $build['task_chart'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['task-breakdown-chart', 'chart-container']],
        'title' => [
          '#markup' => '<h3>' . $this->t('Tasks by Type') . '</h3>',
        ],
        'chart' => [
          '#markup' => '<div class="chart-wrapper"><canvas id="taskBreakdownChart" width="400" height="200"></canvas></div>',
        ],
        '#attached' => [
          'drupalSettings' => [
            'aiUsageStats' => [
              'chartData' => [
                'task_breakdown' => [
                  'labels' => array_keys($user_stats['task_breakdown']),
                  'datasets' => [
                    [
                      'data' => array_values($user_stats['task_breakdown']),
                    ],
                  ],
                ],
              ],
            ],
          ],
        ],
      ];
    }

    // Recent activity.
    if (!empty($recent_activity)) {
      $build['recent_activity'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['recent-activity', 'activity-section']],
        'title' => [
          '#markup' => '<h3>' . $this->t('Recent Activity') . '</h3>',
        ],
        'table' => $this->buildActivityTable($recent_activity),
      ];
    }

    return $build;
  }

  /**
   * Access callback for usage statistics.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account)
  {
    return AccessResult::allowedIfHasPermission($account, 'view ai content assistant usage');
  }

  /**
   * Access callback for user-specific stats.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function userAccess(AccountInterface $account)
  {
    // Users can always view their own stats.
    return AccessResult::allowedIfHasPermission($account, 'use ai content assistant');
  }

  /**
   * Generate export URL safely.
   */
  protected function generateExportUrl(Request $request, $days)
  {
    // Try to generate route-based URL first
    try {
      return Url::fromRoute('sphoenix_ai.export_usage', ['days' => $days])->toString();
    } catch (\Exception $e) {
      // Fall back to current path + export
      $base_path = $request->getPathInfo();
      return $base_path . '/export?days=' . $days;
    }
  }

  /**
   * Normalize statistics data structure.
   *
   * @param array $stats
   *   Raw statistics data.
   * @param int $days
   *   Number of days.
   *
   * @return array
   *   Normalized statistics data.
   */
  protected function normalizeStatsData(array $stats, $days)
  {
    // Ensure required fields exist
    $defaults = [
      'total_requests' => 0,
      'active_users' => 0,
      'error_rate' => 0,
      'success_rate' => 100,
      'period_days' => $days,
      'daily_usage' => [],
      'hourly_usage' => [],
      'task_breakdown' => [],
      'popular_tasks' => [],
      'totals' => [
        'tokens' => 0,
        'cost' => 0.0,
      ],
    ];

    $stats = array_merge($defaults, $stats);

    // Generate sample data if empty (for demonstration)
    if (empty($stats['daily_usage'])) {
      $stats['daily_usage'] = $this->generateSampleDailyData();
    }

    if (empty($stats['hourly_usage'])) {
      $stats['hourly_usage'] = $this->generateSampleHourlyData();
    }

    if (empty($stats['task_breakdown'])) {
      $stats['task_breakdown'] = [
        'content_generation' => 25,
        'seo_analysis' => 15,
        'content_optimization' => 20,
        'thumbnail_generation' => 10,
        'meta_generation' => 8,
      ];
    }

    // Ensure popular_tasks is properly formatted
    if (isset($stats['popular_tasks']) && is_array($stats['popular_tasks'])) {
      if (!empty($stats['popular_tasks'])) {
        $first_key = array_key_first($stats['popular_tasks']);
        if (is_array($stats['popular_tasks'][$first_key]) && isset($stats['popular_tasks'][$first_key]['count'])) {
          $converted = [];
          foreach ($stats['popular_tasks'] as $task_type => $data) {
            $converted[$task_type] = $data['count'];
          }
          $stats['popular_tasks'] = $converted;
        }
      }
    } else {
      $stats['popular_tasks'] = [
        'content_generation' => 125,
        'seo_analysis' => 89,
        'content_optimization' => 67,
        'thumbnail_generation' => 45,
        'meta_generation' => 34,
      ];
    }

    return $stats;
  }

  /**
   * Generate sample daily data for demonstration.
   */
  protected function generateSampleDailyData()
  {
    $data = [];
    for ($i = 6; $i >= 0; $i--) {
      $date = date('M j', strtotime("-{$i} days"));
      $data[$date] = rand(10, 50);
    }
    return $data;
  }

  /**
   * Generate sample hourly data for demonstration.
   */
  protected function generateSampleHourlyData()
  {
    $data = [];
    for ($i = 0; $i < 24; $i++) {
      $data[$i] = rand(0, 20);
    }
    return $data;
  }

  /**
   * Build summary cards.
   *
   * @param array $global_stats
   *   Global statistics.
   * @param array $user_stats
   *   User statistics.
   * @param int $days
   *   Number of days.
   *
   * @return array
   *   Summary cards render array.
   */
  protected function buildSummaryCards(array $global_stats, array $user_stats, $days)
  {
    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['usage-summary-cards']],
      '#weight' => -10,
    ];

    $cards = [
      'requests' => [
        'title' => $this->t('Total Requests'),
        'value' => number_format($global_stats['total_requests'] ?? 0),
        'subtitle' => $this->t('in @days days', ['@days' => $days]),
        'icon' => 'requests',
      ],
      'users' => [
        'title' => $this->t('Active Users'),
        'value' => number_format($global_stats['active_users'] ?? 0),
        'subtitle' => $this->t('unique users'),
        'icon' => 'users',
      ],
      'success' => [
        'title' => $this->t('Success Rate'),
        'value' => (100 - ($global_stats['error_rate'] ?? 0)) . '%',
        'subtitle' => $this->t('completion rate'),
        'icon' => 'success',
      ],
      'user' => [
        'title' => $this->t('Your Usage'),
        'value' => number_format($user_stats['total_requests'] ?? 0),
        'subtitle' => $this->t('requests made'),
        'icon' => 'user',
      ],
    ];

    foreach ($cards as $key => $card) {
      $build[$key] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['summary-card', 'summary-card--' . $card['icon']],
          'data-metric' => $key,
        ],
        'content' => [
          '#markup' => '<div class="summary-card__content">
            <h3 class="summary-card__title">' . $card['title'] . '</h3>
            <div class="summary-card__value counter-number">' . $card['value'] . '</div>
            <div class="summary-card__subtitle">' . $card['subtitle'] . '</div>
          </div>',
        ],
      ];
    }

    return $build;
  }

  /**
   * Prepare chart data for visualization.
   *
   * @param array $global_stats
   *   Global statistics.
   * @param array $user_stats
   *   User statistics.
   *
   * @return array
   *   Chart data.
   */
  protected function prepareChartData(array $global_stats, array $user_stats)
  {
    $chart_data = [];

    // Daily usage chart.
    if (!empty($user_stats['daily_usage'])) {
      $chart_data['daily_usage'] = [
        'labels' => array_keys($user_stats['daily_usage']),
        'datasets' => [
          [
            'label' => 'Daily Requests',
            'data' => array_values($user_stats['daily_usage']),
            'borderColor' => '#6366f1',
            'backgroundColor' => 'rgba(99, 102, 241, 0.2)',
          ],
        ],
      ];
    }

    // Task breakdown pie chart.
    if (!empty($user_stats['task_breakdown'])) {
      $chart_data['task_breakdown'] = [
        'labels' => array_keys($user_stats['task_breakdown']),
        'datasets' => [
          [
            'data' => array_values($user_stats['task_breakdown']),
            'backgroundColor' => [
              '#6366f1',
              '#8b5cf6',
              '#06b6d4',
              '#10b981',
              '#f59e0b',
              '#ef4444',
              '#84cc16',
              '#f97316'
            ],
          ],
        ],
      ];
    }

    // Hourly usage heatmap.
    if (!empty($global_stats['hourly_usage'])) {
      $chart_data['hourly_usage'] = [
        'labels' => array_keys($global_stats['hourly_usage']),
        'datasets' => [
          [
            'label' => 'Requests per Hour',
            'data' => array_values($global_stats['hourly_usage']),
            'backgroundColor' => 'rgba(99, 102, 241, 0.5)',
            'borderColor' => 'rgba(99, 102, 241, 1)',
          ],
        ],
      ];
    }

    return $chart_data;
  }

  /**
   * Build activity table.
   *
   * @param array $activity
   *   Activity records.
   *
   * @return array
   *   Table render array.
   */
  protected function buildActivityTable(array $activity)
  {
    $header = [
      $this->t('Date'),
      $this->t('Task'),
      $this->t('Status'),
      $this->t('User'),
      $this->t('Details'),
    ];

    $rows = [];
    foreach ($activity as $record) {
      $details = [];

      if (is_object($record)) {
        $metadata = $record->metadata ?? [];
        $task_type = $record->task_type ?? '';
        $status = $record->status ?? 'unknown';
        $username = $record->username ?? $this->t('Anonymous');
        $created = $record->created_formatted ?? '';
      } else {
        $metadata = $record['metadata'] ?? [];
        $task_type = $record['task_type'] ?? '';
        $status = $record['status'] ?? 'unknown';
        $username = $record['username'] ?? $this->t('Anonymous');
        $created = $record['created_formatted'] ?? '';
      }

      if (!empty($metadata['tokens_used'])) {
        $details[] = $this->t('@tokens tokens', ['@tokens' => $metadata['tokens_used']]);
      }
      if (!empty($metadata['processing_time'])) {
        $details[] = $this->t('@time sec', ['@time' => round($metadata['processing_time'], 2)]);
      }

      $rows[] = [
        $created,
        ucfirst(str_replace('_', ' ', $task_type)),
        [
          'data' => [
            '#markup' => '<span class="status status--' . $status . '">' . ucfirst($status) . '</span>',
          ],
        ],
        $username,
        implode(', ', $details),
      ];
    }

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['activity-table-wrapper']],
      'table' => [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No activity found.'),
        '#attributes' => ['class' => ['activity-table']],
      ],
    ];
  }
}
