<?php

namespace Drupal\sphoenix_ai\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure content types for SPhoenix AI.
 */
class ContentTypesForm extends ConfigFormBase
{

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a ContentTypesForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed configuration manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, EntityTypeManagerInterface $entity_type_manager)
  {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames()
  {
    return ['sphoenix_ai.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'sphoenix_ai_content_types_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $config = $this->config('sphoenix_ai.settings');
    $enabled_content_types = $config->get('enabled_content_types') ?: [];

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Select which content types should have the SPhoenix AI Content Assistant enabled. The AI chatbot will appear on node add/edit forms for selected content types.') . '</p>',
    ];

    // Get all available content types.
    $node_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();

    if (empty($node_types)) {
      $form['no_content_types'] = [
        '#markup' => '<p>' . $this->t('No content types found. Please create content types first.') . '</p>',
      ];
      return $form;
    }

    $form['content_types'] = [
      '#type' => 'details',
      '#title' => $this->t('Available Content Types'),
      '#open' => TRUE,
    ];

    $options = [];
    $descriptions = [];

    foreach ($node_types as $node_type) {
      $type_id = $node_type->id();
      $options[$type_id] = $node_type->label();
      $descriptions[$type_id] = $node_type->getDescription() ?: $this->t('No description available');
    }

    $form['content_types']['enabled_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enable AI Assistant for these content types:'),
      '#options' => $options,
      '#default_value' => $enabled_content_types,
      '#description' => $this->t('Select the content types where you want the AI chatbot to be available.'),
    ];

    // Show descriptions for each content type.
    foreach ($descriptions as $type_id => $description) {
      $form['content_types']['enabled_content_types'][$type_id]['#description'] = $description;
    }

    // Additional settings for enabled content types.
    $form['settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Content Type Settings'),
      '#open' => FALSE,
    ];

    $form['settings']['auto_field_detection'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-detect fields'),
      '#description' => $this->t('Automatically detect and map common fields like title, body, and meta fields.'),
      '#default_value' => $config->get('auto_field_detection') ?? TRUE,
    ];

    $form['settings']['require_confirmation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require confirmation before inserting content'),
      '#description' => $this->t('Show a confirmation dialog before AI-generated content is inserted into fields.'),
      '#default_value' => $config->get('require_confirmation') ?? TRUE,
    ];

    $form['settings']['show_usage_info'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show usage information'),
      '#description' => $this->t('Display usage statistics and subscription info in the chatbot.'),
      '#default_value' => $config->get('show_usage_info') ?? TRUE,
    ];

    $form['settings']['enable_suggestions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable content suggestions'),
      '#description' => $this->t('Show AI-powered content suggestions based on existing field content.'),
      '#default_value' => $config->get('enable_suggestions') ?? TRUE,
    ];

    // Preview section.
    if (!empty($enabled_content_types)) {
      $form['preview'] = [
        '#type' => 'details',
        '#title' => $this->t('Preview'),
        '#open' => FALSE,
      ];

      $enabled_list = [];
      foreach ($enabled_content_types as $type_id) {
        if (isset($options[$type_id])) {
          $enabled_list[] = $options[$type_id];
        }
      }

      $form['preview']['current_status'] = [
        '#markup' => '<p><strong>' . $this->t('Currently enabled for:') . '</strong> ' . implode(', ', $enabled_list) . '</p>',
      ];

      $form['preview']['preview_description'] = [
        '#markup' => '<p>' . $this->t('Users with the "use ai content assistant" permission will see the AI chatbot on node forms for these content types.') . '</p>',
      ];
    }

    // Permissions info.
    $form['permissions_info'] = [
      '#type' => 'details',
      '#title' => $this->t('Permissions'),
      '#open' => FALSE,
    ];

    $form['permissions_info']['info'] = [
      '#markup' => '<p>' . $this->t('Users need the following permissions to use the AI assistant:') . '</p>
        <ul>
          <li><strong>use ai content assistant</strong> - Required to see and use the AI chatbot</li>
          <li><strong>create [content-type] content</strong> or <strong>edit own [content-type] content</strong> - Required to access node forms</li>
        </ul>
        <p><a href="/admin/people/permissions#module-sphoenix_ai" target="_blank">' . $this->t('Configure permissions') . '</a></p>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $config = $this->config('sphoenix_ai.settings');

    // Get selected content types and filter out empty values.
    $enabled_content_types = array_keys(array_filter($form_state->getValue('enabled_content_types')));

    $config
      ->set('enabled_content_types', $enabled_content_types)
      ->set('auto_field_detection', $form_state->getValue('auto_field_detection'))
      ->set('require_confirmation', $form_state->getValue('require_confirmation'))
      ->set('show_usage_info', $form_state->getValue('show_usage_info'))
      ->set('enable_suggestions', $form_state->getValue('enable_suggestions'))
      ->save();

    $count = count($enabled_content_types);
    if ($count > 0) {
      $this->messenger()->addMessage($this->formatPlural(
        $count,
        'SPhoenix AI is now enabled for 1 content type.',
        'SPhoenix AI is now enabled for @count content types.'
      ));
    } else {
      $this->messenger()->addMessage($this->t('SPhoenix AI has been disabled for all content types.'));
    }

    parent::submitForm($form, $form_state);
  }
}
