<?php

namespace Drupal\sphoenix_ai\Form;

use Drupal\sphoenix_ai\Service\FieldMapperService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure field mapping for AI Content Assistant.
 */
class FieldMappingForm extends ConfigFormBase
{

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The field mapper service.
   *
   * @var \Drupal\sphoenix_ai\Service\FieldMapperService
   */
  protected $fieldMapper;

  /**
   * Constructs a FieldMappingForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed configuration manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\sphoenix_ai\Service\FieldMapperService $field_mapper
   *   The field mapper service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, EntityTypeManagerInterface $entity_type_manager, FieldMapperService $field_mapper)
  {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityTypeManager = $entity_type_manager;
    $this->fieldMapper = $field_mapper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
      $container->get('sphoenix_ai.field_mapper')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames()
  {
    return ['sphoenix_ai.field_mapping'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'sphoenix_ai_field_mapping';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $config = $this->config('sphoenix_ai.field_mapping');
    $settings_config = $this->config('sphoenix_ai.settings');

    // Get enabled content types.
    $enabled_types = $settings_config->get('enabled_content_types') ?: [];

    if (empty($enabled_types)) {
      $form['no_content_types'] = [
        '#markup' => $this->t('<p>No content types are enabled. Please <a href=":url">enable content types</a> first.</p>', [
          ':url' => '/admin/config/content/ai-content-assistant/content-types',
        ]),
      ];
      return $form;
    }

    $form['description'] = [
      '#markup' => $this->t('<p>Configure how AI-generated content maps to your Drupal fields. This determines which fields are populated when you use the "Insert" action in the chatbot.</p>'),
    ];

    // Get all mappings.
    $mappings = $config->get('mappings') ?: [];

    $form['mappings'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Content Type Field Mappings'),
    ];

    // Build form for each enabled content type.
    foreach ($enabled_types as $content_type) {
      $this->buildContentTypeMapping($form, $form_state, $content_type, $mappings[$content_type] ?? []);
    }

    // Global field priority settings.
    $form['global'] = [
      '#type' => 'details',
      '#title' => $this->t('Global Settings'),
      '#group' => 'mappings',
    ];

    $form['global']['default_insertion_method'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Insertion Method'),
      '#default_value' => $config->get('default_insertion_method') ?: 'replace',
      '#options' => [
        'replace' => $this->t('Replace field content'),
        'append' => $this->t('Append to existing content'),
        'prepend' => $this->t('Prepend to existing content'),
        'ask' => $this->t('Ask user each time'),
      ],
      '#description' => $this->t('Default method for inserting AI-generated content into fields.'),
    ];

    $form['global']['auto_map_fields'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-map fields'),
      '#default_value' => $config->get('auto_map_fields') ?: TRUE,
      '#description' => $this->t('Automatically suggest field mappings based on field names and types.'),
    ];

    $form['global']['preserve_formatting'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Preserve HTML formatting'),
      '#default_value' => $config->get('preserve_formatting') ?: TRUE,
      '#description' => $this->t('Keep HTML formatting when inserting content into rich text fields.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Build field mapping form for a specific content type.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $content_type
   *   The content type machine name.
   * @param array $mapping
   *   Existing mapping for this content type.
   */
  protected function buildContentTypeMapping(array &$form, FormStateInterface $form_state, $content_type, array $mapping)
  {
    // Get node type info.
    $node_type = $this->entityTypeManager->getStorage('node_type')->load($content_type);
    if (!$node_type) {
      return;
    }

    $type_label = $node_type->label();

    // Get text fields for this content type.
    $text_fields = $this->fieldMapper->getTextFields($content_type);

    $form[$content_type] = [
      '#type' => 'details',
      '#title' => $this->t('@type Fields', ['@type' => $type_label]),
      '#group' => 'mappings',
      '#tree' => TRUE,
    ];

    $form[$content_type]['description'] = [
      '#markup' => $this->t('<p>Map AI response fields to @type content type fields.</p>', [
        '@type' => $type_label,
      ]),
    ];

    // AI response field to Drupal field mappings.
    $ai_fields = [
      'title' => $this->t('Generated Title'),
      'html_content' => $this->t('Generated Content (HTML)'),
      'plain_text' => $this->t('Generated Content (Plain Text)'),
      'kicker' => $this->t('Kicker/Subtitle'),
      'meta_title' => $this->t('Meta Title'),
      'meta_description' => $this->t('Meta Description'),
      'keywords' => $this->t('Keywords/Tags'),
      'url_alias' => $this->t('URL Alias'),
    ];

    // Build field options.
    $field_options = ['' => $this->t('- Do not map -')];
    foreach ($text_fields as $field_name => $field_info) {
      $field_options[$field_name] = $this->t('@label (@type)', [
        '@label' => $field_info['label'],
        '@type' => $field_info['type'],
      ]);
    }

    foreach ($ai_fields as $ai_field => $ai_label) {
      $form[$content_type][$ai_field . '_field'] = [
        '#type' => 'select',
        '#title' => $ai_label,
        '#default_value' => $mapping[$ai_field . '_field'] ?? '',
        '#options' => $field_options,
        '#description' => $this->t('Select which Drupal field should receive this AI-generated content.'),
      ];

      // Insertion method for this specific field.
      $form[$content_type][$ai_field . '_method'] = [
        '#type' => 'select',
        '#title' => $this->t('@field Insertion Method', ['@field' => $ai_label]),
        '#default_value' => $mapping[$ai_field . '_method'] ?? 'default',
        '#options' => [
          'default' => $this->t('Use global default'),
          'replace' => $this->t('Replace field content'),
          'append' => $this->t('Append to existing content'),
          'prepend' => $this->t('Prepend to existing content'),
        ],
        '#states' => [
          'visible' => [
            ':input[name="' . $content_type . '[' . $ai_field . '_field]"]' => ['!value' => ''],
          ],
        ],
      ];
    }

    // Field priority settings.
    $form[$content_type]['priority'] = [
      '#type' => 'details',
      '#title' => $this->t('Field Priorities'),
      '#description' => $this->t('Set the order in which fields should be considered for content extraction and auto-suggestions.'),
      '#open' => FALSE,
    ];

    $priority_options = [];
    foreach ($text_fields as $field_name => $field_info) {
      $priority_options[$field_name] = $field_info['label'];
    }

    $form[$content_type]['priority']['field_priorities'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Field Priority Order'),
      '#options' => $priority_options,
      '#default_value' => $mapping['field_priorities'] ?? array_keys($text_fields),
      '#description' => $this->t('Fields will be processed in the order selected. Drag to reorder.'),
    ];

    // Custom field selectors for complex forms.
    $form[$content_type]['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form[$content_type]['advanced']['custom_selectors'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Field Selectors'),
      '#default_value' => $mapping['custom_selectors'] ?? '',
      '#description' => $this->t('Enter custom CSS selectors for fields, one per line in format: field_name|selector. Example: body|textarea[name="body[0][value]"]'),
    ];

    $form[$content_type]['advanced']['auto_detect_context'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-detect content context'),
      '#default_value' => $mapping['auto_detect_context'] ?? TRUE,
      '#description' => $this->t('Automatically analyze existing content to provide better AI suggestions.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state)
  {
    parent::validateForm($form, $form_state);

    $settings_config = $this->config('sphoenix_ai.settings');
    $enabled_types = $settings_config->get('enabled_content_types') ?: [];

    // Validate each content type mapping.
    foreach ($enabled_types as $content_type) {
      $content_type_values = $form_state->getValue($content_type);

      if (!$content_type_values) {
        continue;
      }

      // Validate custom selectors format.
      if (!empty($content_type_values['advanced']['custom_selectors'])) {
        $selectors = explode("\n", $content_type_values['advanced']['custom_selectors']);
        foreach ($selectors as $line_num => $selector_line) {
          $selector_line = trim($selector_line);
          if (empty($selector_line)) {
            continue;
          }

          if (strpos($selector_line, '|') === FALSE) {
            $form_state->setErrorByName(
              $content_type . '][advanced][custom_selectors',
              $this->t('Invalid selector format on line @line. Use: field_name|selector', [
                '@line' => $line_num + 1,
              ])
            );
          }
        }
      }

      // Check for duplicate field mappings.
      $mapped_fields = [];
      $ai_fields = ['title', 'html_content', 'plain_text', 'kicker', 'meta_title', 'meta_description', 'keywords', 'url_alias'];

      foreach ($ai_fields as $ai_field) {
        $drupal_field = $content_type_values[$ai_field . '_field'] ?? '';
        if (!empty($drupal_field)) {
          if (in_array($drupal_field, $mapped_fields)) {
            $form_state->setErrorByName(
              $content_type . '][' . $ai_field . '_field',
              $this->t('The field %field is already mapped to another AI response field. Each Drupal field can only be mapped once.', [
                '%field' => $drupal_field,
              ])
            );
          } else {
            $mapped_fields[] = $drupal_field;
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $config = $this->config('sphoenix_ai.field_mapping');
    $settings_config = $this->config('sphoenix_ai.settings');

    $enabled_types = $settings_config->get('enabled_content_types') ?: [];
    $mappings = [];

    // Process each content type.
    foreach ($enabled_types as $content_type) {
      $content_type_values = $form_state->getValue($content_type);

      if ($content_type_values) {
        // Process custom selectors.
        $custom_selectors = [];
        if (!empty($content_type_values['advanced']['custom_selectors'])) {
          $selectors = explode("\n", $content_type_values['advanced']['custom_selectors']);
          foreach ($selectors as $selector_line) {
            $selector_line = trim($selector_line);
            if (empty($selector_line)) {
              continue;
            }

            [$field_name, $selector] = explode('|', $selector_line, 2);
            $custom_selectors[trim($field_name)] = trim($selector);
          }
        }

        // Process field priorities.
        $field_priorities = array_keys(array_filter($content_type_values['priority']['field_priorities'] ?? []));

        $mappings[$content_type] = [
          'title_field' => $content_type_values['title_field'] ?? '',
          'title_method' => $content_type_values['title_method'] ?? 'default',
          'html_content_field' => $content_type_values['html_content_field'] ?? '',
          'html_content_method' => $content_type_values['html_content_method'] ?? 'default',
          'plain_text_field' => $content_type_values['plain_text_field'] ?? '',
          'plain_text_method' => $content_type_values['plain_text_method'] ?? 'default',
          'kicker_field' => $content_type_values['kicker_field'] ?? '',
          'kicker_method' => $content_type_values['kicker_method'] ?? 'default',
          'meta_title_field' => $content_type_values['meta_title_field'] ?? '',
          'meta_title_method' => $content_type_values['meta_title_method'] ?? 'default',
          'meta_description_field' => $content_type_values['meta_description_field'] ?? '',
          'meta_description_method' => $content_type_values['meta_description_method'] ?? 'default',
          'keywords_field' => $content_type_values['keywords_field'] ?? '',
          'keywords_method' => $content_type_values['keywords_method'] ?? 'default',
          'url_alias_field' => $content_type_values['url_alias_field'] ?? '',
          'url_alias_method' => $content_type_values['url_alias_method'] ?? 'default',
          'field_priorities' => $field_priorities,
          'custom_selectors' => $custom_selectors,
          'auto_detect_context' => $content_type_values['advanced']['auto_detect_context'] ?? TRUE,
        ];
      }
    }

    // Save all mappings.
    $config
      ->set('mappings', $mappings)
      ->set('default_insertion_method', $form_state->getValue(['global', 'default_insertion_method']))
      ->set('auto_map_fields', $form_state->getValue(['global', 'auto_map_fields']))
      ->set('preserve_formatting', $form_state->getValue(['global', 'preserve_formatting']))
      ->save();

    $this->messenger()->addMessage($this->t('Field mapping configuration has been saved.'));
    parent::submitForm($form, $form_state);
  }
}
