<?php

namespace Drupal\sphoenix_ai\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Http\ClientFactory;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for communicating with the SPhoenix AI API.
 */
class ApiClientService
{

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * Task type constants.
   */
  const TASK_TYPES = [
    'CONTENT_GENERATION' => 'content_generation',
    'CONTENT_REGENERATION' => 'content_regeneration',
    'CONTENT_OPTIMIZATION' => 'content_optimization',
    'THUMBNAIL_GENERATION' => 'thumbnail_generation',
    'GRAPH_GENERATION' => 'graph_generation',
    'TABLE_GENERATION' => 'table_generation',
    'META_GENERATION' => 'meta_generation',
    'SEO_ANALYSIS' => 'seo_analysis',
    'CONTENT_ANALYSIS' => 'content_analysis',
    'KEYWORD_EXTRACTION' => 'keyword_extraction',
    'CONTENT_SUMMARY' => 'content_summary',
    'TITLE_GENERATION' => 'title_generation',
    'DESCRIPTION_GENERATION' => 'description_generation',
  ];

  /**
   * Constructs an ApiClientService object.
   */
  public function __construct(Client $http_client, ConfigFactoryInterface $config_factory, LoggerChannelInterface $logger, AccountProxyInterface $current_user, CacheBackendInterface $cache)
  {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->currentUser = $current_user;
    $this->cache = $cache;
  }

  /**
   * Make API request to the chat endpoint.
   *
   * @param array $request_data
   *   The request data.
   *
   * @return array
   *   The API response.
   *
   * @throws \Exception
   *   If the request fails.
   */
  public function makeApiRequest(array $request_data)
  {
    // Validate configuration
    $validation = $this->validateConfiguration();
    if (!$validation['valid']) {
      throw new \Exception('API configuration invalid: ' . implode(', ', $validation['errors']));
    }

    // Use Constants for API URL
    $api_url = Constants::getEndpointUrl('CMS_CHAT');

    // Get authentication service to retrieve token
    $auth_service = \Drupal::service('sphoenix_ai.authentication');
    $access_token = $auth_service->getAccessToken();

    if (!$access_token) {
      throw new \Exception('Authentication required. Please login first.');
    }

    // Prepare request payload using Constants and config
    $payload = $this->prepareRequestPayload($request_data);

    // Check cache if enabled
    $cache_key = null;
    if ($this->isCacheResponsesEnabled() && $this->isCacheableRequest($request_data)) {
      $cache_key = $this->generateCacheKey($payload);
      $cached_response = $this->getCachedResponse($cache_key);

      if ($cached_response) {
        $this->logger->info('Serving cached response for user @uid', [
          '@uid' => $this->currentUser->id(),
        ]);
        return $cached_response;
      }
    }

    try {
      $request_options = [
        'json' => $payload,
        'headers' => [
          'Authorization' => 'Bearer ' . $access_token,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
          'User-Agent' => 'SPhoenix-AI-Drupal/1.0.0',
        ],
        'timeout' => 300, // 5 minutes for AI processing
        'connect_timeout' => 30, // 30 seconds to connect
      ];

      $this->logger->info('Making API request to @url for user @uid with task @task', [
        '@url' => $api_url,
        '@uid' => $this->currentUser->id(),
        '@task' => $payload['task_type'],
      ]);

      $response = $this->httpClient->request('POST', $api_url, $request_options);

      $response_body = $response->getBody()->getContents();
      $response_data = json_decode($response_body, TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        throw new \Exception('Invalid JSON response from API: ' . json_last_error_msg());
      }

      // Validate response structure
      if (!isset($response_data['data'])) {
        throw new \Exception('Invalid API response structure - missing data field');
      }

      // Cache successful responses
      if ($cache_key && $this->isCacheResponsesEnabled()) {
        $cache_duration = $this->getCacheDuration();
        $this->setCachedResponse($cache_key, $response_data, $cache_duration);
      }

      // Log successful request
      $this->logger->info('AI API request successful for user @uid, task @task', [
        '@uid' => $this->currentUser->id(),
        '@task' => $payload['task_type'],
      ]);

      return $response_data;
    } catch (RequestException $e) {
      $error_message = 'API request failed';
      $status_code = $e->getCode();

      if ($e->hasResponse()) {
        $error_response = $e->getResponse();
        $status_code = $error_response->getStatusCode();

        try {
          $error_data = json_decode($error_response->getBody()->getContents(), TRUE);
          if (isset($error_data['message'])) {
            $error_message = $error_data['message'];
          } elseif (isset($error_data['error'])) {
            $error_message = $error_data['error'];
          }
        } catch (\Exception $decode_error) {
          // Use default error message
        }
      }

      // Handle specific error codes
      if ($status_code === 401) {
        $error_message = 'Authentication failed. Please login again.';
      } elseif ($status_code === 402) {
        $error_message = 'Usage limit exceeded. Please upgrade your plan.';
      } elseif ($status_code === 429) {
        $error_message = 'Rate limit exceeded. Please wait and try again.';
      } elseif ($status_code >= 500) {
        $error_message = 'AI service temporarily unavailable. Please try again later.';
      }

      $this->logger->error('AI API request failed for user @uid: @error (Status: @status)', [
        '@uid' => $this->currentUser->id(),
        '@error' => $e->getMessage(),
        '@status' => $status_code,
      ]);

      throw new \Exception($error_message);
    } catch (GuzzleException $e) {
      $this->logger->error('AI API connection error for user @uid: @error', [
        '@uid' => $this->currentUser->id(),
        '@error' => $e->getMessage(),
      ]);

      throw new \Exception('Failed to connect to AI service. Please check your internet connection and try again.');
    }
  }

  /**
   * Prepare the request payload according to API specification.
   *
   * @param array $request_data
   *   The original request data.
   *
   * @return array
   *   The formatted payload.
   */
  protected function prepareRequestPayload(array $request_data)
  {
    $config = $this->configFactory->get('sphoenix_ai.settings');

    // Start with defaults from config using Constants fallbacks
    $payload = [
      'tool_id' => Constants::TOOL_ID,
      'temperature' => (float) ($config->get('default_temperature') ?: Constants::DEFAULT_TEMPERATURE),
      'tone' => $config->get('default_tone') ?: Constants::DEFAULT_TONE,
      'word_length' => (int) ($config->get('default_word_length') ?: Constants::DEFAULT_WORD_LENGTH),
      'paragraphs' => (int) ($config->get('default_paragraphs') ?: Constants::DEFAULT_PARAGRAPHS),
      'include_summary' => (bool) ($config->get('include_summary') ?? TRUE),
      'include_faq' => (bool) ($config->get('include_faq') ?? FALSE),
      'width' => (int) ($config->get('default_image_width') ?: 1024),
      'height' => (int) ($config->get('default_image_height') ?: 1024),
      'target_keywords' => $config->get('default_keywords') ?: [],
      'competitor_urls' => $config->get('competitor_urls') ?: [],
      'enable_humanization' => (bool) ($config->get('enable_humanization') ?? TRUE),
      'humanization_level' => $config->get('humanization_level') ?: 'medium',
      'trigger_mode' => 'manual',
      'execution_source' => 'drupal_cms',
      'additional_context' => (object) [], // Empty object as per API spec
    ];

    // Required fields from request
    if (isset($request_data['task_type'])) {
      $payload['task_type'] = $request_data['task_type'];
    }

    if (isset($request_data['prompt'])) {
      $payload['prompt'] = $request_data['prompt'];
    }

    // Optional fields that override defaults
    $optional_fields = [
      'source_url',
      'existing_content',
      'content_type',
      'field_type',
      'data_type',
      'data_source',
      'temperature',
      'tone',
      'word_length',
      'paragraphs',
      'include_summary',
      'include_faq',
      'width',
      'height',
      'target_keywords',
      'competitor_urls',
      'additional_context',
      'enable_humanization',
      'humanization_level'
    ];

    foreach ($optional_fields as $field) {
      if (isset($request_data[$field])) {
        // Type casting for specific fields
        if ($field === 'temperature') {
          $payload[$field] = (float) $request_data[$field];
        } elseif (in_array($field, ['word_length', 'paragraphs', 'width', 'height'])) {
          $payload[$field] = (int) $request_data[$field];
        } elseif (in_array($field, ['include_summary', 'include_faq', 'enable_humanization'])) {
          $payload[$field] = (bool) $request_data[$field];
        } elseif (in_array($field, ['target_keywords', 'competitor_urls'])) {
          // Ensure arrays
          $payload[$field] = is_array($request_data[$field]) ? $request_data[$field] : [$request_data[$field]];
        } elseif ($field === 'additional_context') {
          // Ensure object
          $payload[$field] = is_array($request_data[$field]) ? (object) $request_data[$field] : (object) [];
        } else {
          $payload[$field] = $request_data[$field];
        }
      }
    }

    return $payload;
  }

  /**
   * Check if request should be cached.
   *
   * @param array $request_data
   *   The request data.
   *
   * @return bool
   *   TRUE if cacheable.
   */
  protected function isCacheableRequest(array $request_data)
  {
    // Don't cache requests with user-specific existing content
    if (!empty($request_data['existing_content'])) {
      return FALSE;
    }

    // Don't cache analysis tasks as they depend on specific content
    $non_cacheable_tasks = [
      'content_regeneration',
      'content_optimization',
      'seo_analysis',
      'content_analysis',
      'keyword_extraction',
      'content_summary'
    ];

    return !in_array($request_data['task_type'], $non_cacheable_tasks);
  }

  /**
   * Check if cache responses is enabled.
   *
   * @return bool
   *   TRUE if caching is enabled.
   */
  protected function isCacheResponsesEnabled()
  {
    return $this->configFactory->get('sphoenix_ai.settings')->get('cache_responses') ?? TRUE;
  }

  /**
   * Get cache duration in seconds.
   *
   * @return int
   *   Cache duration in seconds.
   */
  protected function getCacheDuration()
  {
    $hours = (float) ($this->configFactory->get('sphoenix_ai.settings')->get('cache_duration') ?: 1);
    return (int) ($hours * 3600); // Convert hours to seconds
  }

  /**
   * Validate API configuration.
   *
   * @return array
   *   Validation result with 'valid' boolean and 'errors' array.
   */
  public function validateConfiguration()
  {
    $errors = [];

    // Check if Constants can provide API base URL
    try {
      $api_base_url = Constants::getApiBaseUrl();
      if (empty($api_base_url)) {
        $errors[] = 'API Base URL is required';
      } elseif (!filter_var($api_base_url, FILTER_VALIDATE_URL)) {
        $errors[] = 'API Base URL must be a valid URL';
      }
    } catch (\Exception $e) {
      $errors[] = 'API configuration error: ' . $e->getMessage();
    }

    // Check if endpoint is available
    try {
      $endpoint_url = Constants::getEndpointUrl('CMS_CHAT');
      if (empty($endpoint_url)) {
        $errors[] = 'CMS Chat endpoint is not configured';
      }
    } catch (\Exception $e) {
      $errors[] = 'Endpoint configuration error: ' . $e->getMessage();
    }

    // Check tool ID
    if (empty(Constants::TOOL_ID)) {
      $errors[] = 'Tool ID is required';
    }

    return [
      'valid' => empty($errors),
      'errors' => $errors,
    ];
  }

  /**
   * Test API connectivity.
   *
   * @return array
   *   Test result with success status and message.
   */
  public function testApiConnection()
  {
    try {
      $validation = $this->validateConfiguration();
      if (!$validation['valid']) {
        return [
          'success' => FALSE,
          'message' => 'Configuration invalid: ' . implode(', ', $validation['errors']),
        ];
      }

      // Use Constants for API base URL
      $api_base_url = Constants::getApiBaseUrl();

      // Test basic connectivity with a simple request
      $test_response = $this->httpClient->request('GET', $api_base_url . '/health', [
        'timeout' => 30,
        'headers' => [
          'User-Agent' => 'SPhoenix-AI-Drupal/1.0.0',
        ],
      ]);

      if ($test_response->getStatusCode() === 200) {
        return [
          'success' => TRUE,
          'message' => 'API connection successful',
        ];
      } else {
        return [
          'success' => FALSE,
          'message' => 'API returned unexpected status: ' . $test_response->getStatusCode(),
        ];
      }
    } catch (GuzzleException $e) {
      return [
        'success' => FALSE,
        'message' => 'Connection failed: ' . $e->getMessage(),
      ];
    } catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => 'Test failed: ' . $e->getMessage(),
      ];
    }
  }

  /**
   * Get available task types.
   *
   * @return array
   *   Array of task types with their configurations.
   */
  public function getTaskTypes()
  {
    $task_types = [];

    foreach (self::TASK_TYPES as $constant => $value) {
      $task_types[$value] = [
        'id' => $value,
        'name' => ucwords(str_replace('_', ' ', $value)),
        'constant' => $constant,
      ];
    }

    return $task_types;
  }

  /**
   * Check if a task type is valid.
   *
   * @param string $task_type
   *   The task type to validate.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  public function isValidTaskType($task_type)
  {
    return in_array($task_type, array_values(self::TASK_TYPES));
  }

  /**
   * Get cached API response.
   *
   * @param string $cache_key
   *   The cache key.
   *
   * @return mixed|null
   *   The cached data or NULL if not found.
   */
  public function getCachedResponse($cache_key)
  {
    if (!$this->isCacheResponsesEnabled()) {
      return NULL;
    }

    $cache = $this->cache->get($cache_key);

    if ($cache && $cache->valid) {
      $this->logger->debug('Cache hit for key: @key', ['@key' => $cache_key]);
      return $cache->data;
    }

    return NULL;
  }

  /**
   * Cache API response.
   *
   * @param string $cache_key
   *   The cache key.
   * @param mixed $data
   *   The data to cache.
   * @param int $expire
   *   Cache expiration time in seconds.
   */
  public function setCachedResponse($cache_key, $data, $expire = 3600)
  {
    if (!$this->isCacheResponsesEnabled()) {
      return;
    }

    $this->cache->set($cache_key, $data, time() + $expire, ['sphoenix_ai', 'sphoenix_ai_responses']);

    $this->logger->debug('Cached response for key: @key (expires in @expire seconds)', [
      '@key' => $cache_key,
      '@expire' => $expire,
    ]);
  }

  /**
   * Generate cache key for API request.
   *
   * @param array $request_data
   *   The request data.
   *
   * @return string
   *   The cache key.
   */
  public function generateCacheKey(array $request_data)
  {
    // Remove user-specific and time-sensitive data for caching
    $cache_data = $request_data;
    unset($cache_data['existing_content']);
    unset($cache_data['field_data']);

    // Include user ID only for personalized requests
    if (in_array($request_data['task_type'], ['meta_generation', 'title_generation'])) {
      $cache_data['uid'] = $this->currentUser->id();
    }

    ksort($cache_data); // Sort for consistent keys

    return 'sphoenix_ai:api_request:' . md5(serialize($cache_data));
  }

  /**
   * Clear all cached responses.
   */
  public function clearCache()
  {
    $this->cache->invalidateMultiple(['sphoenix_ai', 'sphoenix_ai_responses']);
    $this->logger->info('AI API cache cleared by user @uid', [
      '@uid' => $this->currentUser->id(),
    ]);
  }

  /**
   * Get API configuration for JavaScript.
   *
   * @return array
   *   Configuration array for frontend.
   */
  public function getClientConfiguration()
  {
    $config = $this->configFactory->get('sphoenix_ai.settings');

    return [
      'task_types' => $this->getTaskTypes(),
      'default_settings' => [
        'temperature' => $config->get('default_temperature') ?: Constants::DEFAULT_TEMPERATURE,
        'tone' => $config->get('default_tone') ?: Constants::DEFAULT_TONE,
        'word_length' => $config->get('default_word_length') ?: Constants::DEFAULT_WORD_LENGTH,
        'paragraphs' => $config->get('default_paragraphs') ?: Constants::DEFAULT_PARAGRAPHS,
        'include_summary' => $config->get('include_summary') ?? TRUE,
        'include_faq' => $config->get('include_faq') ?? FALSE,
      ],
      'ui_settings' => [
        'chatbot_position' => $config->get('chatbot_position') ?: 'right',
        'chatbot_theme' => $config->get('chatbot_theme') ?: 'default',
        'auto_expand_chatbot' => $config->get('auto_expand_chatbot') ?? FALSE,
        'auto_suggest_tasks' => $config->get('auto_suggest_tasks') ?? TRUE,
        'require_confirmation' => $config->get('require_confirmation') ?? TRUE,
      ],
      'endpoints' => [
        'chat' => '/ai-assistant/chat',
        'auth_status' => '/ai-assistant/auth/status',
        'login' => '/ai-assistant/auth/login',
        'logout' => '/ai-assistant/auth/logout',
        'subscription_status' => '/ai-assistant/subscription/status',
        'field_content' => '/ai-assistant/field-content',
        'suggestions' => '/ai-assistant/suggestions',
      ],
    ];
  }

  /**
   * Get usage statistics for API performance.
   *
   * @return array
   *   API performance statistics.
   */
  public function getApiStats()
  {
    $cache_stats = [
      'hit_ratio' => 0,
      'total_requests' => 0,
      'cached_requests' => 0,
    ];

    // This could be expanded to track actual cache hit ratios
    // For now, return basic structure

    return [
      'cache_stats' => $cache_stats,
      'configuration_valid' => $this->validateConfiguration()['valid'],
      'last_successful_request' => time(), // This could be tracked in state
      'api_base_url' => Constants::getApiBaseUrl(),
      'frontend_base_url' => Constants::getFrontendBaseUrl(),
    ];
  }
}
