<?php

namespace Drupal\sphoenix_ai\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Service for mapping AI response data to Drupal fields.
 */
class FieldMapperService
{

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a FieldMapperService object.
   *
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(EntityFieldManagerInterface $entity_field_manager, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory)
  {
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * Gets field mapping suggestions for AI response data.
   *
   * @param string $content_type
   *   The content type machine name.
   * @param array $response_data
   *   The AI response data.
   *
   * @return array
   *   Field mapping suggestions.
   */
  public function getFieldMappingSuggestions($content_type, array $response_data)
  {
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('node', $content_type);
    $suggestions = [];

    // Map AI response fields to Drupal fields.
    $mapping_rules = [
      'title' => ['title', 'field_title', 'name'],
      'html_content' => ['body', 'field_body', 'field_content', 'field_description'],
      'plain_text' => ['field_summary', 'field_teaser', 'field_excerpt'],
      'meta_title' => ['field_meta_title', 'field_seo_title', 'field_page_title'],
      'meta_description' => ['field_meta_description', 'field_seo_description'],
      'keywords' => ['field_tags', 'field_keywords', 'field_categories'],
      'url_alias' => ['path'],
      'kicker' => ['field_kicker', 'field_subtitle', 'field_lead'],
    ];

    foreach ($response_data as $response_key => $response_value) {
      if (empty($response_value) || !isset($mapping_rules[$response_key])) {
        continue;
      }

      $possible_fields = $mapping_rules[$response_key];
      $matched_field = $this->findBestFieldMatch($field_definitions, $possible_fields);

      if ($matched_field) {
        $field_definition = $field_definitions[$matched_field];
        $suggestions[$response_key] = [
          'field_name' => $matched_field,
          'field_label' => $field_definition->getLabel(),
          'field_type' => $field_definition->getType(),
          'confidence' => $this->calculateMappingConfidence($matched_field, $possible_fields),
          'insertion_method' => $this->getInsertionMethod($field_definition),
        ];
      }
    }

    return $suggestions;
  }

  /**
   * Gets all text-based fields for a content type.
   *
   * @param string $content_type
   *   The content type machine name.
   *
   * @return array
   *   Array of field information.
   */
  public function getTextFields($content_type)
  {
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('node', $content_type);
    $text_fields = [];

    $text_field_types = [
      'string',
      'string_long',
      'text',
      'text_long',
      'text_with_summary',
    ];

    foreach ($field_definitions as $field_name => $field_definition) {
      if (in_array($field_definition->getType(), $text_field_types)) {
        $text_fields[$field_name] = [
          'label' => $field_definition->getLabel(),
          'type' => $field_definition->getType(),
          'required' => $field_definition->isRequired(),
          'cardinality' => $field_definition->getFieldStorageDefinition()->getCardinality(),
          'priority' => $this->getFieldPriority($field_name, $field_definition),
        ];
      }
    }

    // Sort by priority (higher first).
    uasort($text_fields, function ($a, $b) {
      return $b['priority'] <=> $a['priority'];
    });

    return $text_fields;
  }

  /**
   * Extracts existing content from specified fields.
   *
   * @param string $content_type
   *   The content type machine name.
   * @param array $field_data
   *   The field data from the form.
   * @param array $target_fields
   *   Specific fields to extract from (optional).
   *
   * @return array
   *   Extracted content organized by field type.
   */
  public function extractExistingContent($content_type, array $field_data, array $target_fields = [])
  {
    $text_fields = $this->getTextFields($content_type);
    $extracted_content = [];

    foreach ($text_fields as $field_name => $field_info) {
      // Skip if target fields specified and this field not in targets.
      if (!empty($target_fields) && !in_array($field_name, $target_fields)) {
        continue;
      }

      if (isset($field_data[$field_name]) && !empty($field_data[$field_name])) {
        $field_value = $this->extractFieldValue($field_data[$field_name], $field_info['type']);

        if (!empty($field_value)) {
          $extracted_content[$field_name] = [
            'value' => $field_value,
            'label' => $field_info['label'],
            'type' => $field_info['type'],
            'length' => strlen(strip_tags($field_value)),
          ];
        }
      }
    }

    return $extracted_content;
  }

  /**
   * Prepares insertion instructions for the frontend.
   *
   * @param string $content_type
   *   The content type machine name.
   * @param array $response_data
   *   The AI response data.
   *
   * @return array
   *   Insertion instructions for JavaScript.
   */
  public function prepareInsertionInstructions($content_type, array $response_data)
  {
    $suggestions = $this->getFieldMappingSuggestions($content_type, $response_data);
    $instructions = [];

    foreach ($suggestions as $response_key => $suggestion) {
      $field_name = $suggestion['field_name'];
      $field_type = $suggestion['field_type'];
      $value = $response_data[$response_key];

      $instructions[] = [
        'response_key' => $response_key,
        'field_name' => $field_name,
        'field_type' => $field_type,
        'field_label' => $suggestion['field_label'],
        'value' => $value,
        'insertion_method' => $suggestion['insertion_method'],
        'selector' => $this->getFieldSelector($field_name, $field_type),
        'confidence' => $suggestion['confidence'],
      ];
    }

    return $instructions;
  }

  /**
   * Finds the best matching field from possible options.
   *
   * @param array $field_definitions
   *   Available field definitions.
   * @param array $possible_fields
   *   Array of possible field names to match.
   *
   * @return string|null
   *   The best matching field name or NULL.
   */
  protected function findBestFieldMatch(array $field_definitions, array $possible_fields)
  {
    foreach ($possible_fields as $field_name) {
      if (isset($field_definitions[$field_name])) {
        return $field_name;
      }
    }
    return NULL;
  }

  /**
   * Calculates mapping confidence score.
   *
   * @param string $matched_field
   *   The matched field name.
   * @param array $possible_fields
   *   The possible field names.
   *
   * @return float
   *   Confidence score between 0 and 1.
   */
  protected function calculateMappingConfidence($matched_field, array $possible_fields)
  {
    $position = array_search($matched_field, $possible_fields);
    $total_options = count($possible_fields);

    // Higher confidence for earlier matches.
    return 1.0 - ($position / $total_options);
  }

  /**
   * Determines the insertion method for a field.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   *
   * @return string
   *   The insertion method (replace, append, prepend).
   */
  protected function getInsertionMethod($field_definition)
  {
    $field_type = $field_definition->getType();

    switch ($field_type) {
      case 'string':
      case 'text':
        return 'replace';

      case 'string_long':
      case 'text_long':
      case 'text_with_summary':
        return 'replace_or_append';

      default:
        return 'replace';
    }
  }

  /**
   * Gets field priority for sorting.
   *
   * @param string $field_name
   *   The field name.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   *
   * @return int
   *   Priority score (higher = more important).
   */
  protected function getFieldPriority($field_name, $field_definition)
  {
    $priority = 0;

    // Title field gets highest priority.
    if ($field_name === 'title') {
      $priority += 100;
    }

    // Body/content fields get high priority.
    if (in_array($field_name, ['body', 'field_body', 'field_content'])) {
      $priority += 80;
    }

    // Required fields get bonus points.
    if ($field_definition->isRequired()) {
      $priority += 20;
    }

    // Longer text fields get priority for main content.
    if ($field_definition->getType() === 'text_with_summary') {
      $priority += 15;
    } elseif ($field_definition->getType() === 'text_long') {
      $priority += 10;
    }

    return $priority;
  }

  /**
   * Extracts value from field data based on field type.
   *
   * @param mixed $field_data
   *   The field data.
   * @param string $field_type
   *   The field type.
   *
   * @return string
   *   The extracted value.
   */
  protected function extractFieldValue($field_data, $field_type)
  {
    if (is_string($field_data)) {
      return $field_data;
    }

    if (is_array($field_data)) {
      // Handle different field structures.
      if (isset($field_data[0]['value'])) {
        return $field_data[0]['value'];
      } elseif (isset($field_data['value'])) {
        return $field_data['value'];
      } elseif (isset($field_data[0])) {
        return is_string($field_data[0]) ? $field_data[0] : '';
      }
    }

    return '';
  }

  /**
   * Gets the CSS/JavaScript selector for a field.
   *
   * @param string $field_name
   *   The field name.
   * @param string $field_type
   *   The field type.
   *
   * @return string
   *   The selector string.
   */
  protected function getFieldSelector($field_name, $field_type)
  {
    // Handle special cases.
    if ($field_name === 'title') {
      return 'input[name="title[0][value]"]';
    }

    // Standard field selectors based on field type.
    switch ($field_type) {
      case 'string':
        return sprintf('input[name="%s[0][value]"]', $field_name);

      case 'string_long':
      case 'text':
        return sprintf('textarea[name="%s[0][value]"]', $field_name);

      case 'text_long':
        return sprintf('textarea[name="%s[0][value]"]', $field_name);

      case 'text_with_summary':
        return sprintf('textarea[name="%s[0][value]"]', $field_name);

      default:
        return sprintf('[name^="%s"]', $field_name);
    }
  }
}
