<?php

namespace Drupal\sphoenix_ai\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\sphoenix_ai\Service\AuthenticationService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Url;

/**
 * Controller for handling OAuth callbacks.
 */
class AuthCallbackController extends ControllerBase
{

  /**
   * The authentication service.
   *
   * @var \Drupal\sphoenix_ai\Service\AuthenticationService
   */
  protected $authService;

  /**
   * Constructs an AuthCallbackController object.
   */
  public function __construct(AuthenticationService $auth_service)
  {
    $this->authService = $auth_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('sphoenix_ai.authentication')
    );
  }

  /**
   * Handle OAuth callback with token.
   */
  public function callback(Request $request)
  {
    $token = $request->query->get('token');
    $error = $request->query->get('error');

    if ($error) {
      $this->messenger()->addError($this->t('Authentication failed: @error', [
        '@error' => $error,
      ]));
      return $this->redirect('<front>');
    }

    if (!$token) {
      $this->messenger()->addError($this->t('Authentication failed: No token received.'));
      return $this->redirect('<front>');
    }

    // Process authentication callback
    $result = $this->authService->handleAuthCallback($token);
    if ($result['success']) {
      if (isset($result['requires_subscription']) && $result['requires_subscription']) {
        $this->messenger()->addWarning($this->t('Successfully logged in, but you need a CMS subscription to use AI features. <a href="@url" target="_blank">View CMS Plans</a>', [
          '@url' => \Drupal::service('sphoenix_ai.subscription')->getPlansUrl(),
        ]));
      } else {
        $this->messenger()->addMessage($this->t('Successfully logged in to SPhoenix AI!'));
      }

      // Redirect back to the referring page or node add/edit form
      $destination = $request->query->get('moveTo');
      if (!$destination) {
        // Try to redirect back to content form if possible
        $referer = $request->headers->get('referer');
        if ($referer && (strpos($referer, '/node/add/') !== false || strpos($referer, '/edit') !== false)) {
          return new RedirectResponse($referer);
        }
        $destination = '/<front>';
        $redirectto = Url::fromUserInput($destination)->toString();
      } else {
        $redirectto = $destination;
      }

      return new RedirectResponse($redirectto);
    } else {
      $this->messenger()->addError($this->t('Authentication failed: @error', [
        '@error' => $result['message'],
      ]));
      return $this->redirect('<front>');
    }
  }
}
