/**
 * @file
 * JavaScript for AI Content Assistant field mapping configuration.
 */

(function ($, Drupal, drupalSettings) {
  'use strict';

  /**
   * Field Mapping behavior.
   */
  Drupal.behaviors.aiFieldMapping = {
    attach: function (context, settings) {
      $('.ai-field-mapping-form', context).once('ai-field-mapping').each(function () {
        new FieldMappingManager(this, settings);
      });
    }
  };

  /**
   * Field Mapping Manager class.
   */
  function FieldMappingManager(element, settings) {
    this.element = element;
    this.settings = settings.aiContentAssistant || {};

    this.init();
  }

  FieldMappingManager.prototype = {

    /**
     * Initialize the field mapping manager.
     */
    init: function () {
      this.initSortablePriorities();
      this.initFieldMapping();
      this.initAutoSuggestions();
      this.initPreview();
      this.initValidation();
    },

    /**
     * Initialize sortable field priorities.
     */
    initSortablePriorities: function () {
      const self = this;

      $('.priority-list', this.element).sortable({
        handle: '.priority-handle',
        placeholder: 'priority-placeholder',
        tolerance: 'pointer',
        update: function (event, ui) {
          self.updateFieldPriorities($(this));
        }
      });

      // Style the sortable items.
      $('.priority-item', this.element).addClass('sortable-item');
    },

    /**
     * Update field priorities after sorting.
     */
    updateFieldPriorities: function ($list) {
      const priorities = [];

      $list.find('.priority-item').each(function (index) {
        const fieldName = $(this).find('.priority-checkbox').val();
        if ($(this).find('.priority-checkbox').is(':checked')) {
          priorities.push(fieldName);
        }
      });

      // Update hidden field or trigger save.
      console.log('Updated priorities:', priorities);
    },

    /**
     * Initialize field mapping functionality.
     */
    initFieldMapping: function () {
      const self = this;

      // Handle field mapping changes.
      $('.field-mapping-select', this.element).on('change', function () {
        const $select = $(this);
        const aiField = $select.data('ai-field');
        const drupalField = $select.val();

        self.updateFieldMapping(aiField, drupalField, $select);
        self.validateMapping();
        self.updatePreview();
      });

      // Handle insertion method changes.
      $('.field-mapping-method select', this.element).on('change', function () {
        self.validateMapping();
      });

      // Initialize mapping state.
      this.validateMapping();
      this.updatePreview();
    },

    /**
     * Update field mapping state.
     */
    updateFieldMapping: function (aiField, drupalField, $select) {
      const $container = $select.closest('.field-mapping-grid');

      // Show/hide method selector based on field selection.
      const $methodSelect = $container.find('.field-mapping-method');
      if (drupalField) {
        $methodSelect.show().find('select').prop('disabled', false);
        $container.addClass('field-mapped');
      } else {
        $methodSelect.hide().find('select').prop('disabled', true);
        $container.removeClass('field-mapped');
      }

      // Add confidence indicator.
      this.updateConfidenceIndicator($container, aiField, drupalField);
    },

    /**
     * Update confidence indicator for field mapping.
     */
    updateConfidenceIndicator: function ($container, aiField, drupalField) {
      let $indicator = $container.find('.mapping-confidence');

      if (!$indicator.length) {
        $indicator = $('<div class="mapping-confidence"></div>');
        $container.append($indicator);
      }

      if (!drupalField) {
        $indicator.hide();
        return;
      }

      const confidence = this.calculateMappingConfidence(aiField, drupalField);
      const confidenceClass = confidence > 0.8 ? 'high' : confidence > 0.5 ? 'medium' : 'low';

      $indicator.html(`
        <span class="confidence-text">Confidence:</span>
        <div class="confidence-indicator confidence-${confidenceClass}">
          <div class="confidence-bar"></div>
        </div>
        <span class="confidence-percent">${Math.round(confidence * 100)}%</span>
      `).show();
    },

    /**
     * Calculate mapping confidence score.
     */
    calculateMappingConfidence: function (aiField, drupalField) {
      const mappingRules = {
        title: ['title', 'field_title', 'name'],
        html_content: ['body', 'field_body', 'field_content'],
        meta_title: ['field_meta_title', 'field_seo_title'],
        meta_description: ['field_meta_description', 'field_seo_description'],
        keywords: ['field_tags', 'field_keywords']
      };

      if (!mappingRules[aiField]) {
        return 0.5; // Default confidence.
      }

      const rules = mappingRules[aiField];
      const index = rules.indexOf(drupalField);

      if (index === -1) {
        return 0.3; // Low confidence for non-standard mappings.
      }

      return 1.0 - (index / rules.length) * 0.5; // Higher confidence for earlier matches.
    },

    /**
     * Initialize auto-suggestions.
     */
    initAutoSuggestions: function () {
      const self = this;

      // Check if auto-suggestions are enabled.
      if (!this.settings.autoMapFields) {
        return;
      }

      this.generateAutoSuggestions();

      // Handle applying suggestions.
      $(document).on('click', '.suggestion-apply', function (e) {
        e.preventDefault();
        const aiField = $(this).data('ai-field');
        const drupalField = $(this).data('drupal-field');
        self.applySuggestion(aiField, drupalField);
      });
    },

    /**
     * Generate automatic mapping suggestions.
     */
    generateAutoSuggestions: function () {
      const suggestions = [];
      const self = this;

      $('.field-mapping-select', this.element).each(function () {
        const $select = $(this);
        const aiField = $select.data('ai-field');

        if ($select.val()) {
          return; // Already mapped.
        }

        const suggestion = self.suggestDrupalField(aiField, $select);
        if (suggestion) {
          suggestions.push({
            aiField: aiField,
            drupalField: suggestion.field,
            confidence: suggestion.confidence,
            $select: $select
          });
        }
      });

      if (suggestions.length > 0) {
        this.displaySuggestions(suggestions);
      }
    },

    /**
     * Suggest appropriate Drupal field for AI field.
     */
    suggestDrupalField: function (aiField, $select) {
      const options = $select.find('option').map(function () {
        return { value: $(this).val(), text: $(this).text() };
      }).get();

      const mappingRules = {
        title: ['title', 'field_title', 'name'],
        html_content: ['body', 'field_body', 'field_content', 'field_description'],
        plain_text: ['field_summary', 'field_excerpt', 'field_teaser'],
        meta_title: ['field_meta_title', 'field_seo_title', 'field_page_title'],
        meta_description: ['field_meta_description', 'field_seo_description'],
        keywords: ['field_tags', 'field_keywords', 'field_categories']
      };

      if (!mappingRules[aiField]) {
        return null;
      }

      for (let i = 0; i < mappingRules[aiField].length; i++) {
        const targetField = mappingRules[aiField][i];
        const match = options.find(opt => opt.value === targetField);

        if (match) {
          return {
            field: match.value,
            confidence: 1.0 - (i / mappingRules[aiField].length) * 0.3
          };
        }
      }

      return null;
    },

    /**
     * Display mapping suggestions.
     */
    displaySuggestions: function (suggestions) {
      if (suggestions.length === 0) {
        return;
      }

      let html = `
        <div class="auto-mapping-suggestions">
          <div class="auto-mapping-title">Suggested Field Mappings</div>
      `;

      suggestions.forEach(function (suggestion) {
        html += `
          <div class="suggestion-item">
            <strong>${suggestion.aiField.replace('_', ' ')}</strong> →
            <strong>${suggestion.drupalField}</strong>
            <button class="suggestion-apply"
                    data-ai-field="${suggestion.aiField}"
                    data-drupal-field="${suggestion.drupalField}">
              Apply
            </button>
            <span class="suggestion-confidence">(${Math.round(suggestion.confidence * 100)}% match)</span>
          </div>
        `;
      });

      html += '</div>';

      $(this.element).find('.field-mapping-section').first().before(html);
    },

    /**
     * Apply a mapping suggestion.
     */
    applySuggestion: function (aiField, drupalField) {
      const $select = $(`[data-ai-field="${aiField}"]`, this.element);
      $select.val(drupalField).trigger('change');

      // Remove the applied suggestion.
      $(`.suggestion-apply[data-ai-field="${aiField}"]`).closest('.suggestion-item').fadeOut();

      this.showMappingStatus('Mapping applied successfully!', 'success');
    },

    /**
     * Initialize preview functionality.
     */
    initPreview: function () {
      this.updatePreview();
    },

    /**
     * Update the mapping preview.
     */
    updatePreview: function () {
      const mappings = this.getCurrentMappings();
      let html = '<div class="preview-title">Current Mappings:</div>';

      if (Object.keys(mappings).length === 0) {
        html += '<div class="preview-unmapped">No fields mapped yet.</div>';
      } else {
        Object.entries(mappings).forEach(function ([aiField, drupalField]) {
          html += `
            <div class="preview-mapping">
              <span class="preview-ai-field">${aiField.replace('_', ' ')}</span>
              <span class="preview-arrow">→</span>
              <span class="preview-drupal-field">${drupalField}</span>
            </div>
          `;
        });
      }

      let $preview = $('.field-mapping-preview', this.element);
      if (!$preview.length) {
        $preview = $('<div class="field-mapping-preview"></div>');
        $(this.element).append($preview);
      }

      $preview.html(html);
    },

    /**
     * Get current field mappings.
     */
    getCurrentMappings: function () {
      const mappings = {};

      $('.field-mapping-select', this.element).each(function () {
        const $select = $(this);
        const aiField = $select.data('ai-field');
        const drupalField = $select.val();

        if (drupalField) {
          mappings[aiField] = drupalField;
        }
      });

      return mappings;
    },

    /**
     * Initialize validation.
     */
    initValidation: function () {
      this.validateMapping();
    },

    /**
     * Validate current field mappings.
     */
    validateMapping: function () {
      const mappings = this.getCurrentMappings();
      const drupalFields = Object.values(mappings);
      const duplicates = this.findDuplicates(drupalFields);

      // Clear previous validation states.
      $('.field-mapping-select', this.element)
        .removeClass('field-mapping-error field-mapping-warning field-mapping-success');

      let hasErrors = false;

      // Check for duplicate mappings.
      if (duplicates.length > 0) {
        duplicates.forEach((field) => {
          $(`.field-mapping-select[value="${field}"]`, this.element)
            .addClass('field-mapping-error');
        });

        this.showMappingStatus('Duplicate field mappings detected!', 'error');
        hasErrors = true;
      }

      // Validate required mappings.
      const requiredMappings = ['title', 'html_content'];
      const missingRequired = requiredMappings.filter(field => !mappings[field]);

      if (missingRequired.length > 0) {
        missingRequired.forEach((field) => {
          $(`.field-mapping-select[data-ai-field="${field}"]`, this.element)
            .addClass('field-mapping-warning');
        });

        if (!hasErrors) {
          this.showMappingStatus('Some important fields are not mapped.', 'warning');
        }
      }

      // Show success if no errors or warnings.
      if (!hasErrors && missingRequired.length === 0) {
        $('.field-mapping-select', this.element)
          .filter(function () { return $(this).val(); })
          .addClass('field-mapping-success');

        this.showMappingStatus('Field mappings look good!', 'success');
      }
    },

    /**
     * Find duplicate values in array.
     */
    findDuplicates: function (arr) {
      return arr.filter((item, index) => item && arr.indexOf(item) !== index);
    },

    /**
     * Show mapping status message.
     */
    showMappingStatus: function (message, type) {
      let $status = $('.mapping-status', this.element);

      if (!$status.length) {
        $status = $('<div class="mapping-status"></div>');
        $(this.element).prepend($status);
      }

      $status
        .removeClass('mapping-status-success mapping-status-warning mapping-status-error')
        .addClass(`mapping-status-${type}`)
        .text(message)
        .show();

      // Auto-hide success messages.
      if (type === 'success') {
        setTimeout(() => $status.fadeOut(), 3000);
      }
    }
  };

  /**
   * Helper function to initialize content type tabs.
   */
  Drupal.behaviors.aiContentTypeTabs = {
    attach: function (context, settings) {
      $('.content-type-tab', context).once('content-type-tabs').on('click', function () {
        const $tab = $(this);
        const target = $tab.data('target');

        // Update tab states.
        $('.content-type-tab').removeClass('active');
        $tab.addClass('active');

        // Show/hide panels.
        $('.content-type-panel').hide();
        $(target).show();
      });

      // Show first tab by default.
      $('.content-type-tab', context).first().trigger('click');
    }
  };

})(jQuery, Drupal, drupalSettings);
