/**
 * @file
 * Enhanced JavaScript for AI Content Assistant usage statistics.
 * Modern, interactive, and feature-rich dashboard implementation.
 */

(function ($, Drupal, drupalSettings) {
  'use strict';

  // Global chart instances registry
  window.aiChartInstances = {};

  /**
   * Enhanced Usage Statistics behavior.
   */
  Drupal.behaviors.aiUsageStats = {
    attach: function (context, settings) {
      // Use a more compatible approach for element initialization
      $('.ai-usage-dashboard', context).each(function () {
        var $element = $(this);

        // Check if already initialized
        if ($element.hasClass('ai-stats-initialized')) {
          return;
        }

        // Mark as initialized
        $element.addClass('ai-stats-initialized');

        // Initialize the manager
        new EnhancedUsageStatsManager(this, settings);
      });
    }
  };

  /**
   * Enhanced Usage Statistics Manager class with modern features.
   */
  function EnhancedUsageStatsManager(element, settings) {
    this.element = element;
    this.settings = settings.aiContentAssistant || settings.aiUsageStats || {};
    this.charts = {};
    this.refreshInterval = null;
    this.isVisible = true;
    this.animationQueue = [];

    this.init();
  }

  EnhancedUsageStatsManager.prototype = {

    /**
     * Initialize the enhanced usage stats manager.
     */
    init: function () {
      this.setupLoadingState();
      this.setupEventListeners();
      this.initCharts();
      this.setupInteractions();
      this.initRealTimeUpdates();
      this.initExportFunctionality();
      this.initAccessibility();
      this.startEntranceAnimations();
    },

    /**
     * Setup loading states with smooth transitions
     */
    setupLoadingState: function () {
      const $dashboard = $(this.element);

      // Add loading overlay
      if (!$dashboard.find('.loading-overlay').length) {
        $dashboard.append(`
          <div class="loading-overlay">
            <div class="loading-spinner">
              <div class="spinner-ring"></div>
              <div class="spinner-ring"></div>
              <div class="spinner-ring"></div>
            </div>
            <div class="loading-text">Loading dashboard...</div>
          </div>
        `);
      }

      // Remove loading after initialization
      setTimeout(() => {
        $dashboard.find('.loading-overlay').fadeOut(500, function () {
          $(this).remove();
        });
        $dashboard.addClass('dashboard-loaded');
      }, 1500);
    },

    /**
     * Setup comprehensive event listeners
     */
    setupEventListeners: function () {
      const self = this;

      // Period selector with smooth transitions
      $(this.element).on('change', '.period-selector', function () {
        const newPeriod = $(this).val();
        self.updatePeriod(newPeriod);
      });

      // Refresh button with loading state
      $(this.element).on('click', '.refresh-stats', function (e) {
        e.preventDefault();
        self.refreshDashboard();
      });

      // Chart interactions
      $(this.element).on('click', '.chart-control-btn', function () {
        const action = $(this).data('action');
        const chartContainer = $(this).closest('.chart-container');
        self.handleChartAction(action, chartContainer);
      });

      // Card interactions with animations
      $(this.element).on('mouseenter', '.summary-card', function () {
        $(this).addClass('card-hover');
        self.animateCardValue($(this));
      }).on('mouseleave', '.summary-card', function () {
        $(this).removeClass('card-hover');
      });

      // Tab switching for detailed views
      $(this.element).on('click', '.tab-link', function (e) {
        e.preventDefault();
        const targetTab = $(this).data('tab');
        self.switchTab(targetTab);
      });

      // Keyboard shortcuts
      $(document).on('keydown', function (e) {
        if ($(e.target).is('input, select, textarea')) return;

        // Ctrl+R: Refresh
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 82) {
          e.preventDefault();
          self.refreshDashboard();
        }

        // Ctrl+E: Export CSV
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 69) {
          e.preventDefault();
          $(self.element).find('.export-csv').first().trigger('click');
        }
      });

      // Visibility change handling
      $(document).on('visibilitychange', function () {
        self.isVisible = !document.hidden;
        if (self.isVisible && self.settings.enableRealTime) {
          self.updateStats();
        }
      });

      // Responsive chart resizing
      $(window).on('resize', self.debounce(function () {
        self.handleResize();
      }, 300));
    },

    /**
     * Initialize charts with enhanced configurations and animations.
     */
    initCharts: function () {
      // Check if Chart.js is loaded
      if (typeof Chart === 'undefined') {
        // Try to load from window.Chart or wait for it
        if (typeof window.Chart !== 'undefined') {
          Chart = window.Chart;
        } else {
          this.showNotification('Chart.js library not loaded. Charts will be disabled.', 'warning');
          return;
        }
      }

      // Enhanced Chart.js defaults
      Chart.defaults.font.family = "'Inter', -apple-system, BlinkMacSystemFont, sans-serif";
      Chart.defaults.plugins.tooltip.backgroundColor = 'rgba(15, 23, 42, 0.95)';
      Chart.defaults.plugins.tooltip.titleColor = '#f8fafc';
      Chart.defaults.plugins.tooltip.bodyColor = '#cbd5e1';
      Chart.defaults.plugins.tooltip.borderColor = '#334155';
      Chart.defaults.plugins.tooltip.borderWidth = 1;
      Chart.defaults.plugins.tooltip.cornerRadius = 8;
      Chart.defaults.plugins.tooltip.displayColors = true;

      this.initDailyUsageChart();
      this.initTaskBreakdownChart();
      this.initHourlyUsageChart();
      this.initTrendChart();
    },

    /**
     * Initialize daily usage chart with gradient and smooth animations.
     */
    initDailyUsageChart: function () {
      const canvas = document.getElementById('dailyUsageChart');
      if (!canvas) return;

      const ctx = canvas.getContext('2d');
      const chartData = this.settings.chartData?.daily_usage || this.generateSampleData('daily');

      // Create beautiful gradient
      const gradient = ctx.createLinearGradient(0, 0, 0, 400);
      gradient.addColorStop(0, 'rgba(99, 102, 241, 0.4)');
      gradient.addColorStop(0.5, 'rgba(99, 102, 241, 0.2)');
      gradient.addColorStop(1, 'rgba(99, 102, 241, 0.05)');

      this.charts.dailyUsage = new Chart(ctx, {
        type: 'line',
        data: {
          labels: chartData.labels || this.getLast7Days(),
          datasets: [{
            label: 'Daily Requests',
            data: chartData.datasets?.[0]?.data || this.generateRandomData(7, 10, 100),
            borderColor: '#6366f1',
            backgroundColor: gradient,
            borderWidth: 3,
            fill: true,
            tension: 0.4,
            pointBackgroundColor: '#6366f1',
            pointBorderColor: '#ffffff',
            pointBorderWidth: 3,
            pointRadius: 6,
            pointHoverRadius: 10,
            pointHoverBackgroundColor: '#4f46e5',
            pointHoverBorderWidth: 4
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          interaction: {
            intersect: false,
            mode: 'index'
          },
          plugins: {
            legend: {
              display: false
            },
            tooltip: {
              mode: 'index',
              intersect: false,
              animation: {
                duration: 200
              },
              callbacks: {
                title: function (tooltipItems) {
                  return 'Date: ' + tooltipItems[0].label;
                },
                label: function (context) {
                  return 'Requests: ' + context.parsed.y.toLocaleString();
                }
              }
            }
          },
          scales: {
            y: {
              beginAtZero: true,
              grid: {
                color: 'rgba(148, 163, 184, 0.1)',
                borderDash: [5, 5]
              },
              ticks: {
                color: '#64748b',
                callback: function (value) {
                  return value >= 1000 ? (value / 1000) + 'K' : value;
                }
              }
            },
            x: {
              grid: {
                display: false
              },
              ticks: {
                color: '#64748b',
                maxTicksLimit: 7
              }
            }
          },
          animation: {
            duration: 2000,
            easing: 'easeInOutCubic',
            delay: (context) => context.dataIndex * 100
          }
        }
      });

      window.aiChartInstances.dailyUsage = this.charts.dailyUsage;
      this.addChartControls('dailyUsageChart', 'line');
    },

    /**
     * Initialize task breakdown chart with modern doughnut design.
     */
    initTaskBreakdownChart: function () {
      const canvas = document.getElementById('taskBreakdownChart');
      if (!canvas) return;

      const ctx = canvas.getContext('2d');
      const chartData = this.settings.chartData?.task_breakdown || this.generateSampleData('tasks');

      const colors = [
        '#6366f1', '#8b5cf6', '#06b6d4', '#10b981',
        '#f59e0b', '#ef4444', '#84cc16', '#f97316',
        '#ec4899', '#14b8a6'
      ];

      this.charts.taskBreakdown = new Chart(ctx, {
        type: 'doughnut',
        data: {
          labels: chartData.labels || ['Content Generation', 'SEO Analysis', 'Code Review', 'Summarization', 'Translation'],
          datasets: [{
            data: chartData.datasets?.[0]?.data || this.generateRandomData(5, 5, 50),
            backgroundColor: colors,
            borderWidth: 0,
            hoverBorderWidth: 4,
            hoverBorderColor: '#ffffff',
            hoverOffset: 8
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          cutout: '65%',
          plugins: {
            legend: {
              position: 'right',
              labels: {
                usePointStyle: true,
                pointStyle: 'circle',
                padding: 20,
                color: '#374151',
                font: {
                  size: 14,
                  weight: '500'
                }
              }
            },
            tooltip: {
              callbacks: {
                label: function (context) {
                  const label = context.label || '';
                  const value = context.parsed;
                  const total = context.dataset.data.reduce((a, b) => a + b, 0);
                  const percentage = Math.round((value / total) * 100);
                  return `${label}: ${value} (${percentage}%)`;
                }
              }
            }
          },
          animation: {
            animateRotate: true,
            duration: 2000,
            delay: (context) => context.dataIndex * 200
          }
        }
      });

      window.aiChartInstances.taskBreakdown = this.charts.taskBreakdown;
      this.addChartControls('taskBreakdownChart', 'doughnut');
    },

    /**
     * Initialize hourly usage heatmap-style chart.
     */
    initHourlyUsageChart: function () {
      const canvas = document.getElementById('hourlyUsageChart');
      if (!canvas) return;

      const ctx = canvas.getContext('2d');
      const chartData = this.settings.chartData?.hourly_usage || this.generateSampleData('hourly');

      this.charts.hourlyUsage = new Chart(ctx, {
        type: 'bar',
        data: {
          labels: chartData.labels || Array.from({ length: 24 }, (_, i) => i + ':00'),
          datasets: [{
            label: 'Requests per Hour',
            data: chartData.datasets?.[0]?.data || this.generateRandomData(24, 0, 30),
            backgroundColor: (context) => {
              const value = context.parsed?.y || 0;
              const max = Math.max(...context.dataset.data);
              const intensity = value / max;
              return `rgba(99, 102, 241, ${Math.max(0.2, intensity * 0.8)})`;
            },
            borderColor: '#6366f1',
            borderWidth: 1,
            borderRadius: 4,
            borderSkipped: false,
            hoverBackgroundColor: '#4f46e5',
            hoverBorderWidth: 2
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: {
            legend: {
              display: false
            },
            tooltip: {
              callbacks: {
                title: function (tooltipItems) {
                  return `Hour ${tooltipItems[0].label}`;
                },
                label: function (context) {
                  return `Requests: ${context.parsed.y}`;
                }
              }
            }
          },
          scales: {
            y: {
              beginAtZero: true,
              grid: {
                color: 'rgba(148, 163, 184, 0.1)'
              },
              ticks: {
                color: '#64748b'
              }
            },
            x: {
              grid: {
                display: false
              },
              ticks: {
                color: '#64748b',
                maxTicksLimit: 12
              }
            }
          },
          animation: {
            duration: 1500,
            easing: 'easeOutBounce',
            delay: (context) => context.dataIndex * 50
          }
        }
      });

      window.aiChartInstances.hourlyUsage = this.charts.hourlyUsage;
      this.addChartControls('hourlyUsageChart', 'bar');
    },

    /**
     * Initialize trend analysis chart
     */
    initTrendChart: function () {
      const canvas = document.getElementById('trendChart');
      if (!canvas) return;

      const ctx = canvas.getContext('2d');
      const trendData = this.generateTrendData();

      this.charts.trend = new Chart(ctx, {
        type: 'line',
        data: {
          labels: trendData.labels,
          datasets: [{
            label: 'Success Rate (%)',
            data: trendData.successRate,
            borderColor: '#10b981',
            backgroundColor: 'rgba(16, 185, 129, 0.1)',
            yAxisID: 'y',
            tension: 0.4
          }, {
            label: 'Avg Response Time (ms)',
            data: trendData.responseTime,
            borderColor: '#f59e0b',
            backgroundColor: 'rgba(245, 158, 11, 0.1)',
            yAxisID: 'y1',
            tension: 0.4
          }]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          interaction: {
            mode: 'index',
            intersect: false
          },
          plugins: {
            legend: {
              position: 'top'
            }
          },
          scales: {
            y: {
              type: 'linear',
              display: true,
              position: 'left',
              min: 0,
              max: 100
            },
            y1: {
              type: 'linear',
              display: true,
              position: 'right',
              grid: {
                drawOnChartArea: false
              }
            }
          }
        }
      });
    },

    /**
     * Add interactive controls to charts
     */
    addChartControls: function (canvasId, chartType) {
      const $container = $(`#${canvasId}`).closest('.chart-container');
      if ($container.find('.chart-controls').length) return;

      const controls = `
        <div class="chart-controls">
          <button class="chart-control-btn" data-action="reset" title="Reset Zoom">
            Reset
          </button>
          <button class="chart-control-btn" data-action="export" title="Export Chart">
            Export
          </button>
          <button class="chart-control-btn" data-action="fullscreen" title="Fullscreen">
            Full
          </button>
        </div>
      `;

      $container.find('.chart-header, .chart-title').first().after(controls);
    },

    /**
     * Handle chart actions
     */
    handleChartAction: function (action, chartContainer) {
      const canvasId = chartContainer.find('canvas').attr('id');
      const chartKey = canvasId.replace('Chart', '').replace(/([A-Z])/g, function (match) {
        return match.toLowerCase();
      });
      const chart = this.charts[chartKey];

      switch (action) {
        case 'reset':
          if (chart && chart.resetZoom) {
            chart.resetZoom();
          }
          break;
        case 'export':
          this.exportChart(chartKey);
          break;
        case 'fullscreen':
          this.toggleFullscreenChart(chartContainer);
          break;
      }
    },

    /**
     * Toggle fullscreen chart
     */
    toggleFullscreenChart: function (chartContainer) {
      if (chartContainer.hasClass('chart-fullscreen')) {
        chartContainer.removeClass('chart-fullscreen');
        $('body').removeClass('chart-modal-open');
      } else {
        chartContainer.addClass('chart-fullscreen');
        $('body').addClass('chart-modal-open');
      }

      // Trigger resize after animation
      setTimeout(() => {
        this.handleResize();
      }, 300);
    },

    /**
     * Setup interactive elements and animations
     */
    setupInteractions: function () {
      const self = this;

      // Animated counters for summary cards
      this.animateCounters();

      // Hover effects for cards
      $(this.element).on('mouseenter', '.summary-card', function () {
        $(this).addClass('card-hover');
      }).on('mouseleave', '.summary-card', function () {
        $(this).removeClass('card-hover');
      });

      // Smooth scroll to sections
      $(this.element).on('click', '[data-scroll-to]', function (e) {
        e.preventDefault();
        const target = $(this).data('scroll-to');
        $('html, body').animate({
          scrollTop: $(target).offset().top - 100
        }, 800);
      });

      // Interactive task bars
      $(this.element).on('mouseenter', '.task-item', function () {
        $(this).find('.task-bar-fill').addClass('animated-fill');
      }).on('mouseleave', '.task-item', function () {
        $(this).find('.task-bar-fill').removeClass('animated-fill');
      });

      // Tooltip system
      this.initTooltipSystem();
    },

    /**
     * Animate counter numbers with easing
     */
    animateCounters: function () {
      $('.counter-number, .summary-card__value').each(function () {
        const $counter = $(this);
        const target = parseInt($counter.text().replace(/[^\d.-]/g, '')) || 0;
        const duration = 2000;
        const increment = target / (duration / 16);
        let current = 0;

        const timer = setInterval(function () {
          current += increment;
          if (current >= target) {
            current = target;
            clearInterval(timer);
          }

          let displayValue = Math.floor(current);
          if ($counter.hasClass('format-percentage')) {
            displayValue = displayValue + '%';
          } else if ($counter.hasClass('format-currency')) {
            displayValue = '$' + displayValue.toLocaleString();
          } else {
            displayValue = displayValue.toLocaleString();
          }

          $counter.text(displayValue);
        }, 16);
      });
    },

    /**
     * Animate card values on hover
     */
    animateCardValue: function ($card) {
      const $value = $card.find('.summary-card__value');
      $value.addClass('value-pulse');
      setTimeout(() => {
        $value.removeClass('value-pulse');
      }, 600);
    },

    /**
     * Initialize tooltip system
     */
    initTooltipSystem: function () {
      $(this.element).on('mouseenter', '[data-tooltip]', function (e) {
        const tooltipText = $(this).data('tooltip');
        const $tooltip = $(`<div class="modern-tooltip">${tooltipText}</div>`);

        $('body').append($tooltip);

        const rect = this.getBoundingClientRect();
        $tooltip.css({
          position: 'fixed',
          top: rect.top - $tooltip.outerHeight() - 10,
          left: rect.left + (rect.width / 2) - ($tooltip.outerWidth() / 2),
          opacity: 0,
          transform: 'translateY(10px)'
        }).animate({
          opacity: 1,
          top: rect.top - $tooltip.outerHeight() - 15
        }, 200);

      }).on('mouseleave', '[data-tooltip]', function () {
        $('.modern-tooltip').animate({ opacity: 0 }, 200, function () {
          $(this).remove();
        });
      });
    },

    /**
     * Initialize real-time updates with WebSocket fallback
     */
    initRealTimeUpdates: function () {
      if (!this.settings.enableRealTime) return;

      const self = this;

      // Try WebSocket connection first
      this.tryWebSocketConnection() || this.setupPollingUpdates();

      // Setup auto-refresh
      this.refreshInterval = setInterval(function () {
        if (self.isVisible) {
          self.updateStats();
        }
      }, 30000); // 30 seconds
    },

    /**
     * Try WebSocket connection
     */
    tryWebSocketConnection: function () {
      try {
        const protocol = location.protocol === 'https:' ? 'wss:' : 'ws:';
        const wsUrl = `${protocol}//${location.host}/ws/ai-stats`;

        this.websocket = new WebSocket(wsUrl);

        this.websocket.onmessage = (event) => {
          const data = JSON.parse(event.data);
          this.updateDashboardData(data);
        };

        this.websocket.onerror = () => {
          console.log('WebSocket connection failed, using polling');
          return false;
        };

        return true;
      } catch (error) {
        return false;
      }
    },

    /**
     * Setup polling updates as fallback
     */
    setupPollingUpdates: function () {
      // Polling is handled by the main refresh interval
      console.log('Using polling for real-time updates');
    },

    /**
     * Update statistics via AJAX
     */
    updateStats: function () {
      const self = this;

      $.ajax({
        url: '/admin/reports/ai-content-assistant/api/data',
        method: 'GET',
        data: {
          days: this.getCurrentPeriod(),
          type: 'user'
        },
        success: function (data) {
          self.updateDashboardData(data);
          self.showNotification('Statistics updated', 'success', 2000);
        },
        error: function () {
          self.showNotification('Failed to update statistics', 'error');
        }
      });
    },

    /**
     * Update dashboard with new data
     */
    updateDashboardData: function (data) {
      this.updateSummaryCards(data);
      this.updateCharts(data);
      this.updateActivityTable(data.recent_activity);

      // Trigger custom event
      $(document).trigger('dashboardUpdated', [data]);
    },

    /**
     * Update summary cards with smooth animations
     */
    updateSummaryCards: function (data) {
      const updates = {
        'requests': data.total_requests,
        'users': data.active_users,
        'success': data.success_rate,
        'tokens': data.totals?.tokens
      };

      Object.keys(updates).forEach(key => {
        const $card = $(`.summary-card--${key}`);
        const $value = $card.find('.summary-card__value');
        const newValue = updates[key];

        if ($value.length && newValue !== undefined) {
          $value.addClass('updating');

          setTimeout(() => {
            $value.text(this.formatNumber(newValue, key));
            $value.removeClass('updating').addClass('updated');

            setTimeout(() => {
              $value.removeClass('updated');
            }, 1000);
          }, 300);
        }
      });
    },

    /**
     * Update charts with new data
     */
    updateCharts: function (data) {
      // Update daily usage
      if (this.charts.dailyUsage && data.daily_usage) {
        this.charts.dailyUsage.data.labels = Object.keys(data.daily_usage);
        this.charts.dailyUsage.data.datasets[0].data = Object.values(data.daily_usage);
        this.charts.dailyUsage.update('none');
      }

      // Update task breakdown
      if (this.charts.taskBreakdown && data.task_breakdown) {
        const taskData = Object.values(data.task_breakdown);
        this.charts.taskBreakdown.data.datasets[0].data = taskData;
        this.charts.taskBreakdown.update('none');
      }

      // Update hourly usage
      if (this.charts.hourlyUsage && data.hourly_usage) {
        this.charts.hourlyUsage.data.labels = Object.keys(data.hourly_usage);
        this.charts.hourlyUsage.data.datasets[0].data = Object.values(data.hourly_usage);
        this.charts.hourlyUsage.update('none');
      }
    },

    /**
     * Initialize comprehensive export functionality
     */
    initExportFunctionality: function () {
      const self = this;

      // CSV export with progress indicator
      $(this.element).on('click', '.export-csv', function (e) {
        e.preventDefault();
        self.exportData('csv');
      });

      // PDF export
      $(this.element).on('click', '.export-pdf', function (e) {
        e.preventDefault();
        self.exportData('pdf');
      });

      // Individual chart exports
      $(this.element).on('click', '.export-chart', function (e) {
        e.preventDefault();
        const chartType = $(this).data('chart');
        self.exportChart(chartType);
      });

      // Full dashboard screenshot
      $(this.element).on('click', '.export-dashboard', function (e) {
        e.preventDefault();
        self.exportFullDashboard();
      });
    },

    /**
     * Export chart as high-quality image
     */
    exportChart: function (chartType) {
      const chart = this.charts[chartType];
      if (!chart) {
        this.showNotification(`Chart ${chartType} not found`, 'error');
        return;
      }

      const canvas = chart.canvas;
      const link = document.createElement('a');
      link.download = `ai-usage-${chartType}-${new Date().toISOString().split('T')[0]}.png`;
      link.href = canvas.toDataURL('image/png', 1.0);
      link.click();

      this.showNotification(`${chartType} chart exported successfully`, 'success');
    },

    /**
     * Export data in specified format
     */
    exportData: function (format) {
      const period = this.getCurrentPeriod();
      this.showNotification(`Preparing ${format.toUpperCase()} export...`, 'info');

      const url = `/admin/reports/ai-content-assistant/export?format=${format}&days=${period}`;
      window.open(url, '_blank');
    },

    /**
     * Export full dashboard as PDF
     */
    exportFullDashboard: function () {
      this.showNotification('Generating dashboard PDF...', 'info');

      if (typeof html2pdf !== 'undefined') {
        const element = this.element;
        const opt = {
          margin: 0.5,
          filename: `ai-dashboard-${new Date().toISOString().split('T')[0]}.pdf`,
          image: { type: 'jpeg', quality: 0.98 },
          html2canvas: { scale: 2, useCORS: true },
          jsPDF: { unit: 'in', format: 'letter', orientation: 'landscape' }
        };

        html2pdf().set(opt).from(element).save().then(() => {
          this.showNotification('PDF exported successfully!', 'success');
        });
      } else {
        this.showNotification('PDF export library not available', 'error');
      }
    },

    /**
     * Initialize accessibility features
     */
    initAccessibility: function () {
      // Add ARIA labels to charts
      $('canvas[id*="Chart"]', this.element).each(function () {
        const chartType = this.id.replace('Chart', '').replace(/([A-Z])/g, ' $1').trim();
        $(this).attr({
          'role': 'img',
          'aria-label': `${chartType} chart displaying usage statistics`
        });
      });

      // Keyboard navigation for interactive elements
      $('.summary-card, .task-item', this.element).attr('tabindex', '0').on('keydown', function (e) {
        if (e.which === 13 || e.which === 32) { // Enter or Space
          $(this).trigger('click');
          e.preventDefault();
        }
      });

      // Screen reader announcements
      $(document).on('dashboardUpdated', function () {
        const announcement = $('<div class="sr-only" aria-live="polite">Dashboard statistics updated</div>');
        $('body').append(announcement);
        setTimeout(() => announcement.remove(), 2000);
      });
    },

    /**
     * Start entrance animations
     */
    startEntranceAnimations: function () {
      const elements = $('.summary-card, .chart-container, .stats-section', this.element);

      elements.each(function (index) {
        const $element = $(this);
        setTimeout(() => {
          $element.addClass('animate-in');
        }, index * 100);
      });
    },

    /**
     * Enhanced notification system
     */
    showNotification: function (message, type = 'info', duration = 5000) {
      const notification = $(`
        <div class="notification notification--${type}">
          <div class="notification-icon"></div>
          <span class="notification-message">${message}</span>
          <button class="notification-close">&times;</button>
        </div>
      `);

      let $container = $('.notification-container');
      if (!$container.length) {
        $container = $('<div class="notification-container"></div>');
        $('body').append($container);
      }

      $container.append(notification);

      setTimeout(() => notification.addClass('show'), 100);

      // Auto-hide
      const hideTimer = setTimeout(() => {
        notification.removeClass('show');
        setTimeout(() => notification.remove(), 300);
      }, duration);

      // Manual close
      notification.find('.notification-close').on('click', function () {
        clearTimeout(hideTimer);
        notification.removeClass('show');
        setTimeout(() => notification.remove(), 300);
      });
    },

    /**
     * Switch tabs functionality
     */
    switchTab: function (targetTab) {
      // Hide all tab contents
      $('.tab-content', this.element).removeClass('active');

      // Remove active class from all tab links
      $('.tab-link', this.element).removeClass('active');

      // Show target tab content
      $(targetTab, this.element).addClass('active');

      // Add active class to clicked tab link
      $(`.tab-link[data-tab="${targetTab}"]`, this.element).addClass('active');
    },

    /**
     * Utility functions
     */
    getCurrentPeriod: function () {
      return $('.period-selector', this.element).val() || 30;
    },

    formatNumber: function (num, type) {
      if (typeof num !== 'number') return num;

      switch (type) {
        case 'success':
          return num.toFixed(1) + '%';
        case 'tokens':
          return num >= 1000000 ? (num / 1000000).toFixed(1) + 'M' :
            num >= 1000 ? (num / 1000).toFixed(1) + 'K' :
              num.toLocaleString();
        default:
          return num.toLocaleString();
      }
    },

    handleResize: function () {
      Object.values(this.charts).forEach(chart => {
        if (chart && typeof chart.resize === 'function') {
          chart.resize();
        }
      });
    },

    updatePeriod: function (days) {
      this.showLoadingState();
      const url = new URL(window.location);
      url.searchParams.set('days', days);
      window.history.pushState({}, '', url);
      setTimeout(() => window.location.reload(), 500);
    },

    refreshDashboard: function () {
      this.showLoadingState();
      this.updateStats();
      setTimeout(() => this.hideLoadingState(), 1000);
    },

    showLoadingState: function () {
      $(this.element).addClass('loading');
    },

    hideLoadingState: function () {
      $(this.element).removeClass('loading');
    },

    /**
     * Generate sample data for demonstration
     */
    generateSampleData: function (type) {
      switch (type) {
        case 'daily':
          return {
            labels: this.getLast7Days(),
            datasets: [{ data: this.generateRandomData(7, 10, 100) }]
          };
        case 'tasks':
          return {
            labels: ['Content Gen', 'SEO Analysis', 'Code Review', 'Summary', 'Translation'],
            datasets: [{ data: this.generateRandomData(5, 5, 50) }]
          };
        case 'hourly':
          return {
            labels: Array.from({ length: 24 }, (_, i) => i),
            datasets: [{ data: this.generateRandomData(24, 0, 30) }]
          };
      }
    },

    generateRandomData: function (count, min, max) {
      return Array.from({ length: count }, () =>
        Math.floor(Math.random() * (max - min + 1)) + min
      );
    },

    generateTrendData: function () {
      const labels = this.getLast30Days();
      return {
        labels,
        successRate: labels.map(() => 95 + Math.random() * 5),
        responseTime: labels.map(() => 200 + Math.random() * 300)
      };
    },

    getLast7Days: function () {
      const days = [];
      for (let i = 6; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        days.push(date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' }));
      }
      return days;
    },

    getLast30Days: function () {
      const days = [];
      for (let i = 29; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        days.push(date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' }));
      }
      return days;
    },

    debounce: function (func, wait) {
      let timeout;
      return function executedFunction(...args) {
        const later = () => {
          clearTimeout(timeout);
          func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
      };
    },

    /**
     * Cleanup on destroy
     */
    destroy: function () {
      if (this.refreshInterval) {
        clearInterval(this.refreshInterval);
      }

      if (this.websocket) {
        this.websocket.close();
      }

      Object.values(this.charts).forEach(chart => {
        if (chart && typeof chart.destroy === 'function') {
          chart.destroy();
        }
      });

      $('.notification').remove();
      $('.modern-tooltip').remove();
    }
  };

})(jQuery, Drupal, drupalSettings);
