<?php

namespace Drupal\sphoenix_ai\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\sphoenix_ai\Service\AuthenticationService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for authentication endpoints.
 */
class AuthController extends ControllerBase
{

  /**
   * The authentication service.
   *
   * @var \Drupal\sphoenix_ai\Service\AuthenticationService
   */
  protected $authService;

  /**
   * Constructs an AuthController object.
   *
   * @param \Drupal\sphoenix_ai\Service\AuthenticationService $auth_service
   *   The authentication service.
   */
  public function __construct(AuthenticationService $auth_service)
  {
    $this->authService = $auth_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('sphoenix_ai.authentication')
    );
  }

  /**
   * Handle authentication login.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function login(Request $request)
  {
    try {
      // Get login URL with callback
      $login_url = $this->authService->getLoginUrl();

      return new JsonResponse([
        'success' => TRUE,
        'action' => 'redirect',
        'login_url' => $login_url,
        'message' => 'Redirecting to login page',
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Failed to generate login URL: ' . $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Get authentication status.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function status()
  {
    try {
      $status = $this->authService->getAuthState();

      return new JsonResponse([
        'success' => TRUE,
        'data' => $status,
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Error getting auth status: ' . $e->getMessage(),
        'data' => [
          'authenticated' => FALSE,
          'user' => NULL,
          'subscription' => NULL,
        ],
      ], 500);
    }
  }

  /**
   * Handle user logout.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function logout()
  {
    try {
      $success = $this->authService->logout();

      return new JsonResponse([
        'success' => $success,
        'message' => $success ? 'Logout successful' : 'Logout failed',
        'data' => [
          'authenticated' => FALSE,
          'user' => NULL,
          'subscription' => NULL,
        ],
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Logout error: ' . $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Refresh authentication token.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function refresh()
  {
    try {
      // Use the new refresh method that fetches fresh data from API
      $refresh_result = $this->authService->refreshAuthState();

      if ($refresh_result['success']) {
        return new JsonResponse([
          'success' => TRUE,
          'message' => $refresh_result['message'],
          'data' => $refresh_result['data'],
          'requires_subscription' => $refresh_result['requires_subscription'] ?? FALSE,
        ]);
      } else {
        return new JsonResponse([
          'success' => FALSE,
          'message' => $refresh_result['message'],
          'data' => $refresh_result['data'],
        ], 401);
      }
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Refresh failed: ' . $e->getMessage(),
        'data' => [
          'isAuthenticated' => FALSE,
          'user' => NULL,
          'token' => NULL,
          'subscription' => NULL,
        ],
      ], 500);
    }
  }



  /**
   * Access callback for authentication endpoints.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account)
  {
    return AccessResult::allowedIfHasPermission($account, 'use ai content assistant');
  }
}
