<?php

namespace Drupal\sphoenix_ai\Controller;

use Drupal\sphoenix_ai\Service\ApiClientService;
use Drupal\sphoenix_ai\Service\FieldMapperService;
use Drupal\sphoenix_ai\Service\UsageTrackerService;
use Drupal\sphoenix_ai\Service\AuthenticationService;
use Drupal\sphoenix_ai\Service\SubscriptionService;
use Drupal\sphoenix_ai\Service\ContentAnalyzerService;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for AI Content Assistant chat functionality.
 */
class ChatController extends ControllerBase
{

  /**
   * The API client service.
   *
   * @var \Drupal\sphoenix_ai\Service\ApiClientService
   */
  protected $apiClient;

  /**
   * The field mapper service.
   *
   * @var \Drupal\sphoenix_ai\Service\FieldMapperService
   */
  protected $fieldMapper;

  /**
   * The usage tracker service.
   *
   * @var \Drupal\sphoenix_ai\Service\UsageTrackerService
   */
  protected $usageTracker;

  /**
   * The authentication service.
   *
   * @var \Drupal\sphoenix_ai\Service\AuthenticationService
   */
  protected $authService;

  /**
   * The subscription service.
   *
   * @var \Drupal\sphoenix_ai\Service\SubscriptionService
   */
  protected $subscriptionService;

  /**
   * The content analyzer service.
   *
   * @var \Drupal\sphoenix_ai\Service\ContentAnalyzerService
   */
  protected $contentAnalyzer;

  /**
   * Constructs a ChatController object.
   */
  public function __construct(
    ApiClientService $api_client,
    FieldMapperService $field_mapper,
    UsageTrackerService $usage_tracker,
    AuthenticationService $auth_service,
    SubscriptionService $subscription_service,
    ContentAnalyzerService $content_analyzer
  ) {
    $this->apiClient = $api_client;
    $this->fieldMapper = $field_mapper;
    $this->usageTracker = $usage_tracker;
    $this->authService = $auth_service;
    $this->subscriptionService = $subscription_service;
    $this->contentAnalyzer = $content_analyzer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('sphoenix_ai.api_client'),
      $container->get('sphoenix_ai.field_mapper'),
      $container->get('sphoenix_ai.usage_tracker'),
      $container->get('sphoenix_ai.authentication'),
      $container->get('sphoenix_ai.subscription'),
      $container->get('sphoenix_ai.content_analyzer')
    );
  }

  /**
   * Validate request data.
   *
   * @param array $data
   *   The request data.
   *
   * @return array
   *   Array of validation errors.
   */
  protected function validateRequestData(array $data)
  {
    $errors = [];

    // Check required task_type
    if (empty($data['task_type'])) {
      $errors[] = 'Task type is required';
    } elseif (!$this->apiClient->isValidTaskType($data['task_type'])) {
      $errors[] = 'Invalid task type: ' . $data['task_type'];
    }

    // Check prompt for tasks that require it
    $tasks_requiring_prompt = [
      'content_generation',
      'content_regeneration',
      'content_optimization',
      'thumbnail_generation',
      'graph_generation',
      'table_generation',
      'meta_generation',
      'title_generation',
      'description_generation',
    ];

    if (in_array($data['task_type'], $tasks_requiring_prompt) && empty($data['prompt'])) {
      $errors[] = 'Prompt is required for this task type';
    }

    // Check existing content for analysis tasks
    $tasks_requiring_existing = [
      'content_regeneration',
      'content_optimization',
      'seo_analysis',
      'content_analysis',
      'keyword_extraction',
      'content_summary',
    ];

    if (in_array($data['task_type'], $tasks_requiring_existing)) {
      if (empty($data['existing_content']) && empty($data['field_data'])) {
        $errors[] = 'This task requires existing content to analyze';
      }
    }

    return $errors;
  }

  /**
   * Prepare existing content from field data.
   *
   * @param array &$data
   *   The request data (modified by reference).
   */
  protected function prepareExistingContent(array &$data)
  {
    if (!empty($data['field_data']) && !empty($data['content_type'])) {
      try {
        $extracted_content = $this->fieldMapper->extractExistingContent(
          $data['content_type'],
          $data['field_data']
        );

        if (!empty($extracted_content)) {
          // Combine content from all fields
          $content_parts = [];
          foreach ($extracted_content as $field_name => $field_info) {
            if (!empty($field_info['value'])) {
              $content_parts[] = "=== {$field_info['label']} ===\n{$field_info['value']}";
            }
          }

          if (!empty($content_parts)) {
            $data['existing_content'] = implode("\n\n", $content_parts);
          }
        }
      } catch (\Exception $e) {
        // Log but don't fail the request
        \Drupal::logger('sphoenix_ai')->warning('Failed to extract existing content: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Handle error responses.
   *
   * @param \Exception $e
   *   The exception.
   * @param array $data
   *   The request data.
   *
   * @return array
   *   Error response data with status code.
   */
  protected function handleErrorResponse(\Exception $e, array $data)
  {
    $error_message = $e->getMessage();
    $error_code = 'UNKNOWN_ERROR';
    $status_code = 500;
    $additional_data = [];

    // Enhanced error categorization
    if (
      strpos($error_message, 'Authentication') !== FALSE ||
      strpos($error_message, 'token') !== FALSE ||
      strpos($error_message, 'Unauthorized') !== FALSE
    ) {
      $error_code = 'NOT_AUTHENTICATED';
      $status_code = 401;
      $additional_data = [
        'auth_required' => TRUE,
        'login_url' => $this->authService->getLoginUrl(),
        'sync_suggested' => TRUE,
        'sync_message' => 'Please refresh after logging in'
      ];
    } elseif (
      strpos($error_message, 'Usage limit') !== FALSE ||
      strpos($error_message, 'subscription') !== FALSE
    ) {
      $error_code = 'USAGE_LIMIT_EXCEEDED';
      $status_code = 402;
      $additional_data = [
        'subscription_required' => TRUE,
        'plans_url' => $this->subscriptionService->getPlansUrl(),
        'sync_suggested' => TRUE,
        'sync_message' => 'Please refresh after subscribing to a plan'
      ];
    } elseif (strpos($error_message, 'Token expired') !== FALSE) {
      $error_code = 'TOKEN_EXPIRED';
      $status_code = 401;
      $additional_data = [
        'auth_required' => TRUE,
        'token_expired' => TRUE,
        'sync_suggested' => TRUE,
        'sync_message' => 'Your session has expired. Please refresh to log in again.'
      ];
    }

    return [
      'data' => array_merge([
        'success' => FALSE,
        'error' => $error_message,
        'error_code' => $error_code,
        'data' => null,
        'task_type' => $data['task_type'] ?? null,
      ], $additional_data),
      'status_code' => $status_code,
    ];
  }

  /**
   * Processes API response and adds field mapping information.
   *
   * @param array $api_response
   *   The raw API response.
   * @param array $request_data
   *   The original request data.
   *
   * @return array
   *   Processed response with field mappings.
   */
  protected function processApiResponse(array $api_response, array $request_data)
  {
    // Extract the actual data from the nested structure
    $response_data = $api_response['data']['data'] ?? $api_response['data'] ?? $api_response;

    // Ensure we have the correct structure
    if (!is_array($response_data)) {
      throw new \Exception('Invalid API response structure');
    }

    // Add field mappings if content type is provided and we have content
    if (!empty($request_data['content_type']) && !empty($response_data['content'])) {
      $content_type = $request_data['content_type'];

      try {
        // Get field mapping suggestions for content
        $field_mappings = $this->fieldMapper->getFieldMappingSuggestions(
          $content_type,
          $response_data['content']
        );

        $response_data['field_mappings'] = $field_mappings;
        $response_data['insertion_instructions'] = $this->fieldMapper->prepareInsertionInstructions(
          $content_type,
          $response_data['content']
        );
      } catch (\Exception $e) {
        // Log mapping error but don't fail the request
        \Drupal::logger('sphoenix_ai')->warning('Field mapping failed: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
    }

    // Add task-specific enhancements
    $response_data = $this->enhanceResponseByTaskType($response_data, $request_data);

    // Add metadata
    $response_data['request_metadata'] = [
      'task_type' => $request_data['task_type'],
      'content_type' => $request_data['content_type'] ?? null,
      'user_id' => $this->currentUser()->id(),
      'timestamp' => time(),
    ];

    return $response_data;
  }

  /**
   * Enhances response based on task type.
   *
   * @param array $response_data
   *   The response data.
   * @param array $request_data
   *   The request data.
   *
   * @return array
   *   Enhanced response data.
   */
  protected function enhanceResponseByTaskType(array $response_data, array $request_data)
  {
    $task_type = $request_data['task_type'];

    // Add suggested actions based on task type
    $action_map = [
      'content_generation' => ['insert_content', 'copy_content', 'regenerate', 'optimize_seo'],
      'content_regeneration' => ['insert_content', 'copy_content', 'regenerate', 'compare_original'],
      'content_optimization' => ['insert_content', 'copy_content', 'apply_suggestions', 'reanalyze'],
      'thumbnail_generation' => ['insert_image', 'download_image', 'generate_alt_text', 'regenerate'],
      'graph_generation' => ['insert_content', 'download_data', 'regenerate', 'export_csv'],
      'table_generation' => ['insert_content', 'download_data', 'regenerate', 'export_csv'],
      'meta_generation' => ['insert_meta', 'copy_meta', 'optimize_more', 'generate_variations'],
      'seo_analysis' => ['apply_suggestions', 'export_report', 'reanalyze', 'optimize_content'],
      'content_analysis' => ['apply_suggestions', 'export_report', 'reanalyze', 'improve_readability'],
      'keyword_extraction' => ['insert_keywords', 'copy_keywords', 'analyze_competition', 'expand_keywords'],
      'content_summary' => ['insert_content', 'copy_content', 'create_longer_summary', 'extract_highlights'],
      'title_generation' => ['insert_title', 'copy_title', 'generate_variations', 'seo_optimize_title'],
      'description_generation' => ['insert_content', 'copy_content', 'generate_variations', 'optimize_length'],
    ];

    $response_data['suggested_actions'] = $action_map[$task_type] ?? ['insert_content', 'copy_content', 'regenerate'];

    // Add content type specific suggestions
    if (!empty($request_data['content_type'])) {
      $response_data['content_type_suggestions'] = $this->getContentTypeSuggestions(
        $request_data['content_type'],
        $task_type
      );
    }

    // Add usage context
    $usage_info = $this->subscriptionService->getUsageInfo();
    if ($usage_info) {
      $response_data['usage_context'] = [
        'remaining_usage' => $usage_info['remaining_usage'],
        'usage_percentage' => round($usage_info['usage_percentage'], 1),
        'plan_name' => $usage_info['plan_name'],
        'show_usage_warning' => $usage_info['usage_percentage'] >= 80,
      ];
    }

    return $response_data;
  }

  /**
   * Get content type specific suggestions.
   *
   * @param string $content_type
   *   The content type.
   * @param string $task_type
   *   The task type.
   *
   * @return array
   *   Content type suggestions.
   */
  protected function getContentTypeSuggestions($content_type, $task_type)
  {
    $suggestions = [];

    // Content type specific suggestions
    $content_type_map = [
      'article' => [
        'content_generation' => [
          'Add a compelling hook in the introduction',
          'Include relevant subheadings for better readability',
          'End with a clear call-to-action',
          'Consider adding quotes or statistics',
        ],
        'seo_analysis' => [
          'Check for proper heading hierarchy',
          'Ensure keyword density is appropriate',
          'Add internal and external links',
        ],
      ],
      'blog_post' => [
        'content_generation' => [
          'Use a conversational tone',
          'Include personal insights or experiences',
          'Add social media sharing hooks',
        ],
      ],
      'page' => [
        'content_generation' => [
          'Focus on clear, action-oriented content',
          'Consider user journey and next steps',
          'Keep content scannable with bullet points',
        ],
      ],
      'product' => [
        'content_generation' => [
          'Highlight key features and benefits',
          'Include technical specifications',
          'Add social proof or testimonials',
          'Include pricing and availability info',
        ],
      ],
      'news' => [
        'content_generation' => [
          'Lead with the most important information',
          'Use inverted pyramid structure',
          'Include relevant quotes and sources',
          'Add publication date and author info',
        ],
      ],
    ];

    return $content_type_map[$content_type][$task_type] ?? [];
  }

  /**
   * Gets field content for analysis tasks.
   */
  public function getFieldContent(Request $request)
  {
    try {
      if (!$this->authService->isAuthenticated()) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Authentication required',
        ], 401);
      }

      $data = json_decode($request->getContent(), TRUE);

      if (!$data || empty($data['content_type']) || empty($data['field_data'])) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Content type and field data are required',
        ], 400);
      }

      $content_type = $data['content_type'];
      $field_data = $data['field_data'];
      $target_fields = $data['target_fields'] ?? [];

      // Extract content from specified fields
      $extracted_content = $this->fieldMapper->extractExistingContent(
        $content_type,
        $field_data,
        $target_fields
      );

      // Get text fields for the content type
      $text_fields = $this->fieldMapper->getTextFields($content_type);

      return new JsonResponse([
        'success' => TRUE,
        'extracted_content' => $extracted_content,
        'available_fields' => $text_fields,
        'content_suggestions' => $this->generateContentSuggestions($extracted_content),
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Get AI task suggestions based on current content.
   */
  public function getSuggestions(Request $request)
  {
    try {
      if (!$this->authService->isAuthenticated()) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Authentication required',
        ], 401);
      }

      $data = json_decode($request->getContent(), TRUE);

      if (!$data || empty($data['content_type']) || empty($data['field_data'])) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Content type and field data are required',
        ], 400);
      }

      $suggestions = $this->contentAnalyzer->analyzeAndSuggest(
        $data['content_type'],
        $data['field_data']
      );

      return new JsonResponse([
        'success' => TRUE,
        'suggestions' => $suggestions,
        'task_types' => $this->apiClient->getTaskTypes(),
        'content_type' => $data['content_type'],
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Generates content suggestions based on extracted content.
   *
   * @param array $extracted_content
   *   The extracted content.
   *
   * @return array
   *   Content suggestions.
   */
  protected function generateContentSuggestions(array $extracted_content)
  {
    $suggestions = [];

    foreach ($extracted_content as $field_name => $field_info) {
      $content_length = $field_info['length'];
      $suggestion = $this->createSuggestionForField($field_name, $field_info, $content_length);

      if ($suggestion) {
        $suggestions[] = $suggestion;
      }
    }

    // Sort by priority
    usort($suggestions, function ($a, $b) {
      $priority_order = ['high' => 3, 'medium' => 2, 'low' => 1];
      return $priority_order[$b['priority']] <=> $priority_order[$a['priority']];
    });

    return $suggestions;
  }

  /**
   * Create suggestion for a specific field.
   *
   * @param string $field_name
   *   The field name.
   * @param array $field_info
   *   Field information.
   * @param int $content_length
   *   Content length.
   *
   * @return array|null
   *   Suggestion array or null.
   */
  protected function createSuggestionForField($field_name, array $field_info, $content_length)
  {
    if ($content_length === 0) {
      return [
        'field' => $field_name,
        'suggestion' => 'generate_content',
        'message' => "Generate content for {$field_info['label']}",
        'priority' => 'high',
        'task_type' => $this->suggestTaskType($field_name),
        'icon' => $this->getIconForTask($this->suggestTaskType($field_name)),
      ];
    }

    if ($content_length < 100) {
      return [
        'field' => $field_name,
        'suggestion' => 'expand_content',
        'message' => "Expand content in {$field_info['label']} ({$content_length} chars)",
        'priority' => 'medium',
        'task_type' => 'content_regeneration',
        'icon' => '🔄',
      ];
    }

    if ($content_length > 2000) {
      return [
        'field' => $field_name,
        'suggestion' => 'summarize_content',
        'message' => "Summarize content in {$field_info['label']} ({$content_length} chars)",
        'priority' => 'low',
        'task_type' => 'content_summary',
        'icon' => '📄',
      ];
    }

    // Content exists but could be optimized
    return [
      'field' => $field_name,
      'suggestion' => 'optimize_content',
      'message' => "Optimize content in {$field_info['label']}",
      'priority' => 'medium',
      'task_type' => 'content_optimization',
      'icon' => '📈',
    ];
  }

  /**
   * Suggest appropriate task type for a field.
   *
   * @param string $field_name
   *   The field name.
   *
   * @return string
   *   Suggested task type.
   */
  protected function suggestTaskType($field_name)
  {
    if (strpos($field_name, 'title') !== FALSE) {
      return 'title_generation';
    }

    if (in_array($field_name, ['body', 'field_body', 'field_content'])) {
      return 'content_generation';
    }

    if (strpos($field_name, 'meta') !== FALSE || strpos($field_name, 'description') !== FALSE) {
      return 'meta_generation';
    }

    if (strpos($field_name, 'summary') !== FALSE || strpos($field_name, 'excerpt') !== FALSE) {
      return 'description_generation';
    }

    return 'content_generation';
  }

  /**
   * Get icon for task type.
   *
   * @param string $task_type
   *   The task type.
   *
   * @return string
   *   Icon character.
   */
  protected function getIconForTask($task_type)
  {
    $icons = [
      'content_generation' => '✏️',
      'content_regeneration' => '🔄',
      'content_optimization' => '📈',
      'thumbnail_generation' => '🖼️',
      'graph_generation' => '📊',
      'table_generation' => '📋',
      'meta_generation' => '🏷️',
      'seo_analysis' => '🔍',
      'content_analysis' => '📖',
      'keyword_extraction' => '🔑',
      'content_summary' => '📄',
      'title_generation' => '📰',
      'description_generation' => '📝',
    ];

    return $icons[$task_type] ?? '🤖';
  }


  /**
   * Get chat interface state (like VS Code chatPanel states).
   */
  public function getState(Request $request)
  {
    // Check authentication first
    if (!$this->authService->isAuthenticated()) {
      return new JsonResponse([
        'success' => TRUE,
        'state' => 'auth_required',
        'data' => [
          'login_url' => $this->authService->getLoginUrl(),
          'message' => 'Please login to access AI features',
        ],
      ]);
    }

    // Check subscription
    if (!$this->authService->hasActiveSubscription()) {
      return new JsonResponse([
        'success' => TRUE,
        'state' => 'subscription_required',
        'data' => [
          'plans_url' => $this->subscriptionService->getPlansUrl(),
          'message' => 'Please subscribe to a plan to use AI features',
          'available_plans' => $this->subscriptionService->getAvailablePlans(),
        ],
      ]);
    }

    // Check usage limits
    if (!$this->subscriptionService->hasUsageRemaining()) {
      return new JsonResponse([
        'success' => TRUE,
        'state' => 'usage_exceeded',
        'data' => [
          'plans_url' => $this->subscriptionService->getPlansUrl(),
          'message' => 'Monthly usage limit exceeded',
          'usage_info' => $this->subscriptionService->getUsageInfo(),
        ],
      ]);
    }

    // All good - return chat ready state
    return new JsonResponse([
      'success' => TRUE,
      'state' => 'chat_ready',
      'data' => [
        'user' => $this->authService->getAuthState()['user'],
        'subscription' => $this->authService->getSubscription(),
        'usage_info' => $this->subscriptionService->getUsageInfo(),
        'task_types' => $this->apiClient->getTaskTypes(),
      ],
    ]);
  }

  /**
   * Handles AI chat requests.
   */
  public function chat(Request $request)
  {
    $usage_id = null;

    try {

      // Check authentication first (like VS Code extension)
      if (!$this->authService->isAuthenticated()) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Authentication required. Please login to use AI features.',
          'error_code' => 'NOT_AUTHENTICATED',
          'state' => 'auth_required',
          'login_url' => $this->authService->getLoginUrl(),
        ], 401);
      }

      // Check subscription and usage limits (like VS Code extension)
      if (!$this->subscriptionService->canUseFeature('chat')) {
        $subscription_status = $this->subscriptionService->formatSubscriptionStatus();
        return new JsonResponse([
          'success' => FALSE,
          'error' => $subscription_status['message'],
          'error_code' => $subscription_status['status'] === 'limit_exceeded' ? 'USAGE_LIMIT_EXCEEDED' : 'SUBSCRIPTION_REQUIRED',
          'state' => 'subscription_required',
          'plans_url' => $this->subscriptionService->getPlansUrl(),
          'subscription_status' => $subscription_status,
        ], 402);
      }

      // Parse request data
      $data = json_decode($request->getContent(), TRUE);
      if (!$data) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Invalid request format',
          'error_code' => 'INVALID_REQUEST',
        ], 400);
      }

      // Validate required fields
      $validation_errors = $this->validateRequestData($data);

      if (!empty($validation_errors)) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Validation failed: ' . implode(', ', $validation_errors),
          'error_code' => 'VALIDATION_ERROR',
          'validation_errors' => $validation_errors,
        ], 400);
      }

      // Track usage before making request
      $usage_id = $this->usageTracker->trackRequest($data['task_type'], [
        'content_type' => $data['content_type'] ?? null,
        'prompt_length' => strlen($data['prompt'] ?? ''),
        'has_existing_content' => !empty($data['existing_content']),
        'user_agent' => $request->headers->get('User-Agent'),
      ]);

      // Make API request (server-side only)
      $start_time = microtime(true);
      $api_response = $this->apiClient->makeApiRequest($data);

      $processing_time = microtime(true) - $start_time;

      // Process response
      $processed_response = $this->processApiResponse($api_response, $data);
      $processed_response['processing_time'] = round($processing_time, 3);

      // Record usage and update subscription (like VS Code extension)
      $this->subscriptionService->recordUsage($data['task_type'], 1);
      $this->usageTracker->trackCompletion($data['task_type'], $processed_response, $usage_id);

      // Get updated usage info
      $updated_usage_info = $this->subscriptionService->getUsageInfo();

      return new JsonResponse([
        'success' => TRUE,
        'state' => 'chat_ready',
        'data' => $processed_response,
        'field_mappings' => $processed_response['field_mappings'] ?? [],
        'insertion_instructions' => $processed_response['insertion_instructions'] ?? [],
        'task_type' => $data['task_type'],
        'usage_info' => $updated_usage_info,
        'processing_time' => $processing_time,
      ]);
    } catch (\Exception $e) {
      // Track error
      if ($usage_id) {
        $this->usageTracker->trackError($data['task_type'] ?? 'unknown', $e->getMessage(), $usage_id);
      }

      // Log error for debugging
      \Drupal::logger('sphoenix_ai')->error('Chat request failed: @error', [
        '@error' => $e->getMessage(),
        '@user' => $this->currentUser()->id(),
        '@data' => json_encode($data ?? []),
      ]);

      // Determine error type and response
      $error_response = $this->handleErrorResponse($e, $data ?? []);
      return new JsonResponse($error_response['data'], $error_response['status_code']);
    }
  }


  /**
   * Access callback for chat functionality.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account)
  {
    return AccessResult::allowedIfHasPermission($account, 'use ai content assistant');
  }
}
