<?php

namespace Drupal\sphoenix_ai\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for AI-generated file uploads.
 */
class FileUploadController extends ControllerBase
{

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a FileUploadController object.
   */
  public function __construct(
    FileSystemInterface $file_system
  ) {
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('file_system')
    );
  }

  /**
   * Uploads an AI-generated image file.
   */
  public function uploadImage(Request $request)
  {
    try {

      // Get uploaded file
      $uploaded_file = $request->files->get('image');
      if (!$uploaded_file) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'No image file provided',
          'error_code' => 'NO_FILE',
        ], 400);
      }

      // DEBUG: Log upload details
      \Drupal::logger('sphoenix_ai')->info('Upload debug: @data', [
        '@data' => json_encode([
          'original_name' => $uploaded_file->getClientOriginalName(),
          'mime_type' => $uploaded_file->getMimeType(),
          'client_mime_type' => $uploaded_file->getClientMimeType(),
          'size' => $uploaded_file->getSize(),
          'extension' => pathinfo($uploaded_file->getClientOriginalName(), PATHINFO_EXTENSION),
          'is_valid' => $uploaded_file->isValid(),
          'error' => $uploaded_file->getError(),
        ])
      ]);

      // Also return this in the error response for immediate debugging
      if (!$uploaded_file->isValid()) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'File upload failed: ' . $uploaded_file->getErrorMessage(),
          'error_code' => 'INVALID_FILE',
          'debug' => [
            'original_name' => $uploaded_file->getClientOriginalName(),
            'mime_type' => $uploaded_file->getMimeType(),
            'client_mime_type' => $uploaded_file->getClientMimeType(),
            'size' => $uploaded_file->getSize(),
          ]
        ], 400);
      }

      // Check file size (max 10MB)
      $max_size = 10 * 1024 * 1024;
      if ($uploaded_file->getSize() > $max_size) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'File too large. Maximum 10MB allowed.',
          'error_code' => 'FILE_TOO_LARGE',
        ], 400);
      }

      // FIXED: Enhanced MIME type validation that handles octet-stream issue
      $allowed_types = [
        'image/png',
        'image/jpeg',
        'image/jpg',
        'image/gif',
        'image/webp',
        'image/svg+xml'
      ];

      $server_mime_type = $uploaded_file->getMimeType();
      $client_mime_type = $uploaded_file->getClientMimeType();
      $original_filename = $uploaded_file->getClientOriginalName();
      $extension = strtolower(pathinfo($original_filename, PATHINFO_EXTENSION));

      // Determine the best MIME type to use
      $mime_type = $server_mime_type;

      // If server detection fails with octet-stream, use client MIME type
      if ($server_mime_type === 'application/octet-stream' && in_array($client_mime_type, $allowed_types)) {
        $mime_type = $client_mime_type;
      }

      // For PNG files (common from canvas), be more permissive
      if ($extension === 'png' && ($server_mime_type === 'application/octet-stream' || $client_mime_type === 'image/png')) {
        $mime_type = 'image/png';
      }

      // Final validation
      if (!in_array($mime_type, $allowed_types)) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Only image files are allowed (PNG, JPEG, GIF, WebP, SVG)',
          'error_code' => 'INVALID_FILE_TYPE',
          'debug' => [
            'original_name' => $original_filename,
            'server_mime_type' => $server_mime_type,
            'client_mime_type' => $client_mime_type,
            'used_mime_type' => $mime_type,
            'extension' => $extension,
            'size' => $uploaded_file->getSize(),
          ]
        ], 400);
      }

      // Validate file extension
      $allowed_extensions = ['png', 'jpg', 'jpeg', 'gif', 'webp', 'svg'];

      if (!in_array($extension, $allowed_extensions)) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Invalid file extension. Allowed: ' . implode(', ', $allowed_extensions),
          'error_code' => 'INVALID_EXTENSION',
        ], 400);
      }

      // Additional security check for images (except SVG)
      if (in_array($mime_type, ['image/png', 'image/jpeg', 'image/gif'])) {
        $image_info = getimagesize($uploaded_file->getPathname());
        if ($image_info === FALSE) {
          return new JsonResponse([
            'success' => FALSE,
            'error' => 'Invalid image file',
            'error_code' => 'CORRUPTED_IMAGE',
          ], 400);
        }
      }

      // Generate secure filename
      $user_id = $this->currentUser()->id();
      $timestamp = time();
      $random = substr(md5(uniqid()), 0, 8);
      $upload_type = $request->request->get('type', 'chart');
      $filename = "ai-{$upload_type}-{$user_id}-{$timestamp}-{$random}.{$extension}";

      // Prepare directory
      $upload_directory = 'public://ai-uploads/charts';
      $this->fileSystem->prepareDirectory(
        $upload_directory,
        FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
      );

      // Save file
      $destination = $upload_directory . '/' . $filename;
      $file_uri = $this->fileSystem->saveData(
        file_get_contents($uploaded_file->getPathname()),
        $destination,
        FileSystemInterface::EXISTS_REPLACE
      );

      if (!$file_uri) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Failed to save file to server',
          'error_code' => 'SAVE_ERROR',
        ], 500);
      }

      // Create Drupal file entity
      $file = File::create([
        'uid' => $user_id,
        'filename' => $filename,
        'uri' => $file_uri,
        'status' => 1,
        'created' => $timestamp,
      ]);
      $file->save();

      // Generate URLs
      $file_url = $file->createFileUrl(FALSE);
      if (strpos($file_url, 'http') === 0) {
        $absolute_url = $file_url;
      } else {
        $absolute_url = $request->getSchemeAndHttpHost() . $file_url;
      }

      // Log successful upload
      \Drupal::logger('sphoenix_ai')->info('Image uploaded successfully: @filename by user @uid', [
        '@filename' => $filename,
        '@uid' => $user_id,
      ]);

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'url' => $file_url,
          'absolute_url' => $absolute_url,
          'filename' => $filename,
          'original_name' => $original_filename,
          'file_id' => $file->id(),
          'file_size' => $file->getSize(),
          'mime_type' => $mime_type,
          'upload_type' => $upload_type,
        ],
        'message' => 'File uploaded successfully',
      ]);
    } catch (\Exception $e) {
      // Log error
      \Drupal::logger('sphoenix_ai')->error('Image upload failed: @error', [
        '@error' => $e->getMessage(),
        '@user' => $this->currentUser()->id(),
        '@trace' => $e->getTraceAsString(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Upload failed: ' . $e->getMessage(),
        'error_code' => 'UPLOAD_ERROR',
      ], 500);
    }
  }

  /**
   * Uploads an image from a URL.
   */
  public function uploadImageFromUrl(Request $request)
  {
    try {
      // Get JSON data from request
      $content = $request->getContent();
      $data = json_decode($content, true);

      if (!$data || !isset($data['imageUrl'])) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'No image URL provided',
          'error_code' => 'NO_URL',
        ], 400);
      }

      $image_url = $data['imageUrl'];
      $upload_type = $data['type'] ?? 'ai-image';

      // Validate URL format
      if (!filter_var($image_url, FILTER_VALIDATE_URL)) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Invalid URL format',
          'error_code' => 'INVALID_URL',
        ], 400);
      }

      // Download the image with proper headers and timeout
      $context = stream_context_create([
        'http' => [
          'timeout' => 30,
          'user_agent' => 'Mozilla/5.0 (compatible; DrupalBot/1.0)',
          'follow_location' => true,
          'max_redirects' => 3,
        ]
      ]);

      $image_data = @file_get_contents($image_url, false, $context);

      if ($image_data === FALSE) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Failed to download image from URL',
          'error_code' => 'DOWNLOAD_FAILED',
        ], 400);
      }

      // Check file size (max 10MB)
      $file_size = strlen($image_data);
      $max_size = 10 * 1024 * 1024;
      if ($file_size > $max_size) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Image too large. Maximum 10MB allowed.',
          'error_code' => 'FILE_TOO_LARGE',
        ], 400);
      }

      // Create temporary file to analyze
      $temp_file = tempnam(sys_get_temp_dir(), 'ai_image_');
      file_put_contents($temp_file, $image_data);

      // Get MIME type and validate
      $finfo = new \finfo(FILEINFO_MIME_TYPE);
      $mime_type = $finfo->file($temp_file);

      $allowed_types = [
        'image/png',
        'image/jpeg',
        'image/jpg',
        'image/gif',
        'image/webp',
        'image/svg+xml'
      ];

      if (!in_array($mime_type, $allowed_types)) {
        unlink($temp_file);
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Invalid image type. Only PNG, JPEG, GIF, WebP, SVG allowed',
          'error_code' => 'INVALID_TYPE',
          'debug' => [
            'detected_mime' => $mime_type,
            'url' => $image_url
          ]
        ], 400);
      }

      // Additional validation for non-SVG images
      if (in_array($mime_type, ['image/png', 'image/jpeg', 'image/gif'])) {
        $image_info = getimagesize($temp_file);
        if ($image_info === FALSE) {
          unlink($temp_file);
          return new JsonResponse([
            'success' => FALSE,
            'error' => 'Invalid or corrupted image file',
            'error_code' => 'CORRUPTED_IMAGE',
          ], 400);
        }
      }

      // Determine file extension
      $extension_map = [
        'image/png' => 'png',
        'image/jpeg' => 'jpg',
        'image/gif' => 'gif',
        'image/webp' => 'webp',
        'image/svg+xml' => 'svg'
      ];
      $extension = $extension_map[$mime_type] ?? 'jpg';

      // Generate secure filename
      $user_id = $this->currentUser()->id();
      $timestamp = time();
      $random = substr(md5(uniqid()), 0, 8);
      $filename = "ai-{$upload_type}-{$user_id}-{$timestamp}-{$random}.{$extension}";

      // Prepare directory
      $upload_directory = 'public://ai-uploads/images';
      $this->fileSystem->prepareDirectory(
        $upload_directory,
        FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
      );

      // Save file
      $destination = $upload_directory . '/' . $filename;
      $file_uri = $this->fileSystem->saveData(
        $image_data,
        $destination,
        FileSystemInterface::EXISTS_REPLACE
      );

      // Clean up temp file
      unlink($temp_file);

      if (!$file_uri) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Failed to save image to server',
          'error_code' => 'SAVE_ERROR',
        ], 500);
      }

      // Create Drupal file entity
      $file = File::create([
        'uid' => $user_id,
        'filename' => $filename,
        'uri' => $file_uri,
        'status' => 1,
        'created' => $timestamp,
      ]);
      $file->save();

      // Generate URLs
      $file_url = $file->createFileUrl(FALSE);
      if (strpos($file_url, 'http') === 0) {
        $absolute_url = $file_url;
      } else {
        $absolute_url = $request->getSchemeAndHttpHost() . $file_url;
      }

      // Log successful upload
      \Drupal::logger('sphoenix_ai')->info('Image uploaded from URL: @filename by user @uid from @url', [
        '@filename' => $filename,
        '@uid' => $user_id,
        '@url' => $image_url,
      ]);

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'url' => $file_url,
          'absolute_url' => $absolute_url,
          'filename' => $filename,
          'original_url' => $image_url,
          'file_id' => $file->id(),
          'file_size' => $file_size,
          'mime_type' => $mime_type,
          'upload_type' => $upload_type,
        ],
        'message' => 'Image uploaded successfully from URL',
      ]);
    } catch (\Exception $e) {
      // Clean up temp file if it exists
      if (isset($temp_file) && file_exists($temp_file)) {
        unlink($temp_file);
      }

      // Log error
      \Drupal::logger('sphoenix_ai')->error('Image upload from URL failed: @error', [
        '@error' => $e->getMessage(),
        '@user' => $this->currentUser()->id(),
        '@url' => $image_url ?? 'unknown',
        '@trace' => $e->getTraceAsString(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Upload failed: ' . $e->getMessage(),
        'error_code' => 'UPLOAD_ERROR',
      ], 500);
    }
  }
}
